﻿
using System;
using System.Data;
using System.Collections.Generic;
using System.Configuration;
using System.Data.SqlClient;
using Curse;
using Curse.Logging;


namespace PaymentService
{
    

    [Serializable]
    public class Subscription
    {
        public String Id = null;
        public ESubstriptionStatus Status = 0;
        public Int32 UserId = 0;
        public Int16 Type = 0;
        public DateTime SignupDate;
        public DateTime ExpirationDate;
        public SubscriptionType Subtype = null;

        public Subscription()
        {
            
        }

        public Subscription(String pId, Int32 pUserId)
        {
            Id = pId;
            UserId = pUserId;
        }

        public Subscription(String pId, Int32 pUserId, Int16 pType, ESubstriptionStatus pStatus, DateTime pSignupDate, DateTime pExpirationDate) 
            : this(pId, pUserId)
        {
            Type = pType;
            Status = pStatus;
            SignupDate = pSignupDate;
            ExpirationDate = pExpirationDate;
        }

        public bool CancelSubscription(DateTime pCancellationDate)
        {
            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                conn.Open();

                SqlCommand cmd = conn.CreateCommand();
                cmd.Transaction = conn.BeginTransaction();
                try
                {
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.CommandText = "spCancelSubscription";

                    cmd.Parameters.Add("@strId", SqlDbType.VarChar, 19);
                    cmd.Parameters["@strId"].Value = Id;

                    cmd.Parameters.Add("@dtCancellationDate", SqlDbType.DateTime);
                    cmd.Parameters["@dtCancellationDate"].Value = pCancellationDate;

                    cmd.ExecuteNonQuery();
                    cmd.Transaction.Commit();

                    // Send the info to the auth server
                    return PaymentAuth.CancelPremium(UserId, ExpirationDate);

                }
                catch (Exception ex)
                {
                    Logger.Error("Set Expiration Date: ", ex.Message);
                    cmd.Transaction.Rollback();
                    return false;
                }
            }
        }

        public bool UpdateExpirationDate()
        {
            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                conn.Open();

                SubscriptionType subType = SubscriptionType.GetType(this.Type);


                SqlCommand cmd = conn.CreateCommand();
                cmd.Transaction = conn.BeginTransaction();
                try
                {
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.CommandText = "spSetSubscriptionExpiration";

                    cmd.Parameters.Add("@strId", SqlDbType.VarChar, 19);
                    cmd.Parameters["@strId"].Value = Id;

                    cmd.Parameters.Add("@dtExpirationDate", SqlDbType.DateTime);
                    cmd.Parameters["@dtExpirationDate"].Value = this.ExpirationDate.AddMonths(subType.TermMonths);

                    cmd.ExecuteNonQuery();
                    cmd.Transaction.Commit();

                    // Send the info to the auth server
                    return PaymentAuth.AddSubscriptionCredit(UserId, subType.TermMonths, (Byte)Type);
                    
                }
                catch (Exception ex)
                {
                    Logger.Error("Set Expiration Date: ", ex.Message);
                    cmd.Transaction.Rollback();
                    return false;
                }
            }
        }

        public static Boolean CreateSubscription(String pId, Int32 pUserId, DateTime pSignupDate, Int16 pTypeId)
        {
            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                conn.Open();

                SqlCommand cmd = conn.CreateCommand();
                cmd.Transaction = conn.BeginTransaction();
                try
                {
                    
                    SubscriptionType subType = SubscriptionType.GetType(pTypeId);
                    DateTime expirationDate = pSignupDate.AddMonths(subType.TermMonths).AddDays(1);
                    cmd.CommandText = "spCreateSubscription";
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.Add("@strId", SqlDbType.VarChar, 19);
                    cmd.Parameters["@strId"].Value = pId;
                    cmd.Parameters.Add("@numUserId", SqlDbType.Int);
                    cmd.Parameters["@numUserId"].Value = pUserId;
                    cmd.Parameters.Add("@dtSignupDate", SqlDbType.DateTime);
                    cmd.Parameters["@dtSignupDate"].Value = pSignupDate;
                    cmd.Parameters.Add("@numType", SqlDbType.SmallInt);
                    cmd.Parameters["@numType"].Value = pTypeId;
                    cmd.Parameters.Add("@dtExpirationDate", SqlDbType.DateTime);
                    cmd.Parameters["@dtExpirationDate"].Value = expirationDate;
                    
                    cmd.ExecuteNonQuery();
                    cmd.Transaction.Commit();

                    // Send a welcome e-mail
                    string username = PaymentAuth.GetUsername(pUserId);
                    string emailAddress = PaymentAuth.GetUserEmail(pUserId);

                    // Send the info to the auth server
                    return PaymentAuth.AddSubscriptionCredit(pUserId, subType.TermMonths, (Byte)pTypeId);
                }
                catch(Exception ex)
                {                    
                    Logger.Error("Create Subscription Failed: ", ex.Message);
                    cmd.Transaction.Rollback();
                    return false;
                }
            }

            
        }


        public static Subscription CreateCompedSubscription(CompedSubscription compedSub, int pUserID, int months)
        {
            //get the existing subscription
            Subscription existingSub = GetFromDatabase("C-" + compedSub.Id.ToString());

            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                conn.Open();

                SqlCommand cmd = conn.CreateCommand();
                cmd.Transaction = conn.BeginTransaction();
                try
                {

                    if (existingSub != null)
                    {
                        cmd.CommandText = "update subscription set expiration_date = @exp_date output inserted.expiration_date where id=@id";
                        cmd.Parameters.Add(new SqlParameter("@exp_date", existingSub.ExpirationDate.AddMonths(months)));
                        cmd.Parameters.Add(new SqlParameter("@id", existingSub.Id));
                        existingSub.ExpirationDate = (DateTime)cmd.ExecuteScalar();
                    }
                    else
                    {
                        cmd.CommandText = "insert into subscription(id, user_id, type_id, status, signup_date, expiration_date, complimentary) " +
                            "output inserted.id, inserted.user_id, inserted.type_id, inserted.status, inserted.signup_date, inserted.expiration_date " +
                            "values(@id, @user_id, 0, 1, GetUTCDate(), @expiration_date, 1)";
                        cmd.Parameters.Add(new SqlParameter("@id", "C-" + compedSub.Id.ToString()));
                        cmd.Parameters.Add(new SqlParameter("@user_id", pUserID));
                        cmd.Parameters.Add(new SqlParameter("@expiration_date", DateTime.UtcNow.AddMonths(months)));
                        SqlDataReader rdr = cmd.ExecuteReader();
                        if (rdr.Read())
                        {
                            existingSub = new Subscription(rdr.GetString(0), rdr.GetInt32(1), rdr.GetInt16(2), (ESubstriptionStatus)rdr.GetByte(3), rdr.GetDateTime(4), rdr.GetDateTime(5));
                        }
                        rdr.Close();
                    }

                    cmd.Transaction.Commit();

      
                }
                catch (Exception ex)
                {
                    Logger.Error("Create Comped Subscription Failed: ", ex.Message);
                    cmd.Transaction.Rollback();
                    return null;
                }
            }

            if (existingSub != null) existingSub.Subtype = SubscriptionType.GetType(existingSub.Type);

            return existingSub;
           
        }

        /// <summary>
        /// Creates a comped subscripotion from an existing comp subscription that could be quite old.
        /// </summary>
        /// <param name="compedSub">The current comped subscription for the user</param>
        /// <param name="pUserID">The user's ID that is getting a new comp</param>
        /// <param name="months">Number of months</param>
        /// <returns></returns>
        public static Subscription CreateCompedSubscriptionFromOldComp(CompedSubscription compedSub, int pUserID, int months)
        {
            //get the existing subscription
            Subscription existingSub = GetFromDatabase("C-" + compedSub.Id.ToString());

            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                conn.Open();

                SqlCommand cmd = conn.CreateCommand();
                cmd.Transaction = conn.BeginTransaction();
                try
                {

                    if (existingSub != null)
                    {
                        cmd.CommandText = "update subscription set expiration_date = DATEADD(month, 1, GETUTCDATE()) output inserted.expiration_date where id=@id";
                        cmd.Parameters.Add(new SqlParameter("@id", existingSub.Id));
                        existingSub.ExpirationDate = (DateTime)cmd.ExecuteScalar();
                    }
                    else
                    {
                        cmd.CommandText = "insert into subscription(id, user_id, type_id, status, signup_date, expiration_date, complimentary) " +
                            "output inserted.id, inserted.user_id, inserted.type_id, inserted.status, inserted.signup_date, inserted.expiration_date " +
                            "values(@id, @user_id, 0, 1, GetUTCDate(), @expiration_date, 1)";
                        cmd.Parameters.Add(new SqlParameter("@id", "C-" + compedSub.Id.ToString()));
                        cmd.Parameters.Add(new SqlParameter("@user_id", pUserID));
                        cmd.Parameters.Add(new SqlParameter("@expiration_date", DateTime.UtcNow.AddMonths(months)));
                        SqlDataReader rdr = cmd.ExecuteReader();
                        if (rdr.Read())
                        {
                            existingSub = new Subscription(rdr.GetString(0), rdr.GetInt32(1), rdr.GetInt16(2), (ESubstriptionStatus)rdr.GetByte(3), rdr.GetDateTime(4), rdr.GetDateTime(5));
                        }
                        rdr.Close();
                    }

                    cmd.Transaction.Commit();


                }
                catch (Exception ex)
                {
                    Logger.Error("Create Comped Subscription Failed: ", ex.Message);
                    cmd.Transaction.Rollback();
                    return null;
                }
            }

            if (existingSub != null) existingSub.Subtype = SubscriptionType.GetType(existingSub.Type);

            return existingSub;

        }

        public static Boolean ModifySubscription(String pId, Int32 pUserId, DateTime pEffectiveDate, Int16 pTypeId)
        {
            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                conn.Open();

                SqlCommand cmd = conn.CreateCommand();
                cmd.Transaction = conn.BeginTransaction();
                try
                {

                    SubscriptionType subType = SubscriptionType.GetType(pTypeId);
                    DateTime expirationDate = pEffectiveDate.AddMonths(subType.TermMonths).AddDays(1);
                    cmd.CommandText = "spModifySubscription";
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.Add("@strId", SqlDbType.VarChar, 19);
                    cmd.Parameters["@strId"].Value = pId;
                    cmd.Parameters.Add("@numType", SqlDbType.SmallInt);
                    cmd.Parameters["@numType"].Value = pTypeId;
                    cmd.Parameters.Add("@dtExpirationDate", SqlDbType.DateTime);
                    cmd.Parameters["@dtExpirationDate"].Value = expirationDate;

                    cmd.ExecuteNonQuery();
                    cmd.Transaction.Commit();

                    // Send the info to the auth server
                    return PaymentAuth.AddSubscriptionCredit(pUserId, subType.TermMonths, (Byte)pTypeId);
                }
                catch (Exception ex)
                {
                    Logger.Error("Modify Subscription Failed: ", ex.Message);
                    cmd.Transaction.Rollback();
                    return false;
                }
            }
        }

        public static Boolean Exists(String pSubscriptionId)
        {
            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                conn.Open();
                SqlCommand command = new SqlCommand("select top 1 1 from subscription where id = @id ", conn);
                command.Parameters.Add("@id", SqlDbType.VarChar, 19);
                command.Parameters["@id"].Value = pSubscriptionId;
                return (command.ExecuteScalar() != null);
            }
        }

        public static Boolean Exists(int pUserId)
        {
            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                conn.Open();
                SqlCommand command = new SqlCommand("select top 1 1 from subscription where user_id = @id ", conn);
                command.Parameters.Add("@id", SqlDbType.Int);
                command.Parameters["@id"].Value = pUserId;
                return (command.ExecuteScalar() != null);
            }
        }

        public static Subscription GetFromDatabase(String pSubscriptionId)
        {
            Subscription sub = new Subscription();
            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                conn.Open();
                SqlCommand command = new SqlCommand("select * from subscription where id = @id ", conn);
                command.Parameters.Add("@id", SqlDbType.VarChar, 19);
                command.Parameters["@id"].Value = pSubscriptionId;
                using (SqlDataReader reader = command.ExecuteReader())
                {
                    if (!reader.Read())
                    {
                        Logger.Error("Subscription Not Found! ID: " + pSubscriptionId);
                        return null;
                    }

                    sub.Id = reader.GetString(reader.GetOrdinal("id"));
                    sub.UserId = reader.GetInt32(reader.GetOrdinal("user_id"));
                    sub.Type = reader.GetInt16(reader.GetOrdinal("type_id"));
                    sub.Status = (ESubstriptionStatus)reader.GetByte(reader.GetOrdinal("status"));
                    sub.SignupDate = reader.GetDateTime(reader.GetOrdinal("signup_date"));
                    sub.ExpirationDate = reader.GetDateTime(reader.GetOrdinal("expiration_date"));
                    
                }
            }

            if (sub != null) sub.Subtype = SubscriptionType.GetType(sub.Type);
            return sub;
        }

        public static List<Subscription> GetFromDatabaseBySubscriber(int pSubscriberID)
        {
            List<Subscription> subs = new List<Subscription>();
            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                conn.Open();
                SqlCommand command = new SqlCommand("select * from subscription where user_id = @user_id ", conn);
                command.Parameters.Add(new SqlParameter("@user_id", pSubscriberID));
                using (SqlDataReader reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        Subscription sub = new Subscription();
                        sub.Id = reader.GetString(reader.GetOrdinal("id"));
                        sub.UserId = reader.GetInt32(reader.GetOrdinal("user_id"));
                        sub.Type = reader.GetInt16(reader.GetOrdinal("type_id"));
                        sub.Status = (ESubstriptionStatus)reader.GetByte(reader.GetOrdinal("status"));
                        sub.SignupDate = reader.GetDateTime(reader.GetOrdinal("signup_date"));
                        sub.ExpirationDate = reader.GetDateTime(reader.GetOrdinal("expiration_date"));
                        subs.Add(sub);

                    }
                    reader.Close();
                    if (subs.Count < 1)
                    {
                        Logger.Error("Subscription Not Found! ID: " + pSubscriberID);
                        return null;
                    }

                }

            }

            subs.ForEach(s =>
            {
                s.Subtype = SubscriptionType.GetType(s.Type);
            });

            return subs;
        }

        private static string GenerateInClause(string columnName, int[] values, int maxValuesPerInClause, bool useCharacterQualifiers)
        {
            string result = String.Concat(columnName, " IN (");
            int count = 0;
            int totalCount = 0;
            int numValues = values.Length;
            foreach(object value in values)
            {
                if (count == maxValuesPerInClause && totalCount < numValues)
                {
                    count = 0;
                    result = String.Concat(result, ") OR ", columnName, " IN (");
                }
                if (count > 0) result = string.Concat(result, ",");
                result = String.Concat(result, useCharacterQualifiers ? "'" : "", value, useCharacterQualifiers ? "'" : "");
                count++;
                totalCount++;
            }
            result = String.Concat(result, ")");
            return result;
        }

        public static List<Subscription> GetActiveFromDatabase(int[] userIDs)
        {
            List<Subscription> subs = new List<Subscription>();
            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                conn.Open();
                SqlCommand command = new SqlCommand("select * from subscription where status = 1 and expiration_date >= GETUTCDATE() and " + GenerateInClause("user_id", userIDs, 100, false), conn);
                using (SqlDataReader reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        Subscription sub = new Subscription();
                        sub.Id = reader.GetString(reader.GetOrdinal("id"));
                        sub.UserId = reader.GetInt32(reader.GetOrdinal("user_id"));
                        sub.Type = reader.GetInt16(reader.GetOrdinal("type_id"));
                        sub.Status = (ESubstriptionStatus)reader.GetByte(reader.GetOrdinal("status"));
                        sub.SignupDate = reader.GetDateTime(reader.GetOrdinal("signup_date"));
                        sub.ExpirationDate = reader.GetDateTime(reader.GetOrdinal("expiration_date"));
                        subs.Add(sub);

                    }
                    reader.Close();
                    if (subs.Count < 1)
                    {
                        Logger.Error("No active subscriptions found!");
                        return null;
                    }

                }

            }

            subs.ForEach(s =>
            {
                s.Subtype = SubscriptionType.GetType(s.Type);
            });

            return subs;
        }

        public static List<Subscription> GetActiveCompsFromDatabase()
        {
            List<Subscription> subs = new List<Subscription>();
            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                conn.Open();
                //SqlCommand command = new SqlCommand("select * from subscription where status = 1 and expiration_date >= GETUTCDATE() and " + GenerateInClause("user_id", userIDs, 100, false), conn);
                SqlCommand command = new SqlCommand("select * from subscription where status = 1 and expiration_date >= GETUTCDATE() and ID LIKE 'C-%'", conn);
                using (SqlDataReader reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        Subscription sub = new Subscription();
                        sub.Id = reader.GetString(reader.GetOrdinal("id"));
                        sub.UserId = reader.GetInt32(reader.GetOrdinal("user_id"));
                        sub.Type = reader.GetInt16(reader.GetOrdinal("type_id"));
                        sub.Status = (ESubstriptionStatus)reader.GetByte(reader.GetOrdinal("status"));
                        sub.SignupDate = reader.GetDateTime(reader.GetOrdinal("signup_date"));
                        sub.ExpirationDate = reader.GetDateTime(reader.GetOrdinal("expiration_date"));
                        subs.Add(sub);

                    }
                    reader.Close();
                    if (subs.Count < 1)
                    {
                        Logger.Error("No active subscriptions found!");
                        return null;
                    }

                }

            }

            subs.ForEach(s =>
            {
                s.Subtype = SubscriptionType.GetType(s.Type);
            });

            return subs;
        }
    }

    
}
