﻿using Curse;
using System;
using System.Data;
using System.Data.SqlClient;
using System.Configuration;
using System.Web;
using System.Collections.Generic;
using System.Collections.Specialized;
using Curse.Extensions;
using Curse.Logging;


namespace PaymentService
{
    public class TransactionData
    {
        public int Guid { get; set; }
        public string ID { get; set; }
        public string SubscriptionID { get; set; }
        public string Data { get; set; }
        public DateTime Posted { get; set; }
        public bool SandboxMode { get; set; }
        public bool Verified { get; set; }
    }

    public class Transaction
    {
        private static readonly DataTable sLogTableSchema = null;
        private static Dictionary<String, Transaction> mPendingPayments = new Dictionary<String, Transaction>();

        private Int32 mGuid = 0;
        private String mTransactionId = null;
        private String mSubscriptionId = null;
        private String mLogData = null;
        private Boolean mSandboxMode = true;
        private ETransactionStatus mStatus = ETransactionStatus.Pending;
        private ETransactionType mType = ETransactionType.Undefined;
        private NameValueCollection mPostedData = null;

        static Transaction()
        {
            sLogTableSchema = new DataTable();
            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                conn.Open();
                DBUtility.LoadSchema(ref sLogTableSchema, "transaction_log", conn);
            }
        }

        public static void AddPendingPayment(String pSubscriptionId, Transaction pTransaction)
        {
            lock (mPendingPayments)
            {
                mPendingPayments.Add(pSubscriptionId, pTransaction);
            }
        }

        public ETransactionStatus Status
        {

            get
            {
                return mStatus;
            }
            set
            {
                mStatus = value;
            }    
            
        }

        public Transaction(ETransactionType pType, NameValueCollection pTransData, String pData, Boolean pSandboxMode)
        {            
            mLogData = pData;
            mSandboxMode = pSandboxMode;
            mType = pType;
            Log();
        }

        public Transaction(ETransactionType pType, NameValueCollection pTransData, String pSubscriptionId, String pData, Boolean pSandboxMode)
        {
            mType = pType;
            mPostedData = pTransData;
            mLogData = pData;
            mSandboxMode = pSandboxMode;
            mSubscriptionId = pSubscriptionId;
            Log();
        }

        public Transaction(ETransactionType pType, NameValueCollection pTransData, String pId, String pSubscriptionId, String pData, Boolean pSandboxMode)
        {
            mType = pType;
            mPostedData = pTransData;
            mTransactionId = pId;
            mLogData = pData;
            mSandboxMode = pSandboxMode;
            mSubscriptionId = pSubscriptionId;
            mType = pType;
            Log();
        }

        public static List<TransactionData> GetFromDatabase(String pSubscriptionId)
        {
            List<TransactionData> transactions = new List<TransactionData>();
            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                SqlCommand command = new SqlCommand("select * from transaction_log where subscription_id = @id ", conn);
                command.Parameters.Add("@id", SqlDbType.VarChar, 19).Value = pSubscriptionId;
                
                conn.Open();
                using (SqlDataReader reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var trans = new TransactionData
                        { 
                            Guid = reader.GetInt32(reader.GetOrdinal("guid")),
                            ID = reader.GetNullableValue<string>("id"),
                            SubscriptionID = reader.GetString(reader.GetOrdinal("subscription_id")),
                            Data = reader.GetString(reader.GetOrdinal("data")),
                            Posted = reader.GetDateTime(reader.GetOrdinal("posted")),
                            SandboxMode = reader.GetBoolean(reader.GetOrdinal("sandbox_mode")),
                            Verified = reader.GetBoolean(reader.GetOrdinal("verified"))
                        };

                        transactions.Add(trans);
                    }
                }
            }

            return transactions;
        }

        public void Commit()
        {

            // Process the transaction

            switch (mType)
            {
                case ETransactionType.SubscriptionSignup:
                    CreateSubscription();
                    break;
                case ETransactionType.ModifySubscription:
                    ModifySubscription();
                    break;
                case ETransactionType.SubscriptionPayment:
                    ProcessSubscriptionPayment();
                    break;
                case ETransactionType.CancelSubscription:
                    ProcessSubscriptionCancellation();
                    break;
                case ETransactionType.NonRecurringPayment:
                    ProcessNonRecurringPayment();
                    break;
            }

            // Log the transaction
            
            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception)
                {
                    return;
                }

                SqlCommand command = conn.CreateCommand();
                command.CommandText = "update transaction_log set verified = 1 where guid = @id;";
                command.Parameters.Add("@id", SqlDbType.Int);
                command.Parameters["@id"].Value = mGuid;
                command.ExecuteNonQuery();
            }
        }

        private bool ModifySubscription()
        {
            Int32 userId = Int32.Parse(mPostedData["custom"]);
            Int16 typeId = Int16.Parse(mPostedData["item_number"]);
            DateTime effectiveDate = PaymentUtility.ParseDateTime(mPostedData["subscr_effective"]);

            bool status = Subscription.ModifySubscription(mSubscriptionId, userId, effectiveDate, typeId);

            if (!status)
            {
                return status;
            }

            lock (mPendingPayments)
            {
                if (mPendingPayments.ContainsKey(mSubscriptionId))
                {
                    mPendingPayments[mSubscriptionId].Commit();
                }
                mPendingPayments.Remove(mSubscriptionId);
            }

            return true;            
        }

        private bool CreateSubscription()
        {
            Int32 userId = Int32.Parse(mPostedData["custom"]);
            Int16 typeId = Int16.Parse(mPostedData["item_number"]);
            DateTime signupDate = PaymentUtility.ParseDateTime(mPostedData["subscr_date"]);

            bool status = Subscription.CreateSubscription(mSubscriptionId, userId, signupDate, typeId);

            if (!status)
            {
                return status;
            }

            lock (mPendingPayments)
            {
                if (mPendingPayments.ContainsKey(mSubscriptionId))
                {
                    mPendingPayments[mSubscriptionId].Commit();
                }
                mPendingPayments.Remove(mSubscriptionId);
            }

            return true;

        }

        private bool ProcessSubscriptionPayment()
        {            
            DateTime paymentDate = PaymentUtility.ParseDateTime(mPostedData["payment_date"]);
            Subscription sub = Subscription.GetFromDatabase(mSubscriptionId);
            SubscriptionType subType = SubscriptionType.GetType(sub.Type);
            // If the difference between the signup date and the payment date does not exceed 5 days, it's likely an echeck.
            if (Math.Abs(sub.SignupDate.Subtract(paymentDate).TotalMinutes) < 10)
            {

                Logger.Info("Ignored Payment, due to time. Signup Date: " + sub.SignupDate + ", Payment Date: " + paymentDate);

                return true;
            }            
            sub.UpdateExpirationDate();
            return true;
        }

        private bool ProcessNonRecurringPayment()
        {
            Int32 userId = Int32.Parse(mPostedData["custom"]);
            Int16 typeId = Int16.Parse(mPostedData["item_number"]);
            DateTime transactionDate = PaymentUtility.ParseDateTime(mPostedData["payment_date"]);

            if (!Subscription.Exists(mSubscriptionId))
            {
                return Subscription.CreateSubscription(mSubscriptionId, userId, transactionDate, typeId);            
            }
            else
            {
                Subscription sub = Subscription.GetFromDatabase(mSubscriptionId);
                SubscriptionType subType = SubscriptionType.GetType(sub.Type);
                sub.UpdateExpirationDate();
                return true;
            }                        
        }

        private bool ProcessSubscriptionCancellation()
        {
            DateTime cancellationDate = PaymentUtility.ParseDateTime(mPostedData["subscr_date"]);
            Subscription sub = Subscription.GetFromDatabase(mSubscriptionId);
            sub.CancelSubscription(cancellationDate);          
            return true;
        }

        public Boolean Log()
        {
            
            using(SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception)
                {
                    return false;
                }
            
                try
                {

                    SqlCommand command = new SqlCommand("spLogTransaction", conn);
                    command.CommandType = CommandType.StoredProcedure;
                    
                    command.Parameters.Add("@id", SqlDbType.VarChar, 19);
                    if (mTransactionId != null)
                    {
                        command.Parameters["@id"].Value = mTransactionId;
                    }
                    else
                    {
                        command.Parameters["@id"].Value = System.DBNull.Value;
                    }


                    command.Parameters.Add("@subscriptionId", SqlDbType.VarChar, 19);
                    if (mSubscriptionId != null)
                    {
                        command.Parameters["@subscriptionId"].Value = mSubscriptionId;
                    }
                    else
                    {
                        command.Parameters["@subscriptionId"].Value = System.DBNull.Value;
                    }

                    command.Parameters.Add("@data", SqlDbType.NVarChar);
                    command.Parameters["@data"].Value = mLogData;

                    command.Parameters.Add("@sandboxMode", SqlDbType.Bit);
                    command.Parameters["@sandboxMode"].Value = mSandboxMode;

                    mGuid = (Int32)command.ExecuteScalar();
                    
                    return true;
                }
                catch (Exception ex)
                {
                    Logger.Error("Error Logging Transaction: ", ex.Message);
                    return false;
                }

            }

        }
    }
}
