﻿using Curse;
using Curse.Auth;
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Configuration;
using System.Data;
using System.Data.SqlClient;
using System.IO;
using System.Net;
using System.Security.Cryptography;
using System.Text;
using System.Web;
using System.Web.Services;
using System.Web.Services.Protocols;

namespace ProfileCenter
{
    /**
     * Roaming Profile Front-End Web Service Class
     * Inherits AuthenticatableWebService
     *
     * @author Shane Bryldt
     */
    [WebService(Namespace = "http://profileservice.curse.com/")]
    [WebServiceBinding(ConformsTo = WsiProfiles.BasicProfile1_1)]
    [ToolboxItem(false)]
    public class ProfileService
        : AuthenticatableWebService
    {
        /**
         * Static constructor
         * Does one-time initialization of the profile service
         */
        static ProfileService()
        {
            Logger.SetLogLevel = (ELogLevel)Array.IndexOf<String>(Enum.GetNames(typeof(ELogLevel)),
                                                                  ConfigurationManager.AppSettings["LogLevel"]);
            Logger.SetLogPath = ConfigurationManager.AppSettings["LogPath"];
            sDataPath = ConfigurationManager.AppSettings["DataPath"];
            sClientCipher = new StringCipher(ConfigurationManager.AppSettings["ClientKey"]);
            String authService = ConfigurationManager.AppSettings["AuthenticationService"];
            Int32 authId = Convert.ToInt32(ConfigurationManager.AppSettings["AuthenticationId"]);
            String authKey = ConfigurationManager.AppSettings["AuthenticationKey"];
            Int32 sessionExpiration = Convert.ToInt32(ConfigurationManager.AppSettings["SessionExpiration"]);

            sAuthentication = new Authentication(authService,
                                                 authId,
                                                 authKey,
                                                 sessionExpiration);

            sRoamingDB = ConfigurationManager.ConnectionStrings["RoamingDB"].ConnectionString;


            using (SqlConnection conn = new SqlConnection(sRoamingDB))
            {
                conn.Open();
                using (SqlTransaction trans = conn.BeginTransaction())
                {
                    SqlCommand cmd = new SqlCommand();
                    cmd.Connection = conn;
                    cmd.Transaction = trans;

                    cmd.CommandText = "SELECT game_id FROM game";
                    using (SqlDataReader dr = cmd.ExecuteReader())
                    {
                        while (dr.Read())
                        {
                            sGames.Add(Convert.ToInt32(dr["game_id"]));
                        }
                    }

                    trans.Commit();
                }
            }
            Logger.Log(ELogLevel.Info,
                       null,
                       "Ready");
        }

        /**
         * Publically exposed web method for updating the game addon database
         * Returns a StatusCode in the response stream
         *
         * @param  pSession the session key
         * @param  pGameId  the game id to associate addons with
         * @param  pAddons  an array of curse-based addon ids
         */
        [WebMethod]
        public void UpdateAddons(String pSession,
                                 Int32 pGameId,
                                 Int32[] pAddons)
        {
            try
            {
                if (!sGames.Contains(pGameId))
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "Invalid game id {0}",
                               pGameId);
                    Context.Response.OutputStream.WriteByte((byte)StatusCode.ProfileInvalidGame);
                    return;
                }

                Int32 userId;
                StatusCode status = sAuthentication.ValidateSessionKeyToUserId(pSession, out userId);
                if (status != StatusCode.Ok)
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "ValidateSessionKeyToUserId failed with status {0}",
                               status);
                    Context.Response.OutputStream.WriteByte((Byte)status);
                    return;
                }

                using (SqlConnection conn = new SqlConnection(sRoamingDB))
                {
                    conn.Open();
                    using (SqlTransaction trans = conn.BeginTransaction())
                    {
                        SqlCommand cmd = new SqlCommand();
                        cmd.Connection = conn;
                        cmd.Transaction = trans;

                        cmd.CommandText = string.Format("DELETE FROM addon WHERE user_id = {0}",
                                                        userId);
                        cmd.ExecuteNonQuery();

                        foreach (Int32 addonId in pAddons)
                        {
                            cmd.CommandText = string.Format("INSERT INTO addon VALUES ({0},{1},{2})",
                                                            userId,
                                                            pGameId,
                                                            addonId);
                            cmd.ExecuteNonQuery();
                        }

                        trans.Commit();
                    }
                }

                Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);
                return;
            }
            catch (Exception exc)
            {
                Logger.Log(ELogLevel.Error,
                           Context.Request.UserHostAddress,
                           "UpdateAddons Exception: {0}",
                           exc.Message);
            }

            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Unknown);
            return;
        }

        /**
         * Publically exposed web method for uploading roaming profile data
         * Parameters are taken through form fields
         * Returns a StatusCode in the response stream
         */
        [WebMethod]
        public void UploadData()
        {
            try
            {
                String session = Context.Request.Form["pSession"];
                Int32 gameId = Convert.ToInt32(Context.Request.Form["pGameId"]);

                if (!sGames.Contains(gameId))
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "Invalid game id {0}",
                               gameId);
                    Context.Response.OutputStream.WriteByte((Byte)StatusCode.ProfileInvalidGame);
                    return;
                }

                if (Context.Request.Files.Count == 0)
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "No file to read from the stream");
                    Context.Response.OutputStream.WriteByte((Byte)StatusCode.InvalidStream);
                    return;
                }

                Int32 userId;
                StatusCode status = sAuthentication.ValidateSessionKeyToUserId(session, out userId);
                if (status != StatusCode.Ok)
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "ValidateSessionKeyToUserId failed with status {0}",
                               status);
                    Context.Response.OutputStream.WriteByte((Byte)status);
                    return;
                }

                if (sAuthentication.GetPremiumLevel(session) == 0)
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "User {0} is not premium and attempted to upload data",
                               userId);
                    Context.Response.OutputStream.WriteByte((Byte)StatusCode.AuthenticationNotPremium);
                    return;
                }

                Int32 pathHash = Convert.ToInt32(Math.Floor(Convert.ToDouble(userId) / FILES_PER_DIRECTORY));
                String path = sDataPath + gameId.ToString() + Path.DirectorySeparatorChar + pathHash.ToString();
                if (!Directory.Exists(path))
                {
                    Directory.CreateDirectory(path);
                }
                path += Path.DirectorySeparatorChar + userId.ToString() + ".dat";

                HttpPostedFile postedFile = Context.Request.Files[0];
                postedFile.SaveAs(path);
                String hash = ComputeDataHash(path);
                using (SqlConnection conn = new SqlConnection(sRoamingDB))
                {
                    conn.Open();
                    using (SqlTransaction trans = conn.BeginTransaction())
                    {
                        SqlCommand cmd = new SqlCommand();
                        cmd.Connection = conn;
                        cmd.Transaction = trans;

                        cmd.CommandText = String.Format("UPDATE hash SET hash='{0}' WHERE user_id = {1} AND game_id = {2}",
                                                        hash,
                                                        userId,
                                                        gameId);
                        if (cmd.ExecuteNonQuery() == 0)
                        {
                            cmd.CommandText = String.Format("INSERT INTO hash VALUES ({0},{1},'{2}')",
                                                            userId,
                                                            gameId,
                                                            hash);
                            cmd.ExecuteNonQuery();
                        }

                        trans.Commit();
                    }
                }

                Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);
                return;
            }
            catch (Exception exc)
            {
                Logger.Log(ELogLevel.Error,
                           Context.Request.UserHostAddress,
                           "UploadData Exception: {0}",
                           exc.Message);
            }

            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Unknown);
            return;
        }

        /**
         * Publically exposed web method for determining if a roaming profile has changed
         * Returns a StatusCode in the response stream
         * Returns a hex string of a SHA1 hash if successful in the response stream
         * 
         * @param  pSession the session key
         * @param  pGameId  the game id to get the roaming profile hash from
         */
        [WebMethod]
        public void GetDataHash(String pSession,
                                Int32 pGameId)
        {
            try
            {
                if (!sGames.Contains(pGameId))
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "Invalid game id {0}",
                               pGameId);
                    Context.Response.OutputStream.WriteByte((Byte)StatusCode.ProfileInvalidGame);
                    return;
                }

                Int32 userId;
                StatusCode status = sAuthentication.ValidateSessionKeyToUserId(pSession, out userId);
                if (status != StatusCode.Ok)
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "ValidateSessionKeyToUserId failed with status {0}",
                               status);
                    Context.Response.OutputStream.WriteByte((Byte)status);
                    return;
                }

                if (sAuthentication.GetPremiumLevel(pSession) == 0)
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "User {0} is not premium and attempted to get data hash",
                               userId);
                    Context.Response.OutputStream.WriteByte((Byte)StatusCode.AuthenticationNotPremium);
                    return;
                }

                String hash = null;
                using (SqlConnection conn = new SqlConnection(sRoamingDB))
                {
                    conn.Open();
                    using (SqlTransaction trans = conn.BeginTransaction())
                    {
                        SqlCommand cmd = new SqlCommand();
                        cmd.Connection = conn;
                        cmd.Transaction = trans;

                        cmd.CommandText = String.Format("SELECT hash FROM hash WHERE user_id = {0} AND game_id = {1}",
                                                        userId,
                                                        pGameId);
                        using (SqlDataReader dr = cmd.ExecuteReader())
                        {
                            if (dr.Read())
                            {
                                hash = Convert.ToString(dr["hash"]);
                            }
                        }

                        trans.Commit();
                    }
                }

                if (hash == null)
                {
                    Logger.Log(ELogLevel.Warning,
                               Context.Request.UserHostAddress,
                               "User {0} attempted to get data hash with no data",
                               userId);
                    Context.Response.OutputStream.WriteByte((Byte)StatusCode.ProfileNoData);
                    return;
                }

                Byte[] hashBytes = Encoding.ASCII.GetBytes(hash);

                Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);
                Context.Response.OutputStream.Write(hashBytes, 0, hashBytes.Length);
                return;
            }
            catch (Exception exc)
            {
                Logger.Log(ELogLevel.Error,
                           Context.Request.UserHostAddress,
                           "UploadData Exception: {0}",
                           exc.Message);
            }

            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Unknown);
            return;
        }

        /**
         * Publically exposed web method for downloading a roaming profile
         * Returns a StatusCode in the response stream
         * Writes the roaming profile as a file if successful
         * 
         * @param  pSession the session key
         * @param  pGameId  the game id to get the roaming profile hash from
         */
        [WebMethod]
        public void DownloadData(String pSession,
                                 Int32 pGameId)
        {
            try
            {
                if (!sGames.Contains(pGameId))
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "Invalid game id {0}",
                               pGameId);
                    Context.Response.OutputStream.WriteByte((Byte)StatusCode.ProfileInvalidGame);
                    return;
                }

                Int32 userId;
                StatusCode status = sAuthentication.ValidateSessionKeyToUserId(pSession, out userId);
                if (status != StatusCode.Ok)
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "ValidateSessionKeyToUserId failed with status {0}",
                               status);
                    Context.Response.OutputStream.WriteByte((Byte)status);
                    return;
                }

                if (sAuthentication.GetPremiumLevel(pSession) == 0)
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "User {0} is not premium and attempted to download data",
                               userId);
                    Context.Response.OutputStream.WriteByte((Byte)StatusCode.AuthenticationNotPremium);
                    return;
                }

                Int32 pathHash = Convert.ToInt32(Math.Floor(Convert.ToDouble(userId) / FILES_PER_DIRECTORY));
                String path = sDataPath + pGameId.ToString() + Path.DirectorySeparatorChar + pathHash.ToString();
                path += Path.DirectorySeparatorChar + userId.ToString() + ".dat";
                if (!File.Exists(path))
                {
                    Logger.Log(ELogLevel.Warning,
                               Context.Request.UserHostAddress,
                               "User {0} attempted to download data with no data",
                               userId);
                    Context.Response.OutputStream.WriteByte((Byte)StatusCode.ProfileNoData);
                    return;
                }

                Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);
                Context.Response.WriteFile(path, true);
                return;
            }
            catch (Exception exc)
            {
                Logger.Log(ELogLevel.Error,
                           Context.Request.UserHostAddress,
                           "UploadData Exception: {0}",
                           exc.Message);
            }

            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Unknown);
            return;
        }

        /**
         * Private static method to compute a SHA1 hash from a roaming profile on disk
         * Returns the SHA1 hash as a hex string lower cased
         * 
         * @param  pPath the path to the file to be hashed
         * @return       the SHA1 hash as a hex string
         */
        private static String ComputeDataHash(String pPath)
        {
            SHA1Managed sha1 = new SHA1Managed();
            Byte[] data = File.ReadAllBytes(pPath);
            Byte[] hash = sha1.ComputeHash(data);
            return BitConverter.ToString(hash).Replace("-", "").ToLower();
        }

        /**
         * Private constants
         */
        private const Int32 FILES_PER_DIRECTORY = 16384;

        /**
         * Private static member data
         */
        private static String sDataPath = null;
        private static String sRoamingDB = null;
        private static HashSet<Int32> sGames = new HashSet<Int32>();

    }
}
