﻿using System;
using System.Data;
using System.Data.SqlClient;
using System.ServiceModel;
using Curse.ServiceModels.Configuration;
using Curse.RewardPointsService.Models;
using Curse.RewardPointsService.Faults;

namespace Curse.RewardPointsService
{
    // NOTE: You can use the "Rename" command on the "Refactor" menu to change the class name "RewardPointsService" in code, svc and config file together.
    [ServiceBehavior(Name = "RewardPointsService", Namespace = "http://rewardpointsservice.curse.local/", AddressFilterMode = AddressFilterMode.Any, ConcurrencyMode = ConcurrencyMode.Single, InstanceContextMode = InstanceContextMode.Single)]
    public class RewardPointsService : IRewardPointsService
    {

        public int GetUserPoints(int userID)
        {
            int pointsTotalForUser;
            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["RewardPoints"].ConnectionString))
            {
                using (var cmd = new SqlCommand("GetUserRewardPointsTotal"))
                {
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.AddWithValue("@UserId", userID);
                    cmd.CommandTimeout = 120;

                    try
                    {
                        var userTotalPoints = new SqlParameter("@userTotalPoints", SqlDbType.Money) {Direction = ParameterDirection.Output};
                        cmd.Parameters.Add(userTotalPoints);
                        conn.Open();
                        cmd.Connection = conn;
                        cmd.ExecuteNonQuery();
                        var pointsDecimal = Decimal.Parse(cmd.Parameters["@userTotalPoints"].Value.ToString());
                        pointsTotalForUser = (Int32) Math.Floor(pointsDecimal);
                    }
                    catch (SqlException sqlExc)
                    {
                        var rewardFault = ProcessError(sqlExc.Message, PointFaultCode.GeneralSqlError);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralSqlError.ToString()));
                    }
                    catch (Exception exc)
                    {
                        var rewardFault = ProcessError(exc.Message, PointFaultCode.GeneralException);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralException.ToString()));
                    }
                }
                
            }
            return pointsTotalForUser;
        }

        public RewardPointTransactions GetUserTransactions(int userID, int page, int recordsPerPage)
        {
            var rewardpointTransactionsList = new RewardPointTransactions();

            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["RewardPoints"].ConnectionString))
            {
                using (var cmd = new SqlCommand("GetRewardPointTransactions"))
                {
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.AddWithValue("@userID", userID);
                    cmd.CommandTimeout = 120;
                    try
                    {
                        conn.Open();
                        cmd.Connection = conn;
                        using (var reader = cmd.ExecuteReader())
                        {
                            while (reader.Read())
                            {
                                var record = new Record();
                                record.SetFromDataReader(reader);
                                rewardpointTransactionsList.Add(record);
                            }
                        }
                    }
                    catch (SqlException sqlExc)
                    {
                        var rewardFault = ProcessError(sqlExc.Message, PointFaultCode.GeneralSqlError);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralSqlError.ToString()));
                    }
                    catch (Exception exc)
                    {
                        var rewardFault = ProcessError(exc.Message, PointFaultCode.GeneralException);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralException.ToString()));
                    }
                }
            }


            var lastPage = (rewardpointTransactionsList.Count / recordsPerPage) + 1;
            var pagedRewardpointTransactionsList = new RewardPointTransactions();
            pagedRewardpointTransactionsList.AddRange(rewardpointTransactionsList);
            if (lastPage > page)
            {
                if (page == 1)
                {
                    pagedRewardpointTransactionsList.RemoveRange(recordsPerPage, (pagedRewardpointTransactionsList.Count - recordsPerPage));
                }
                else
                {
                    pagedRewardpointTransactionsList.RemoveRange((page) * recordsPerPage, pagedRewardpointTransactionsList.Count - ((page) * recordsPerPage));
                }
                if (page > 1)
                {
                    pagedRewardpointTransactionsList.RemoveRange(0, ((page - 1) * recordsPerPage));
                }
                rewardpointTransactionsList.Clear();
                rewardpointTransactionsList.AddRange(pagedRewardpointTransactionsList);

            }
            else if (lastPage == page)
            {
                pagedRewardpointTransactionsList.RemoveRange(0, ((page - 1) * recordsPerPage));
                rewardpointTransactionsList.Clear();
                rewardpointTransactionsList.AddRange(pagedRewardpointTransactionsList);
            }
            else
            {
                rewardpointTransactionsList.Clear();
            }

            return rewardpointTransactionsList;
        }

        public RecordGenerationBreakdowns GetGenerationBreakdown(int recordID)
        {
            var recordGenerationBreakdownList = new RecordGenerationBreakdowns();

            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["RewardPoints"].ConnectionString))
            {
                using (var cmd = new SqlCommand("GetGenerationBreakdown"))
                {
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.AddWithValue("@recordID", recordID);
                    cmd.CommandTimeout = 120;
                    try
                    {
                        conn.Open();
                        cmd.Connection = conn;
                        using (var reader = cmd.ExecuteReader())
                        {
                            while (reader.Read())
                            {
                                var recordGenerationBreakdown = new RecordGenerationBreakdown();
                                recordGenerationBreakdown.SetFromDataReader(reader);
                                recordGenerationBreakdownList.Add(recordGenerationBreakdown);
                            }
                        }
                    }
                    catch (SqlException sqlExc)
                    {
                        var rewardFault = ProcessError(sqlExc.Message, PointFaultCode.GeneralSqlError);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralSqlError.ToString()));
                    }
                    catch (Exception exc)
                    {
                        var rewardFault = ProcessError(exc.Message, PointFaultCode.GeneralException);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralException.ToString()));
                    }
                }
            }

            return recordGenerationBreakdownList;
        }

        public int GetCountUserTransactions(int userID)
        {
            int recordCount;
            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["RewardPoints"].ConnectionString))
            {
                using (var cmd = new SqlCommand("GetCountUserTransactions"))
                {
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.AddWithValue("@UserId", userID);
                    cmd.CommandTimeout = 120;

                    try
                    {
                        var userTotalPoints = new SqlParameter("@recordCount", SqlDbType.Int) { Direction = ParameterDirection.Output };
                        cmd.Parameters.Add(userTotalPoints);
                        conn.Open();
                        cmd.Connection = conn;
                        cmd.ExecuteNonQuery();
                        recordCount = int.Parse(cmd.Parameters["@recordCount"].Value.ToString());
                        
                    }
                    catch (SqlException sqlExc)
                    {
                        var rewardFault = ProcessError(sqlExc.Message, PointFaultCode.GeneralSqlError);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralSqlError.ToString()));
                    }
                    catch (Exception exc)
                    {
                        var rewardFault = ProcessError(exc.Message, PointFaultCode.GeneralException);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralException.ToString()));
                    }
                }

            }
            return recordCount;
        }

        public bool SetMonthlyPremiumBudget(DateTime month, decimal budget)
        {
            if (budget < 0)
            {
                var rewardFault = new RewardFault { Code = PointFaultCode.NegativeBudget, Message = "Budget can not be a negative amount" };
                throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.NegativeBudget.ToString()));
            }
            var budgetSet = false;

            const string updateMonthlyBudgetQuery = @"UPDATE RewardPoints.dbo.MonthlyBudget SET PremiumBudget = @budget WHERE MonthlyBudgetDate = '@month'";

            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["RewardPoints"].ConnectionString))
            {
                using (var cmd = new SqlCommand(updateMonthlyBudgetQuery))
                {
                    try
                    {
                        cmd.Parameters.AddWithValue("@budget", budget);
                        cmd.Parameters.AddWithValue("@month", month);
                        conn.Open();
                        cmd.Connection = conn;
                        var rows = cmd.ExecuteNonQuery();
                        if (rows == 1)
                        {
                            budgetSet = true;
                        }
                    }
                    catch (SqlException sqlExc)
                    {
                        var rewardFault = ProcessError(sqlExc.Message, PointFaultCode.GeneralSqlError);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralSqlError.ToString()));
                    }
                    catch (FaultException<RewardFault> rfExc)
                    {
// ReSharper disable PossibleIntendedRethrow
                        throw rfExc;
// ReSharper restore PossibleIntendedRethrow
                    }
                    catch (Exception exc)
                    {
                        var rewardFault = ProcessError(exc.Message, PointFaultCode.GeneralException);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralException.ToString()));
                    }
                }
            }
            return budgetSet;
        }

        public MonthlyBudgets GetMonthlyBudgets(int page, int recordsPerPage)
        {
            MonthlyBudgets monthlyBudgetList = new MonthlyBudgets();

            const string query = @"SELECT * FROM RewardPoints.dbo.MonthlyBudget ORDER BY ID DESC";

            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["RewardPoints"].ConnectionString))
            {
                using (var cmd = new SqlCommand(query, conn))
                {
                    try
                    {
                        conn.Open();
                        using (var reader = cmd.ExecuteReader())
                        {
                            while (reader.Read())
                            {
                                MonthlyBudget monthlyBudget = new MonthlyBudget();
                                monthlyBudget.SetFromDataReader(reader);
                                monthlyBudgetList.Add(monthlyBudget);
                            }
                        }
                    }
                    catch (SqlException sqlExc)
                    {
                        var rewardFault = ProcessError(sqlExc.Message, PointFaultCode.GeneralSqlError);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralSqlError.ToString()));
                    }
                    catch (Exception exc)
                    {
                        var rewardFault = ProcessError(exc.Message, PointFaultCode.GeneralException);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralException.ToString()));
                    }
                }
            }
            var lastPage = (monthlyBudgetList.Count / recordsPerPage) + 1;
            var pagedmonthlyBudgetList = new MonthlyBudgets();
            pagedmonthlyBudgetList.AddRange(monthlyBudgetList);
            if (lastPage >  page)
            {
                if (page == 1)
                {
                    pagedmonthlyBudgetList.RemoveRange(recordsPerPage, (pagedmonthlyBudgetList.Count - recordsPerPage));
                }
                else
                {
                    pagedmonthlyBudgetList.RemoveRange(((page) * recordsPerPage), (pagedmonthlyBudgetList.Count - ((page) * recordsPerPage)));
                }
                if (page > 1)
                {
                    pagedmonthlyBudgetList.RemoveRange(0, ((page-1) * recordsPerPage));
                }
                monthlyBudgetList.Clear();
                monthlyBudgetList.AddRange(pagedmonthlyBudgetList);

            }
            else if (lastPage == page)
                {
                    pagedmonthlyBudgetList.RemoveRange(0, ((page-1) * recordsPerPage));
                    monthlyBudgetList.Clear();
                    monthlyBudgetList.AddRange(pagedmonthlyBudgetList);
                }
            else
            {
                monthlyBudgetList.Clear();
            }

            return monthlyBudgetList;
        }

        public bool SpendPoints(int userID, decimal amount, int orderID)
        {
            if (amount >= 0)
            {
                var rewardFault = new RewardFault { Code = PointFaultCode.SpendPointsNotnegative, Message = "Points to spend should be negative" };
                throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.SpendPointsNotnegative.ToString()));
            }
            var pointsSpent = false;
            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["RewardPoints"].ConnectionString))
            {
                using (var cmd = new SqlCommand("RecordUserSpendingPoints"))
                {
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.AddWithValue("@userID", userID);
                    cmd.Parameters.AddWithValue("@amount", amount);
                    cmd.Parameters.AddWithValue("@orderID", orderID);
                    cmd.CommandTimeout = 120;

                    try
                    {
                        var returnValue = new SqlParameter("@Return_Value", DbType.Int32) { Direction = ParameterDirection.ReturnValue };
                        cmd.Parameters.Add(returnValue);
                        var errorMessage = new SqlParameter("@errorMessage", SqlDbType.VarChar, 400) { Direction = ParameterDirection.Output };
                        cmd.Parameters.Add(errorMessage);

                        conn.Open();
                        cmd.Connection = conn;

                        cmd.ExecuteNonQuery();
                        var returnCode = Int32.Parse(cmd.Parameters["@Return_Value"].Value.ToString());

                        if (returnCode != 1)
                        {
                            
                            switch (returnCode)
                            {
                                case (int)PointFaultCode.UserDoesNotHaveEnoughPoints:
                                    {
                                        var rewardFault = new RewardFault { Code = PointFaultCode.UserDoesNotHaveEnoughPoints, Message = cmd.Parameters["@errorMessage"].Value.ToString() };
                                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.UserDoesNotHaveEnoughPoints.ToString()));
                                    }

                                case (int)PointFaultCode.OrderAlreadyCompleted:
                                    {
                                        var rewardFault = new RewardFault {Code = PointFaultCode.OrderAlreadyCompleted, Message = cmd.Parameters["@errorMessage"].Value.ToString()};
                                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.OrderAlreadyCompleted.ToString()));
                                    }
                                case (int)PointFaultCode.OrderAlreadyCompletedDifferentAmount:
                                    {
                                        var rewardFault = new RewardFault { Code = PointFaultCode.OrderAlreadyCompletedDifferentAmount, Message = cmd.Parameters["@errorMessage"].Value.ToString() };
                                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.OrderAlreadyCompletedDifferentAmount.ToString()));
                                    }
                                case (int)PointFaultCode.OrderAlreadyCompletedDifferentUser:
                                    {
                                        var rewardFault = new RewardFault { Code = PointFaultCode.OrderAlreadyCompletedDifferentUser, Message = cmd.Parameters["@errorMessage"].Value.ToString() };
                                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.OrderAlreadyCompletedDifferentUser.ToString()));
                                    }
                                case (int)PointFaultCode.TransactionSqlError:
                                    {
                                        var rewardFault = new RewardFault { Code = PointFaultCode.TransactionSqlError, Message = cmd.Parameters["@errorMessage"].Value.ToString() };
                                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.TransactionSqlError.ToString()));
                                    }
                            }

                        }
                        else
                        {
                            pointsSpent = true;
                        }
                    }
                    catch (SqlException sqlExc)
                    {
                        var rewardFault = ProcessError(sqlExc.Message, PointFaultCode.GeneralSqlError);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralSqlError.ToString()));
                    }
                    catch (FaultException<RewardFault> rfExc)
                    {
// ReSharper disable PossibleIntendedRethrow
                        throw rfExc;
// ReSharper restore PossibleIntendedRethrow
                    }
                    catch (Exception exc)
                    {
                        var rewardFault = ProcessError(exc.Message, PointFaultCode.GeneralException);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralException.ToString()));
                    }
                }
            }
            return pointsSpent;
        }

        public bool RefundPoints(int userID, decimal amount, int orderID)
        {
            if (amount <= 0)
            {
                var rewardFault = new RewardFault { Code = PointFaultCode.RefundPointsNotPositive, Message = "Points to refund should be positive" };
                throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.RefundPointsNotPositive.ToString()));
            }
            var pointsRefunded = false;
            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["RewardPoints"].ConnectionString))
            {
                using (var cmd = new SqlCommand("RecordUserRefundedPoints"))
                {
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.AddWithValue("@userID", userID);
                    cmd.Parameters.AddWithValue("@amount", amount);
                    cmd.Parameters.AddWithValue("@orderID", orderID);
                    cmd.CommandTimeout = 120;

                    try
                    {
                        var returnValue = new SqlParameter("@Return_Value", DbType.Int32) { Direction = ParameterDirection.ReturnValue };
                        cmd.Parameters.Add(returnValue);
                        var errorMessage = new SqlParameter("@errorMessage", SqlDbType.VarChar, 400) { Direction = ParameterDirection.Output };
                        cmd.Parameters.Add(errorMessage);

                        conn.Open();
                        cmd.Connection = conn;

                        cmd.ExecuteNonQuery();
                        var returnCode = Int32.Parse(cmd.Parameters["@Return_Value"].Value.ToString());

                        if (returnCode != 1)
                        {

                            switch (returnCode)
                            {
                                case (int)PointFaultCode.RefundedAlready:
                                    {
                                        var rewardFault = new RewardFault { Code = PointFaultCode.RefundedAlready, Message = cmd.Parameters["@errorMessage"].Value.ToString() };
                                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.RefundedAlready.ToString()));
                                    }

                                case (int)PointFaultCode.RefundedWithDifferentAmount:
                                    {
                                        var rewardFault = new RewardFault { Code = PointFaultCode.RefundedWithDifferentAmount, Message = cmd.Parameters["@errorMessage"].Value.ToString() };
                                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.RefundedWithDifferentAmount.ToString()));
                                    }
                                case (int)PointFaultCode.OrderUserComboNotFound:
                                    {
                                        var rewardFault = new RewardFault { Code = PointFaultCode.OrderUserComboNotFound, Message = cmd.Parameters["@errorMessage"].Value.ToString() };
                                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.OrderUserComboNotFound.ToString()));
                                    }
                                case (int)PointFaultCode.OrderUserFoundWithDiferentAmount:
                                    {
                                        var rewardFault = new RewardFault { Code = PointFaultCode.OrderUserFoundWithDiferentAmount, Message = cmd.Parameters["@errorMessage"].Value.ToString() };
                                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.OrderUserFoundWithDiferentAmount.ToString()));
                                    }
                                case (int)PointFaultCode.TransactionSqlError:
                                    {
                                        var rewardFault = new RewardFault { Code = PointFaultCode.TransactionSqlError, Message = cmd.Parameters["@errorMessage"].Value.ToString() };
                                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.TransactionSqlError.ToString()));
                                    }
                            }

                        }
                        else
                        {
                            pointsRefunded = true;
                        }
                    }
                    catch (SqlException sqlExc)
                    {
                        var rewardFault = ProcessError(sqlExc.Message, PointFaultCode.GeneralSqlError);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralSqlError.ToString()));
                    }
                    catch (FaultException<RewardFault> rfExc)
                    {
// ReSharper disable PossibleIntendedRethrow
                        throw rfExc;
// ReSharper restore PossibleIntendedRethrow
                    }
                    catch (Exception exc)
                    {
                        var rewardFault = ProcessError(exc.Message, PointFaultCode.GeneralException);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralException.ToString()));
                    }
                }
            }
            return pointsRefunded;
        }

        public bool TransferPoints(int senderUserID, int recipientUserID, decimal amountToGive)
        {
            if (amountToGive <= 0)
            {
                var rewardFault = new RewardFault { Code = PointFaultCode.TransferPointsNotPositive, Message = "Points to tranfer should be positive" };
                throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.TransferPointsNotPositive.ToString()));
            }
            var pointsTransferded = false;
            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["RewardPoints"].ConnectionString))
            {
                using (var cmd = new SqlCommand("RecordUserTranferingPoints"))
                {
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.AddWithValue("@senderUserID", senderUserID);
                    cmd.Parameters.AddWithValue("@amountToGive", amountToGive);
                    cmd.Parameters.AddWithValue("@recipientUserID", recipientUserID);
                    cmd.CommandTimeout = 120;

                    try
                    {
                        var returnValue = new SqlParameter("@Return_Value", DbType.Int32) {Direction = ParameterDirection.ReturnValue};
                        cmd.Parameters.Add(returnValue);
                        var errorMessage = new SqlParameter("@errorMessage", SqlDbType.VarChar, 400) {Direction = ParameterDirection.Output};
                        cmd.Parameters.Add(errorMessage);

                        conn.Open();
                        cmd.Connection = conn;

                        cmd.ExecuteNonQuery();
                        var returnCode = Int32.Parse(cmd.Parameters["@Return_Value"].Value.ToString());

                        if (returnCode != 1)
                        {

                            switch (returnCode)
                            {
                                case (int) PointFaultCode.UserNotEnoughPointsToTranfer:
                                    {
                                        var rewardFault = new RewardFault {Code = PointFaultCode.UserNotEnoughPointsToTranfer, Message = cmd.Parameters["@errorMessage"].Value.ToString()};
                                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.UserNotEnoughPointsToTranfer.ToString()));
                                    }

                                case (int) PointFaultCode.RecipientNotEligeble:
                                    {
                                        var rewardFault = new RewardFault {Code = PointFaultCode.RecipientNotEligeble, Message = cmd.Parameters["@errorMessage"].Value.ToString()};
                                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.RecipientNotEligeble.ToString()));
                                    }
                                case (int) PointFaultCode.TransactionSqlError:
                                    {
                                        var rewardFault = new RewardFault {Code = PointFaultCode.TransactionSqlError, Message = cmd.Parameters["@errorMessage"].Value.ToString()};
                                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.TransactionSqlError.ToString()));
                                    }
                            }

                        }
                        else
                        {
                            pointsTransferded = true;
                        }
                    }
                    catch (SqlException sqlExc)
                    {
                        var rewardFault = ProcessError(sqlExc.Message, PointFaultCode.GeneralSqlError);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralSqlError.ToString()));
                    }
                    catch (FaultException<RewardFault> rfExc)
                    {
// ReSharper disable PossibleIntendedRethrow
                        throw rfExc;
// ReSharper restore PossibleIntendedRethrow
                    }
                    catch (Exception exc)
                    {
                        var rewardFault = ProcessError(exc.Message, PointFaultCode.GeneralException);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralException.ToString()));
                    }
                }
            }
            return pointsTransferded;
        }

        public Decimal GetBonusPoints()
        {
            Object availableBounsPoints;

            const string query = @"SELECT BonusPoints FROM RewardPoints.dbo.RewardsProgram";

            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["RewardPoints"].ConnectionString))
            {
                try
                {
                    var cmd = new SqlCommand(query, conn);
                    conn.Open();
                    availableBounsPoints = cmd.ExecuteScalar();
                    if (availableBounsPoints == DBNull.Value)
                    {
                        availableBounsPoints = 0.0;
                    }
                }
                catch (SqlException sqlExc)
                {
                    var rewardFault = ProcessError(sqlExc.Message, PointFaultCode.GeneralSqlError);
                    throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralSqlError.ToString()));
                }
                catch (Exception exc)
                {
                    var rewardFault = ProcessError(exc.Message, PointFaultCode.GeneralException);
                    throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralException.ToString()));
                }
                
            }
            return (Decimal)availableBounsPoints;
        }

        public bool AwardBonusPoints(int userID, decimal amount)
        {
            var bonusPointsAwarded = false;

            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["RewardPoints"].ConnectionString))
            {
                using (var cmd = new SqlCommand("AwardBonusPointsToUser"))
                {
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.AddWithValue("@userID", userID);
                    cmd.Parameters.AddWithValue("@bonusPointsGiven", amount);
                    cmd.CommandTimeout = 120;

                    try
                    {
                        var returnValue = new SqlParameter("@Return_Value", DbType.Int32) {Direction = ParameterDirection.ReturnValue};
                        cmd.Parameters.Add(returnValue);
                        var errorMessage = new SqlParameter("@errorMessage", SqlDbType.VarChar, 400) {Direction = ParameterDirection.Output};
                        cmd.Parameters.Add(errorMessage);

                        conn.Open();
                        cmd.Connection = conn;

                        cmd.ExecuteNonQuery();
                        var returnCode = (Int32)cmd.Parameters["@Return_Value"].Value;

                        if (returnCode != 1)
                        {

                            switch (returnCode)
                            {
                                case (int)PointFaultCode.NotEnoughBounsPoints:
                                    {
                                        var rewardFault = new RewardFault { Code = PointFaultCode.NotEnoughBounsPoints, Message = cmd.Parameters["@errorMessage"].Value.ToString() };
                                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.NotEnoughBounsPoints.ToString()));
                                    }

                                case (int)PointFaultCode.UserNotInRewardProgram:
                                    {
                                        var rewardFault = new RewardFault { Code = PointFaultCode.UserNotInRewardProgram, Message = cmd.Parameters["@errorMessage"].Value.ToString() };
                                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.UserNotInRewardProgram.ToString()));
                                    }
                                case (int)PointFaultCode.TransactionSqlError:
                                    {
                                        var rewardFault = new RewardFault { Code = PointFaultCode.TransactionSqlError, Message = cmd.Parameters["@errorMessage"].Value.ToString() };
                                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.TransactionSqlError.ToString()));
                                    }
                            }

                        }
                        else
                        {
                            bonusPointsAwarded = true;
                        }
                    }
                    catch (SqlException sqlExc)
                    {
                        var rewardFault = ProcessError(sqlExc.Message, PointFaultCode.GeneralSqlError);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralSqlError.ToString()));
                    }
                    catch (FaultException<RewardFault> rfExc)
                    {
                        // ReSharper disable PossibleIntendedRethrow
                        throw rfExc;
                        // ReSharper restore PossibleIntendedRethrow
                    }
                    catch (Exception exc)
                    {
                        var rewardFault = ProcessError(exc.Message, PointFaultCode.GeneralException);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralException.ToString()));
                    }
                }
            }

            return bonusPointsAwarded;
        }

        public bool AddProjectToBlacklist(int projectID)
        {
            var projectedAdded = false;

            const string insertQuery = @"INSERT INTO RewardPoints.dbo.ProjectBlackList (ProjectID) VALUES (@projectID)";


            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["RewardPoints"].ConnectionString))
            {
                using (var cmd = new SqlCommand(insertQuery))
                {
                    try
                    {
                        cmd.Parameters.AddWithValue("@projectID", projectID);
                        conn.Open();
                        cmd.Connection = conn;
                        var rows = cmd.ExecuteNonQuery();
                        if (rows == 1)
                        {
                            projectedAdded = true;
                        }
                    }
                    catch (SqlException sqlExc)
                    {
                        var rewardFault = ProcessError(sqlExc.Message, PointFaultCode.GeneralSqlError);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralSqlError.ToString()));
                    }
                    catch (Exception exc)
                    {
                        var rewardFault = ProcessError(exc.Message, PointFaultCode.GeneralException);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralException.ToString()));
                    }
                }
            }
            return projectedAdded;
        }

        public bool RemoveProjectFromBlacklist(int projectID)
        {
            var projectRemoved = false;

            const string insertQuery = @"DELETE FROM RewardPoints.dbo.ProjectBlackList WHERE ProjectID = @projectID";

            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["RewardPoints"].ConnectionString))
            {
                using (var cmd = new SqlCommand(insertQuery))
                {
                    try
                    {
                        cmd.Parameters.AddWithValue("@projectID", projectID);
                        conn.Open();
                        cmd.Connection = conn;
                        var rows = cmd.ExecuteNonQuery();
                        if (rows == 1)
                        {
                            projectRemoved = true;
                        }
                    }
                    catch (SqlException sqlExc)
                    {
                        var rewardFault = ProcessError(sqlExc.Message, PointFaultCode.GeneralSqlError);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralSqlError.ToString()));
                    }
                    catch (Exception exc)
                    {
                        var rewardFault = ProcessError(exc.Message, PointFaultCode.GeneralException);
                        throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralException.ToString()));
                    }
                }
            }
            return projectRemoved;
        }

        public Decimal GetPointExchangeRate()
        {
            Decimal pointExchangeRate;

            const string query = @"SELECT PointExchangeRate FROM RewardsProgram";

            using (var conn = new SqlConnection(DatabaseConfiguration.Instance["RewardPoints"].ConnectionString))
            {
                try
                {
                    var cmd = new SqlCommand(query, conn);
                    conn.Open();
                    pointExchangeRate = (Decimal)cmd.ExecuteScalar();
                }
                catch (SqlException sqlExc)
                {
                    var rewardFault = ProcessError(sqlExc.Message, PointFaultCode.GeneralSqlError);
                    throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralSqlError.ToString()));
                }
                catch (Exception exc)
                {
                    var rewardFault = ProcessError(exc.Message, PointFaultCode.GeneralException);
                    throw new FaultException<RewardFault>(rewardFault, new FaultReason(PointFaultCode.GeneralException.ToString()));
                }
                
            }

            return pointExchangeRate;
        }


        private static RewardFault ProcessError(String errorMessage, PointFaultCode pfCode)
        {
            Logger.Log("Error in GetUserPoints. Details: {0}", ELogLevel.Error, errorMessage);
            var rewardFault = new RewardFault { Code = pfCode, Message = errorMessage };
            return rewardFault;
        }

    }
}
