﻿using System;
using System.Collections.Generic;
using System.ServiceModel;
using System.Windows.Forms;
using System.Diagnostics;

using Curse.CurseClient.Common.ClientService;
using Curse.ClientService.Models;
using Curse.DownloadSecurity;
using Curse.AddOns;
using System.ComponentModel;
using Curse.DownloadSecurity.Tokens;
//using Curse.CurseClient.Enumerations;
using Curse.CurseClient.Common.ClientBackupService;
using System.IO;
using System.Net.Sockets;
using System.Net;
using System.Xml;
using System.Text;
using Curse.CurseClient.Common.ClientRegistrationService;
using System.ServiceModel.Channels;

namespace Curse.CurseClient.Common
{
    public class CServiceProxy : BaseSingleton<CServiceProxy>
    {

        #region Events

        public event EventHandler<CEventArgs<bool, string>> ServiceDisconnected;
        public event EventHandler ServiceConnected;

        #endregion

        #region Fields

        private static readonly Uri _soapClientServiceUri = new Uri(CClientServiceConstants.ServiceHost + "CClientService.svc/Soap12");
        private static readonly Uri _binaryClientServiceUri = new Uri(CClientServiceConstants.ServiceHost + "CClientService.svc/Binary");
        private static readonly Uri _registrationServiceUri = new Uri(CClientServiceConstants.ServiceHost + "CClientRegistrationService.svc");
        private static readonly Uri _backupServiceUri = new Uri(CClientServiceConstants.UploadHost + "CClientBackupService.svc");

        private object _shutdownLock = new object();
        private CClientServiceClient _serviceClient;
        private string _plainTextPassword;
        private string _encryptedPassword;
        private bool _isPremium;
        private bool _isSubscribed;

        #endregion

        #region Properties

        public string Username
        {
            get;
            set;
        }

        public string EncryptedPassword
        {
            get
            {
                return _encryptedPassword;
            }
            set
            {
                if (_encryptedPassword != value)
                {
                    _encryptedPassword = value;
                    if (!string.IsNullOrEmpty(value))
                    {
                        _plainTextPassword = CClientLogin.DecryptLocalString(value);
                    }
                    else
                    {
                        _plainTextPassword = string.Empty;
                    }

                }
            }
        }

        public string PlainTextPassword
        {
            get
            {
                return _plainTextPassword;
            }
            set
            {
                if (_plainTextPassword != value)
                {
                    _plainTextPassword = value;
                    _encryptedPassword = CClientLogin.EncryptLocalString(value);
                }
            }
        }

        public int UserID
        {
            get;
            set;
        }

        public string SessionID
        {
            get;
            set;
        }

        public int SubscriptionToken
        {
            get;
            set;
        }

        public bool IsPremium
        {
            get
            {
                return _isPremium;
            }
            set
            {
                if (_isPremium != value)
                {
                    _isPremium = value;
                    OnPropertyChanged("IsPremium");
                }
            }
        }

        public bool IsSubscribed
        {
            get
            {
                return _isSubscribed;
            }
            set
            {
                if (_isSubscribed != value)
                {
                    _isSubscribed = value;
                    OnPropertyChanged("IsSubscribed");
                }
            }
        }

        public bool IsConnected
        {
            get;
            set;
        }


        private CClientServiceClient Client
        {
            get
            {
                if (_serviceClient == null)
                {
                    Reconnect();
                }
                return _serviceClient;
            }
        }

        #endregion

        public CServiceProxy()
        {
            IsConnected = false;
            ServiceConnected += new EventHandler(CServiceProxy_ServiceConnected);
            ServiceDisconnected += new EventHandler<CEventArgs<bool, string>>(CServiceProxy_ServiceDisconnected);
        }

        void CServiceProxy_ServiceDisconnected(object sender, CEventArgs<bool> e)
        {
            IsConnected = false;
        }

        void CServiceProxy_ServiceConnected(object sender, EventArgs e)
        {
            IsConnected = true;
        }

        #region Static Methods

        private static XmlDictionaryReaderQuotas GetDefaultReaderQuotas()
        {
            XmlDictionaryReaderQuotas readerQuotas = new XmlDictionaryReaderQuotas();
            readerQuotas.MaxArrayLength = 16384;
            readerQuotas.MaxBytesPerRead = 4096;
            readerQuotas.MaxNameTableCharCount = 16384;
            readerQuotas.MaxDepth = 64;
            readerQuotas.MaxStringContentLength = 640000;
            return readerQuotas;
        }

        private static CustomBinding GetBinaryHttpBinding()
        {

            CustomBinding binding = new CustomBinding();

            binding.CloseTimeout = TimeSpan.FromSeconds(10);
            binding.OpenTimeout = TimeSpan.FromSeconds(10);
            binding.ReceiveTimeout = TimeSpan.FromMinutes(1);
            binding.SendTimeout = TimeSpan.FromMinutes(1);

            // Binary Encoding
            BinaryMessageEncodingBindingElement binaryEncoding = new BinaryMessageEncodingBindingElement();
            binaryEncoding.ReaderQuotas.MaxArrayLength = 16384;
            binaryEncoding.ReaderQuotas.MaxBytesPerRead = 4096;
            binaryEncoding.ReaderQuotas.MaxNameTableCharCount = 16384;
            binaryEncoding.ReaderQuotas.MaxDepth = 64;
            binaryEncoding.ReaderQuotas.MaxStringContentLength = 640000;
            binding.Elements.Add(binaryEncoding);

            // Http Transport
            HttpTransportBindingElement httpTransport = new HttpTransportBindingElement();
            httpTransport.AuthenticationScheme = AuthenticationSchemes.Anonymous;
            httpTransport.MaxBufferPoolSize = 524288;
            httpTransport.MaxReceivedMessageSize = 2147483647;
            httpTransport.MaxBufferSize = 2147483647;
            httpTransport.AllowCookies = false;
            httpTransport.HostNameComparisonMode = HostNameComparisonMode.StrongWildcard;
            httpTransport.TransferMode = TransferMode.Buffered;
            binding.Elements.Add(httpTransport);

            return binding;
        }

        private static WSHttpBinding GetDefaultWSHttpBinding()
        {
            // Reader quotas
            XmlDictionaryReaderQuotas readerQuotas = GetDefaultReaderQuotas();

            WSHttpBinding binding = new WSHttpBinding(SecurityMode.None, false);
            binding.AllowCookies = false;
            binding.CloseTimeout = TimeSpan.FromSeconds(10);
            binding.OpenTimeout = TimeSpan.FromSeconds(10);
            binding.ReceiveTimeout = TimeSpan.FromMinutes(1);
            binding.SendTimeout = TimeSpan.FromMinutes(1);
            binding.UseDefaultWebProxy = true;
            binding.BypassProxyOnLocal = false;
            binding.TransactionFlow = false;
            binding.HostNameComparisonMode = HostNameComparisonMode.StrongWildcard;
            binding.MaxBufferPoolSize = 524288;
            binding.MaxReceivedMessageSize = 2097152;
            binding.MessageEncoding = WSMessageEncoding.Text;
            binding.TextEncoding = Encoding.UTF8;
            binding.ReaderQuotas = readerQuotas;

            // Disable all builtin security
            binding.Security.Message.EstablishSecurityContext = false;
            binding.Security.Message.ClientCredentialType = MessageCredentialType.None;
            binding.Security.Message.NegotiateServiceCredential = false;
            binding.Security.Transport.ClientCredentialType = HttpClientCredentialType.None;

            return binding;
        }

        private static BasicHttpBinding GetDefaultBasicHttpBinding()
        {
            // Reader quotas
            XmlDictionaryReaderQuotas readerQuotas = GetDefaultReaderQuotas();

            // Basic Http Binding
            BasicHttpBinding binding = new BasicHttpBinding(BasicHttpSecurityMode.None);
            binding.AllowCookies = false;
            binding.CloseTimeout = TimeSpan.FromSeconds(10);
            binding.OpenTimeout = TimeSpan.FromSeconds(10);
            binding.ReceiveTimeout = TimeSpan.FromSeconds(30);
            binding.SendTimeout = TimeSpan.FromMinutes(5);
            binding.UseDefaultWebProxy = true;
            binding.BypassProxyOnLocal = false;
            binding.HostNameComparisonMode = HostNameComparisonMode.StrongWildcard;
            binding.MaxBufferPoolSize = 524288;
            binding.MaxBufferSize = 65536;
            binding.MaxReceivedMessageSize = 65536;
            binding.MessageEncoding = WSMessageEncoding.Text;
            binding.TextEncoding = Encoding.UTF8;
            binding.TransferMode = TransferMode.StreamedRequest;
            binding.ReaderQuotas = readerQuotas;

            return binding;
        }

        public static CClientRegistrationServiceClient GetRegistrationServiceClient()
        {
            Logger.Log("Creating new WCF client for endpoint at: '{0}'", ELogLevel.Info, _registrationServiceUri.AbsoluteUri);

            // The binding configuration
            WSHttpBinding binding = GetDefaultWSHttpBinding();

            // Create the client
            CClientRegistrationServiceClient client = new CClientRegistrationServiceClient(binding, new EndpointAddress(_registrationServiceUri));

            Logger.Log("WCF client has been created for endpoint at: '{0}'", ELogLevel.Info, _registrationServiceUri.AbsoluteUri);

            return client;
        }

        public static CClientBackupServiceClient GetBackupServiceClient()
        {
            Uri serviceUri = _backupServiceUri;

            Logger.Log("Creating new WCF client for endpoint at: '{0}'", ELogLevel.Debug, serviceUri.AbsoluteUri);

            // The binding configuration
            BasicHttpBinding binding = GetDefaultBasicHttpBinding();

            // Create the client
            CClientBackupServiceClient client = new CClientBackupServiceClient(binding, new EndpointAddress(serviceUri));

            client.ChannelFactory.Endpoint.Behaviors.Add(new CAuthTokenExtension());

            Logger.Log("WCF client has been created for endpoint at: '{0}'", ELogLevel.Debug, serviceUri.AbsoluteUri);

            return client;
        }

        public static CClientServiceClient GetSoapClientServiceClient()
        {

            Uri serviceUri = _soapClientServiceUri;

            string encryptedPassword = CClientLogin.EncryptOutgoingString(CServiceProxy.Instance.PlainTextPassword);

            Logger.Log("Creating new WCF client for endpoint at: '{0}'", ELogLevel.Info, serviceUri.AbsoluteUri);

            // Set the authentication context
            CAuthToken.Outgoing = new CAuthToken(CServiceProxy.Instance.Username, encryptedPassword);

            // The binding configuration
            WSHttpBinding binding = GetDefaultWSHttpBinding();

            // Create the client
            CClientServiceClient client = new CClientServiceClient(binding, new EndpointAddress(serviceUri));


            // Add the behaviour to send the credentials with each request
            client.ChannelFactory.Endpoint.Behaviors.Add(new CAuthTokenExtension());

            Logger.Log("WCF client has been created for endpoint at: '{0}'", ELogLevel.Info, serviceUri.AbsoluteUri);

            return client;
        }

        public static CClientServiceClient GetBinaryClientServiceClient()
        {
            Uri serviceUri = _binaryClientServiceUri;

            Logger.Log("Creating new WCF client for endpoint at: '{0}'", ELogLevel.Info, serviceUri.AbsoluteUri);

            string encryptedPassword = CClientLogin.EncryptOutgoingString(CServiceProxy.Instance.PlainTextPassword);

            // Set the authentication context
            CAuthToken.Outgoing = new CAuthToken(CServiceProxy.Instance.Username, encryptedPassword);

            // The binding configuration
            CustomBinding binding = GetBinaryHttpBinding();

            // Create the client
            CClientServiceClient client = new CClientServiceClient(binding, new EndpointAddress(serviceUri));


            // Add the behaviour to send the credentials with each request
            client.ChannelFactory.Endpoint.Behaviors.Add(new CAuthTokenExtension());

            Logger.Log("WCF client has been created for endpoint at: '{0}'", ELogLevel.Info, serviceUri.AbsoluteUri);

            return client;
        }

        #endregion


        #region Instance Methods

        public CLoginResult LoginUser()
        {
            try
            {
                Reconnect();

                CSession session = _serviceClient.GetCurrentSession();

                UserID = session.UserID;
                SessionID = session.SessionID;
                IsPremium = session.EffectivePremiumStatus;
                IsSubscribed = session.ActualPremiumStatus;

                SubscriptionToken = session.SubscriptionToken;

                if (Username.ToLowerInvariant() == "zusername")
                {
                    IsPremium = false;
                }

                CLoginResult result = new CLoginResult()
                {
                    IsSuccessful = true,
                    Status = EAuthenticationStatus.Success
                };

                ServiceConnected(null, null);

                return result;
            }
            catch (Exception ex)
            {
                int reasonCode = -1;

                if (int.TryParse(ex.Message, out reasonCode))
                {
                    EAuthenticationStatus status = (EAuthenticationStatus)reasonCode;
                    Logger.Log("Login - Authentication Failure: {0}", ELogLevel.Info, status.ToString());
                }
                else
                {
                    Logger.Log("Login - Unhandled Exception! Details: {0}", ELogLevel.Error, ex.CompleteDetails());
                }

                return CLoginResult.GetFromException(ex);
            }
        }

        public delegate T CallDelegate<T>(CClientServiceClient client);

        private bool IsConnectionAvailable
        {
            get
            {
                try
                {
                    using (TcpClient clnt = new TcpClient("www.google.com", 80))
                    {
                        clnt.Close();
                        return true;
                    }
                }
                catch
                {
                    return false;
                }
            }
        }

        public T TryCall<T>(CallDelegate<T> call)
        {
            return this.TryCall<T>(call, 2);
        }

        public T TryCall<T>(CallDelegate<T> call, int attemptLimit)
        {
            T result = default(T);
            int attemptCounter = 0;
            bool continueAttempting = true;

            while (continueAttempting)
            {
                ++attemptCounter;

                if (attemptCounter > attemptLimit)
                {
                    break;
                }

                try
                {
                    return call(Client);
                }
                catch (FaultException ex)
                {
                    Logger.Log("Service Proxy - Fault Exception! Details: {0}", ELogLevel.Info, ex.CompleteDetails());

                    int reasonCode = -1;

                    if (int.TryParse(ex.Message, out reasonCode))
                    {
                        this.Disconnect(ex, false, reasonCode);

                        throw;
                    }
                    else if (attemptCounter >= attemptLimit)
                    {
                        throw;
                    }
                }
                catch (EndpointNotFoundException ex)
                {
                    if (!IsConnectionAvailable)
                    {
                        Logger.Log("Service Proxy - An active Internet connection is not currently available.", ELogLevel.Info);
                        throw;
                    }

                    this.Disconnect(ex, false, "Your connection to the Curse Client web service has been lost. Please check the status of your Internet connection.");
                    throw;
                }
                catch (CommunicationException ex)
                {
                    if (!IsConnectionAvailable)
                    {
                        Logger.Log("Service Proxy - An active Internet connection is not currently available.", ELogLevel.Info);
                        throw;
                    }

                    Logger.Log("Service Proxy - Communication Exception! Details: {0}", ELogLevel.Info, ex.CompleteDetails());

                    if (attemptCounter >= attemptLimit)
                    {
                        throw;
                    }
                    else
                    {
                        this.Reconnect();
                    }

                }
                catch (Exception ex)
                {
                    Logger.Log("Service Proxy - Unknown Exception! Details: {0}", ELogLevel.Info, ex.CompleteDetails());

                    throw;
                }

            }

            return result;
        }

        public void Logout()
        {
            Logger.Log("Login - User Logged Out", ELogLevel.Info);

            this.Disconnect(null, true, "Logging Out");
        }

        private void Disconnect(Exception ex, bool isLoggingOut, int reasonCode)
        {

            EAuthenticationStatus status = (EAuthenticationStatus)reasonCode;
            string baseMessage = "Your connection to the Curse Client web service has been disconnected for the following reason:\r\n\r\n";
            string message = null;

            switch (status)
            {
                case EAuthenticationStatus.InvalidPassword:
                    message = baseMessage + "Your password is no longer valid";
                    break;

                case EAuthenticationStatus.UnknownEmail:
                    message = baseMessage + "Your e-mail address is no longer valid";
                    break;

                case EAuthenticationStatus.UnknownUsername:
                    message = baseMessage + "Your username is no longer valid";
                    break;

                case EAuthenticationStatus.SubscriptionExpired:
                    message = "Due to a change in the status of your Curse Premium subscription, you have been automatically logged out.\r\n\r\nPlease check that it has not expired at: http://www.curse.com/Premium";
                    break;

                case EAuthenticationStatus.SubscriptionMismatch:
                    message = "Due to a change in the status of your Curse Premium subscription, you have been automatically logged out.\r\n\r\nIf you recently subscribed to Curse Premium, simply log back in to receive Curse Premium benefits.\r\n\r\nIf you have had a Curse Premium subscription for a while, please check that it has not expired at: http://www.curse.com/Premium";
                    break;

                case EAuthenticationStatus.UnknownError:
                default:
                    message = baseMessage + ex.Message;
                    break;

            }

            this.Disconnect(ex, isLoggingOut, message);
        }

        private void Disconnect(Exception ex, bool isLoggingOut, string reason)
        {
            if (IsConnected)
            {
                ServiceDisconnected(null, new CEventArgs<bool, string>(isLoggingOut, reason));
            }

            if (ex != null)
            {
                Logger.Log("Service Disconnection! Details: {0}", ELogLevel.Error, ex.CompleteDetails());
            }

            this.CloseConnection();
        }

        private bool ConnectionIsFaulted(CommunicationException ex)
        {
            // Underlying channel is definitely faulted.
            if (_serviceClient == null
                || _serviceClient.InnerChannel == null
                || _serviceClient.InnerChannel.State == CommunicationState.Faulted
                || _serviceClient.InnerChannel.State == CommunicationState.Closed)
            {
                return true;
            }

            // The exception is related to a faulted state.
            if (ex.GetBaseException() is FaultException
                || ex.GetBaseException() is CommunicationObjectFaultedException
                || ex is CommunicationObjectAbortedException)
            {
                return true;
            }

            return false;
        }

        public void SaveBackup(long fingerprint, int instanceID, int screenHeight, int screenWidth, Stream stream)
        {
            using (CClientBackupServiceClient backupService = CServiceProxy.GetBackupServiceClient())
            {
                backupService.SaveUserBackup(fingerprint, instanceID, screenHeight, screenWidth, CServiceProxy.Instance.SessionID, stream);
            }
        }

        private void Reconnect()
        {
            CloseConnection();
            _serviceClient = GetBinaryClientServiceClient();
        }

        private void CloseConnection()
        {
            if (_serviceClient == null)
            {
                IsConnected = false;
                return;
            }

            if (_serviceClient.State != CommunicationState.Opened)
            {
                _serviceClient = null;
                IsConnected = false;
                return;
            }

            try
            {
                _serviceClient.Close();
            }
            catch { }
            finally
            {
                if (_serviceClient.State != CommunicationState.Closing && _serviceClient.State != CommunicationState.Closed)
                {
                    _serviceClient.Abort();
                    Logger.Log("CServiceProxy - CloseConnection - Service client aborted.", ELogLevel.Info);
                }
                _serviceClient = null;
                IsConnected = false;
            }
        }

        public void Shutdown()
        {
            lock (_shutdownLock)
            {
                CloseConnection();
            }
        }

        public string GetSecureDownloadUrl(string baseUrl, int addOnId, string ipAddress, DateTime date)
        {
            return SecureDownload.GetDownloadLink(baseUrl, addOnId, date) + "&p2=" + ThrottlingToken.GetHashCode(IsPremium ? 2 : 1, date);
        }

        #endregion
    }
}
