﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using IWshRuntimeLibrary;
using System.Security.Policy;
using System.IO;
using System.ComponentModel;
using System.Deployment.Application;
using System.Reflection;
using Microsoft.Win32;
using Curse.CurseClient.Common.Properties;
using System.Diagnostics;
using System.Security.AccessControl;
using System.Windows.Forms;
using System.Text.RegularExpressions;

using Curse.Extensions;

namespace Curse.CurseClient.Common
{


    public static class CDeploymentUtility
    {

        public const string LivePublicKeyToken = "eee711038731a406";
        public const string TestPublicKeyToken = "0000000000000000";

        public const string LiveDeploymentUrl = "http://clientupdate.curse.com/CurseClient.application";
        public const string TestDeploymentUrl = "http://testclient.curse.com/CurseClient.application";

        private const string AppIDFormat = "CurseClient.application, Culture=neutral, PublicKeyToken={0}, processorArchitecture=msil";

        private static Regex _desktopShortcutExpression = new Regex(@"Curse Client([ \-0-9]*?)\.appref-ms", RegexOptions.IgnoreCase | RegexOptions.Compiled);
        private static string _deploymentUrl;

        public static string AppID
        {
            get
            {
#if TEST
                return AppIDFormat.FormatWith(CDeploymentUtility.TestPublicKeyToken);                
#else
                return AppIDFormat.FormatWith(CDeploymentUtility.LivePublicKeyToken);
#endif
            }
        }

        public static string ShortcutContents
        {
            get
            {
                return DeploymentUrl + "#" + AppID;
            }
        }

        public static bool IsOutOfDate
        {

            get
            {
                if (IsNetworkDeployed)
                {
                    try
                    {
                        return ApplicationDeployment.CurrentDeployment.CheckForUpdate(false);
                    }
                    catch
                    {
                        return false;
                    }
                }
                else
                {
                    return false;
                }
            }
        }

        public static bool IsFirstRun
        {
            get
            {
                if (IsNetworkDeployed && ApplicationDeployment.CurrentDeployment.IsFirstRun)
                {
                    return true;
                }
                else
                {
                    return false;
                }
            }
        }

        public static bool IsNetworkDeployed
        {
            get
            {
#if DEBUG
                return false;
#else
                try
                {
                    return ApplicationDeployment.IsNetworkDeployed;
                }
                catch
                {
                    return false;
                }
#endif
            }
        }

        public static string DeploymentUrl
        {
            get
            {
                if (_deploymentUrl == null)
                {
                    if (CDeploymentUtility.IsNetworkDeployed)
                    {

                        _deploymentUrl = ApplicationDeployment.CurrentDeployment.UpdateLocation.ToString();
                    }
                    else
                    {
#if TEST
                        _deploymentUrl = CDeploymentUtility.TestDeploymentUrl;
#else
                        _deploymentUrl = CDeploymentUtility.LiveDeploymentUrl;
#endif
                    }
                }
                return _deploymentUrl;
            }
        }

        public static string ShortcutFilename
        {
            get
            {
                return "Curse Client.appref-ms";
            }
        }

        public static string DefaultStartMenuShortcutPath
        {
            get
            {
                return Path.Combine(StartMenuShortcutRootPath, ShortcutFilename);
            }
        }

        public static string StartMenuShortcutRootPath
        {
            get
            {
                return Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.Programs), @"Curse\");
            }
        }

        public static string DesktopShortcutPath
        {
            get
            {
                return Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.Desktop), ShortcutFilename);
            }
        }


        public static string StartupFilename
        {
            get
            {
                return "CurseClientStartup.ccip";
            }
        }

        public static string LegacyStartupFilename
        {
            get
            {
                return "CurseClientStartup.ccipm";
            }
        }



        private static void RemoveLegacyStartupFile()
        {
            try
            {
                string legacyStartupPath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.Startup), LegacyStartupFilename);
                FileInfo legacyFile = new FileInfo(legacyStartupPath);
                if (legacyFile.Exists)
                {
                    legacyFile.Delete();
                }
            }
            catch { }
        }

        public static void AddShortcutToDesktop(bool isEnabled)
        {
            if (isEnabled)
            {
                CreateDesktopShortcut();
            }
            else
            {
                DeleteDesktopShortcut();
            }
        }

        private static void DeleteDesktopShortcut()
        {
            try
            {
                string desktopPath = Environment.GetFolderPath(Environment.SpecialFolder.Desktop);
                FileInfo[] files = new DirectoryInfo(desktopPath).GetFiles();
                foreach (FileInfo file in files)
                {
                    string fileName = file.Name;
                    if (_desktopShortcutExpression.Match(fileName).Success)
                    {
                        file.Delete();
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Log("CDeploymentUtility - Unable to delete desktop shortcut. Details: {0}", ELogLevel.Error, ex.CompleteDetails());
            }
        }

        private static string FindBestStartMenuShortcut()
        {
            DirectoryInfo startMenuFolder = new DirectoryInfo(StartMenuShortcutRootPath);
            if (!startMenuFolder.Exists)
            {
                return DefaultStartMenuShortcutPath;
            }

            FileInfo[] files = startMenuFolder.GetFiles("*.appref-ms");
            if (files.Length == 0)
            {
                return DefaultStartMenuShortcutPath;
            }

            foreach (FileInfo file in files)
            {
                using (StreamReader reader = new StreamReader(file.FullName))
                {
                    try
                    {
                        string contents = reader.ReadToEnd();
                        if (contents == ShortcutContents)
                        {
                            return file.FullName;
                        }
                    }
                    catch { }
                    finally
                    {
                        reader.Close();
                    }
                }
            }

            return DefaultStartMenuShortcutPath;

        }

        private static void CreateDesktopShortcut()
        {
            try
            {
                FileInfo desktopShortcutFile = new FileInfo(DesktopShortcutPath);
                if (desktopShortcutFile.Exists)
                {
                    return;
                }

                FileInfo startMenuShortcutFile = new FileInfo(FindBestStartMenuShortcut());
                if (!startMenuShortcutFile.Exists)
                {
                    return;
                }

                startMenuShortcutFile.CopyTo(desktopShortcutFile.FullName);
            }
            catch (Exception ex)
            {
                Logger.Log("CDeploymentUtility - Unable to create desktop shortcut. Details: {0}", ELogLevel.Error, ex.CompleteDetails());
            }

        }

        public static void AddShortcutToStartupGroup(bool isEnabled, string publisherName, string productName)
        {

            try
            {
                RemoveLegacyStartupFile();
                string startupPath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.Startup), StartupFilename);

                FileInfo x = new FileInfo(startupPath);
                if (isEnabled && !x.Exists)
                {
                    x.Create();
                }
                else if (!isEnabled && x.Exists)
                {
                    x.Delete();
                }
                LegacyAddShortcutToStartupGroup(false, publisherName, productName);
            }
            catch (Exception ex)
            {
                Logger.Log("AddShortcutToStartupGroup Error! Message: {0}. Stack: {1}", ELogLevel.Error, ex.Message, ex.StackTrace);
            }

        }

        public static void LegacyAddShortcutToStartupGroup(bool isEnabled, string publisherName, string productName)
        {

            if (!CDeploymentUtility.IsNetworkDeployed)
            {
                return;
            }

            try
            {
                string startupPath = Environment.GetFolderPath(Environment.SpecialFolder.Startup);
                startupPath = Path.Combine(startupPath, productName) + ".appref-ms";
                if (isEnabled)
                {

                    if (!System.IO.File.Exists(startupPath))
                    {

                        string allProgramsPath = Environment.GetFolderPath(Environment.SpecialFolder.Programs);
                        string shortcutPath = Path.Combine(allProgramsPath, publisherName);
                        shortcutPath = Path.Combine(shortcutPath, productName) + ".appref-ms";
                        if (System.IO.File.Exists(shortcutPath))
                        {
                            System.IO.File.Copy(shortcutPath, startupPath);
                        }
                    }
                }
                else
                {
                    if (System.IO.File.Exists(startupPath))
                    {
                        System.IO.File.Delete(startupPath);
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Log("AddShortcutToStartupGroup Error! Message: {0}. Stack: {1}", ELogLevel.Error, ex.Message, ex.StackTrace);
            }

        }

        public static string GetPublicKeyTokenFromManifest()
        {
            if (AppDomain.CurrentDomain.ActivationContext == null)
            {
                return null;
            }

            ApplicationSecurityInfo asi = new ApplicationSecurityInfo(AppDomain.CurrentDomain.ActivationContext);
            byte[] pk = asi.ApplicationId.PublicKeyToken;
            StringBuilder pkt = new StringBuilder();
            for (int i = 0; i < pk.GetLength(0); i++)
            {
                pkt.Append(String.Format("{0:x2}", pk[i]));
            }
            return "{" + pkt.ToString() + "}";
        }

        /// <summary>
        /// Gets the public key token for the current ClickOnce app.
        /// </summary>
        /// <returns></returns>
        public static string GetAssemblyToken()
        {

            if (AppDomain.CurrentDomain.ActivationContext != null)
            {
                return GetPublicKeyTokenFromManifest();
            }
            else
            {
                return Settings.Default.PublicKeyToken;
            }
        }

        public static bool SafelyResetAppStore()
        {
            try
            {
                ResetAppStore();
                return true;
            }
            catch
            {
                return false;
            }
        }

        public static void ResetAppStore()
        {
            string appPath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData), "Apps\\2.0");
            if (Directory.Exists(appPath))
            {
                Directory.Delete(appPath, true);
            }
        }
    }

}
