﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using ServiceUnitTests.TestFixtures;
using ServiceUnitTests.Attributes;
using ServiceUnitTests.AuthService;
using NUnit.Framework;
using System.Data;
using System.Data.SqlClient;
using Curse.Extensions;

namespace ServiceUnitTests.ObjectValidation.AuthService {
    [AuthObjectValidation]
    class UserObjectValidation : AuthTestFixture {
        static User _user;
        static UserProfile _profile;
        static DataTable _profileData;
        static List<UserEntitlement> _entitlements;
        static List<UserSubscripion> _subscriptions;

        [SetUp]
        public new void Init() {
            base.Init();

            #region Get the user profile
            _user = AuthClient.v2GetUserProfile(CurseSiteID, UserID);
            Assert.IsNotNull(_user, "_profile is null");
            #endregion

            #region Load the user data from the DB
            using (var da = new SqlDataAdapter("spUserFromID", AuthConnectionString)) {
                da.SelectCommand.CommandType = CommandType.StoredProcedure;
                da.SelectCommand.Parameters.Add("numUserID", SqlDbType.Int).Value = UserID;

                _profileData = new DataTable();
                var count = da.Fill(_profileData);
                Assert.AreNotEqual(0, count);
            }
            #endregion

            #region Load the User Profile from the DB
            using (var conn = new SqlConnection(AuthConnectionString)) {
                var cmd = new SqlCommand("SELECT _firstName, _lastName, _birthDate, _gender, _defaultEmail, _country, _region, _city, _wantsmail, _registeredVia, _registeredOn FROM userprofiles WHERE _uid = @uid", conn);
                cmd.Parameters.Add("uid", SqlDbType.Int).Value = UserID;

                conn.Open();
                using (var reader = cmd.ExecuteReader()) {
                    if (!reader.Read()) {
                        Assert.Inconclusive("Failed to pull the user profile.");
                    }

                    _profile = new UserProfile {
                        ID = UserID,
                        FirstName = reader.GetString(reader.GetOrdinal("_firstName")),
                        Country = reader.GetString(reader.GetOrdinal("_country")),
                        WantsEmail = reader.GetBoolean(reader.GetOrdinal("_wantsmail")),
                        RegisteredVia = reader.GetInt32(reader.GetOrdinal("_registeredVia")),
                        RegisteredOn = reader.GetDateTime(reader.GetOrdinal("_registeredOn")),
                        PrimaryEmail = reader.GetInt32(reader.GetOrdinal("_defaultEmail"))
                    };

                    if (reader["_lastName"] == DBNull.Value) {
                        _profile.LastName = null;
                    }
                    else _profile.LastName = reader.GetNullableValue<string>("_lastName");

                    if (reader["_birthDate"] == DBNull.Value) {
                        _profile.BirthDate = null;
                    }
                    else _profile.BirthDate = reader.GetNullableValue<DateTime>("_birthDate");

                    if (reader["_gender"] == DBNull.Value) {
                        _profile.Gender = null;
                    }
                    else _profile.Gender = reader.GetNullableValue<Byte>("_gender");

                    if (reader["_region"] == DBNull.Value) {
                        _profile.Region = null;
                    }
                    else _profile.Region = reader.GetNullableValue<string>("_region");

                    if (reader["_city"] == DBNull.Value) {
                        _profile.City = null;
                    }
                    else _profile.City = reader.GetNullableValue<string>("_city");
                }
            }
            Assert.IsNotNull(_profile);
            #endregion

            #region Load the user subscriptions
            _subscriptions = new List<UserSubscripion>();
            using (var conn = new SqlConnection(AuthConnectionString)) {
                var cmd = new SqlCommand("spSubscriptionsFromUserID", conn);
                cmd.Parameters.Add("numUserID", SqlDbType.Int).Value = UserID;
                cmd.CommandType = CommandType.StoredProcedure;

                conn.Open();
                using (var reader = cmd.ExecuteReader()) {
                    while (reader.Read()) {
                        _subscriptions.Add(new UserSubscripion {
                            Type = (ESubscriptionType)reader.GetInt16(1),
                            Level = reader.GetByte(2),
                            SignupDate = reader.GetDateTime(3),
                            ExpirationDate = reader.GetDateTime(4),
                            Cancelled = !reader.GetBoolean(5)
                        });
                    }
                }
            }
            #endregion

            #region Load the entitlements from the DB
            _entitlements = new List<UserEntitlement>();
            using (var conn = new SqlConnection(BillingConnectionString)) {
                var cmd = new SqlCommand("SELECT * FROM AccountEntitlement WHERE AccountID = @AccountID", conn);
                cmd.Parameters.Add("AccountID", SqlDbType.VarChar).Value = "U-" + UserID;

                conn.Open();
                using (var reader = cmd.ExecuteReader()) {
                    while (reader.Read()) {
                        var entitlement = new UserEntitlement {
                            ID = reader.GetString(2),
                            Expires = reader.GetDateTime(4)
                        };

                        switch (entitlement.ID) {
                            case "CP":
                                entitlement.Type = ESubscriptionType.Premium;
                                break;
                            case "GH":
                                entitlement.Type = ESubscriptionType.GuildHosting;
                                break;
                            case "VH":
                                entitlement.Type = ESubscriptionType.VentriloHosting;
                                break;
                        }

                        _entitlements.Add(entitlement);
                    }
                }
            }

            var activeLegacy = _subscriptions.FirstOrDefault(p => p.Type == ESubscriptionType.Premium && p.Active);
            if (activeLegacy != null) {
                var hasActivePremium = _entitlements.Count(p => p.Type == ESubscriptionType.Premium && p.Expires > DateTime.UtcNow) > 0;
                if (!hasActivePremium) {
                    _entitlements.Add(new UserEntitlement {
                        Expires = activeLegacy.ExpirationDate,
                        ID = "CP",
                        Type = ESubscriptionType.Premium
                    });
                }
            }
            #endregion
        }

        [Test]
        public void Entitlements() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(_entitlements.Count, _user.Entitlements.Length);
            for (int i = 0; i < _entitlements.Count; i++) {
                Assert.AreEqual(_entitlements[i], _user.Entitlements[i]);
            }
        }

        [Test]
        public void HasRenameCredit() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            using (var conn = new SqlConnection(AuthConnectionString)) {
                var cmd = new SqlCommand("SELECT _uid FROM userrenames WHERE _uid = @userid", conn);
                cmd.Parameters.Add("userid", SqlDbType.Int).Value = UserID;

                conn.Open();
                var hasRenameCredit = (cmd.ExecuteScalar() != null);
                Assert.AreEqual(hasRenameCredit, _user.HasRenameCredit);
            }
        }

        [Test]
        public void ID() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(_profileData.Rows[0]["_uid"], _user.ID);
        }

        [Test]
        public void Legacy() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            var userLegacy = new List<UserLegacy>();
            using (var conn = new SqlConnection(AuthConnectionString)) {
                var cmd = new SqlCommand("spGetAllLegacyUsers", conn);
                cmd.Parameters.Add("numUserID", SqlDbType.Int).Value = UserID;
                cmd.CommandType = CommandType.StoredProcedure;

                conn.Open();
                using (var reader = cmd.ExecuteReader()) {
                    while (reader.Read()) {
                        userLegacy.Add(new UserLegacy {
                            SiteID = reader.GetInt32(0),
                            Status = (ELegacyUserStatus)reader.GetByte(1)
                        });
                    }
                }
            }

            Assert.AreEqual(userLegacy.Count, _user.Legacy.Length);
            for (int i = 0; i < userLegacy.Count; i++) {
                Assert.AreEqual(userLegacy[i], _user.Legacy[i]);
            }
        }

        [Test]
        public void Level() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(_profileData.Rows[0]["_level"], _user.Level);
        }

        [Test]
        public void Name() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(_profileData.Rows[0]["_username"], _user.Name);
        }

        [Test]
        public void ProfileFirstName() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(_profile.FirstName, _user.Profile.FirstName);
        }

        [Test]
        public void ProfileLastName() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(_profile.LastName, _user.Profile.LastName);
        }

        [Test]
        public void ProfileBirthDate() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(_profile.BirthDate, _user.Profile.BirthDate);
        }

        [Test]
        public void ProfileGender() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(_profile.Gender, _user.Profile.Gender);
        }

        [Test]
        public void ProfileCountry() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(_profile.Country, _user.Profile.Country);
        }

        [Test]
        public void ProfileRegion() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(_profile.Region, _user.Profile.Region);
        }

        [Test]
        public void ProfileCity() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(_profile.City, _user.Profile.City);
        }

        [Test]
        public void ProfileRegisteredVia() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(_profile.RegisteredVia, _user.Profile.RegisteredVia);
        }

        [Test]
        public void ProfileRegisteredOn() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(_profile.RegisteredOn, _user.Profile.RegisteredOn);
        }

        [Test]
        public void ProfilePrimaryEmail() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(_profile.PrimaryEmail, _user.Profile.PrimaryEmail);
        }

        [Test]
        public void ProfileWantsMail() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(_profile.WantsEmail, _user.Profile.WantsEmail);
        }

        [Test]
        public void SessionIsSecure() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(false, _user.SessionIsSecure);
        }

        [Test]
        public void Status() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(EUserStatus.Normal, _user.Status);
        }

        [Test]
        public void Subscriptions() {
            if (_user == null) {
                Assert.Inconclusive("Unable to pull a profile to check against.");
            }

            Assert.AreEqual(_subscriptions.Count, _user.Subscriptions.Length);
            for (int i = 0; i < _subscriptions.Count; i++) {
                Assert.AreEqual(_subscriptions[i], _user.Subscriptions[i]);
            }
        }
    }
}
