﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using ServiceUnitTests.TestFixtures;
using NUnit.Framework;
using System.Data.SqlClient;
using System.Text.RegularExpressions;
using System.Data;
using System.Configuration;
using Curse.CurseClient.Common.ClientService;
using ServiceUnitTests.Attributes;
using Curse.AddOns;

namespace ServiceUnitTests.ObjectValidation.ClientService {
    [ClientObjectValidation]
    class AddonObjectValidation : ClientServiceTestFixture {
        int _addonId;
        CAddOn _addon;
        DataTable _addonDt;
        CAddOnFile _defaultFile;

        [SetUp]
        public new void Init() {
            base.Init();

            // pull the addon in question from the appconfig
            _addonId = Int32.Parse(ConfigurationManager.AppSettings["addonId"]);

            _addon = Proxy.TryCall<CAddOn>(delegate(CClientServiceClient svc) {
                return svc.GetAddOn(_addonId);
            });

            if (_addon == null) {
                Assert.IsNotNull(_addon);
                return;
            }

            #region SqlQuery
            var query = "select " +
                "Project.ID as addon_id, " +
                "Project.DateSynced as addon_last_updated, " +
                "Project.DateCreated as addon_date_created, " +
                "Project.PrimaryCategoryId as addon_primary_category_id, " +
                "IsNull(Project.DonationHtml, '') as addon_donation_link, " +
                "IsNull(Project.DateLatestReleaseFile, Project.DateCreated) as addon_date_released, " +
                "stuff(((select ',' + Language.FullCultureCode from Language where Language.ID in (select ProjectLanguageMap.LanguageID from ProjectLanguageMap where ProjectID = Project.ID) for xml path(''))), 1, 1, '') as addon_languages, " +
                "Project.ExternalUrl as addon_external_url, " +
                "Project.Name as addon_name, " +
		        "Project.CategorySectionID as addon_section_id, " +
                "ProjectCategorySection.Slug as addon_section, " +
                "Project.Slug as addon_download_name, " +
                "ProjectDescription.[Text] as addon_description, " +
                "Project.Summary as addon_summary, " +
                "[User].Username as addon_primary_author, " +
                "(select COUNT(Comment.EntityID) from Comment where EntityID = Project.ID) as addon_num_comments, " +
                "Game.Name as addon_game, " +
                "lower(Game.Slug) as addon_game_slug, " +
                "'www.curse.com' as addon_portal, " +
                "Project.GameId as addon_game_id, " +
                "IsNull((select top 1 Id from ProjectFile where ProjectFile.ProjectId = Project.Id and ReleaseType = 1 order by DateCreated desc),0) as addon_default_file, " +
                "Project.LikeCount as addon_average_rating, " +
                "Project.InstallCount as addon_install_count, " +
                "Project.DownloadCount as addon_num_downloads, " +
                "case when Project.Status != 1 then 0 when Project.Stage not in (2,4,5,7) then 0 else 1 end as addon_is_available, " +
                "Project.Status as addon_status, " +
                "Project.Stage as addon_stage " +
            "from Project with(nolock) " +
                "inner join Game with(nolock) " +
                "on Game.ID = project.GameId " +
                "inner join RoleAssignment with(nolock) " +
                "on RoleAssignment.EntityID = Project.ID " +
                "inner join [User] with(nolock) " +
                "on [User].ID = RoleAssignment.UserSurrogateID " +
                "inner join ProjectCategorySection with(nolock) " +
                "on ProjectCategorySection.Id = Project.CategorySectionId " +
                "inner join ProjectDescription with(nolock) " +
                "on ProjectDescription.ProjectID = Project.ID " +
                "inner join GameSectionPackageMap on GameSectionPackageMap.GameID = Project.GameID and GameSectionPackageMap.ProjectCategorySectionID = Project.CategorySectionId " +
            "where " +
                "exists(select top 1 1 from ProjectFile where ProjectFile.ProjectId = Project.Id) " +
                "and RoleAssignment.RoleID = 2 " +
                "and Project.GameId in (1, 38, 203, 335, 424, 449) " +
                "and Project.ID = @ID";
            #endregion

            // get all the addon data
            _addonDt = new DataTable();
            using (var adapter = new SqlDataAdapter(query, RadonConnectionString)) {
                adapter.SelectCommand.Parameters.Add("ID", SqlDbType.Int).Value = _addonId;

                int count = adapter.Fill(_addonDt);
                Assert.AreNotEqual(0, count);
            }

            // Get the default file
            _defaultFile = GetDefaultFile();
            Assert.IsNotNull(_defaultFile);
        }

        [TearDown] // I don't want this to happen
        public new void ResetConnection() { }

        #region Tests
        [Test]
        public void Authors() {
            using (var conn = new SqlConnection(RadonConnectionString)) {
                using (var reader = GetAuthorReader(conn)) {
                    while (reader.Read()) {
                        string authorName = reader.GetString(reader.GetOrdinal("author_username"));

                        var author = _addon.Authors.FirstOrDefault(p => p.Name == authorName);
                        if (author == null) {
                            Assert.IsNotNull(author);
                        }
                    }
                    Assert.AreEqual(true, reader.HasRows);
                }
            }
        }

        [Test]
        public void AuthorList() {
            using (var conn = new SqlConnection(RadonConnectionString)) {
                using (var reader = GetAuthorReader(conn)) {
                    while (reader.Read()) {
                        string authorName = reader.GetString(reader.GetOrdinal("author_username"));

                        var contains = _addon.AuthorList.Contains(authorName);
                        Assert.AreEqual(true, contains);
                    }
                    Assert.AreEqual(true, reader.HasRows);
                }
            }
        }

        [Test]
        public void Categories() {
            using (var conn = new SqlConnection(RadonConnectionString)) {
                using (var reader = GetCategoryReader(conn)) {
                    while (reader.Read()) {
                        string categoryName = reader.GetString(reader.GetOrdinal("addon_category_name"));

                        var category = _addon.Categories.FirstOrDefault(p => p.Name == categoryName);
                        if (category == null) {
                            Assert.IsNotNull(category);
                        }
                    }
                    Assert.AreEqual(true, reader.HasRows);
                }
            }
        }

        [Test]
        public void CategoryList() {
            using (var conn = new SqlConnection(RadonConnectionString)) {
                using (var reader = GetCategoryReader(conn)) {
                    while (reader.Read()) {
                        string categoryName = reader.GetString(reader.GetOrdinal("addon_category_name"));

                        var contains = _addon.CategoryList.Contains(categoryName);
                        Assert.AreEqual(true, contains);
                    }
                    Assert.AreEqual(true, reader.HasRows);
                }
            }
        }

        [Test]
        public void ClientUrl() {
            var portalName = _addonDt.Rows[0]["addon_portal"];
            var sectionName = _addonDt.Rows[0]["addon_section"];
            var gameSlug = _addonDt.Rows[0]["addon_game_slug"];
            var downloadName = _addonDt.Rows[0]["addon_download_name"];
            var baseDownloadUrl = string.Format(@"http://{0}/{1}/{2}/{3}", portalName, sectionName, gameSlug, downloadName);

            Assert.AreEqual(baseDownloadUrl, _addon.ClientUrl);
        }

        [Test]
        public void CommentCount() {
            var commentCount = _addonDt.Rows[0]["addon_num_comments"];
            Assert.AreEqual(commentCount, _addon.CommentCount);
        }

        [Test(Description = "This field is not-serialized and should be null or default")]
        public void DateCreated() {
            Assert.AreEqual(default(DateTime), _addon.DateCreated);
        }

        [Test(Description = "This field is not-serialized and should be null or default")]
        public void DateModified() {
            Assert.AreEqual(default(DateTime), _addon.DateModified);
        }

        [Test(Description = "This field is not-serialized and should be null or default")]
        public void DateReleased() {
            Assert.AreEqual(default(DateTime), _addon.DateReleased);
        }

        [Test]
        public void DefaultFile() {
            Assert.IsNotNull(_addon.DefaultFile);
        }

        [Test]
        public void DefaultFileDate() {
            Assert.AreEqual(_defaultFile.FileDate, _addon.DefaultFileDate);
        }

        [Test]
        public void DefaultFileId() {
            Assert.AreEqual(_defaultFile.Id, _addon.DefaultFile.Id);
        }

        [Test]
        public void DefaultFileName() {
            Assert.AreEqual(_defaultFile.FileName, _addon.DefaultFileName);
        }

        [Test]
        public void DefaultFileType() {
            Assert.AreEqual(_defaultFile.ReleaseType, _addon.DefaultFileType);
        }

        [Test]
        public void DonationUrl() {
            var donationLink = (string)_addonDt.Rows[0]["addon_donation_link"];
            if (donationLink == string.Empty) {
                Assert.IsNull(_addon.DonationUrl);
            }
            else {
                var sHrefGrabber = new Regex("href.*?\"(?<href>.*?)\"", RegexOptions.Compiled);
                var match = sHrefGrabber.Match(donationLink);
                if (!match.Success) {
                    Assert.Inconclusive("Donation Url is not in the right format.");
                }

                donationLink = match.Groups["href"].Value;
                Assert.AreEqual(donationLink, _addon.DonationUrl);
            }
        }

        [Test]
        public void DownloadCount() {
            var downloadCount = _addonDt.Rows[0]["addon_num_downloads"];
            Assert.AreEqual(downloadCount, _addon.DownloadCount);
        }

        [Test]
        public void ExternalUrl() {
            var externalUrl = (string)_addonDt.Rows[0]["addon_external_url"];
            if (externalUrl == string.Empty) {
                Assert.IsNull(_addon.ExternalUrl);
            }
            else Assert.AreEqual(externalUrl, _addon.ExternalUrl);
        }

        [Test(Description = "This field is not-serialized and should be null or default")]
        public void Files() {
            Assert.IsNull(_addon.Files);
        }

        [Test(Description = "This field is not-serialized and should be null or default")]
        public void FullDescription() {
            Assert.IsNull(_addon.FullDescription);
        }

        [Test]
        public void GameId() {
            var gameId = _addonDt.Rows[0]["addon_game_id"];
            Assert.AreEqual(gameId, _addon.GameId);
        }

        [Test(Description = "This field is not-serialized and should be null or default")]
        public void GameName() {
            Assert.IsNull(_addon.GameName);
        }

        [Test]
        public void IconId() {
            Assert.AreEqual(1, _addon.IconId);
        }

        [Test]
        public void Id() {
            var addonId = _addonDt.Rows[0]["addon_id"];
            Assert.AreEqual(addonId, _addon.Id);
        }

        [Test]
        public void IncludeInFingerprinting() {
            var status = (EProjectStatus)(Byte)_addonDt.Rows[0]["addon_status"];
            Assert.AreEqual((status != EProjectStatus.Deleted), _addon.IncludeInFingerprinting);
        }

        [Test]
        public void InstallCount() {
            var installCount = _addonDt.Rows[0]["addon_install_count"];
            Assert.AreEqual(installCount, _addon.InstallCount);
        }

        [Test(Description = "This field is not-serialized and should be null or default")]
        public void IsAvailable() {
            Assert.AreEqual(default(bool), _addon.IsAvailable);
        }

        [Test]
        public void LatestFiles() {
            var latestFiles = GetLatestFiles();
            foreach (var latestFile in latestFiles) {
                var file = _addon.LatestFiles.FirstOrDefault(p => p.Id == latestFile.Id);
                Assert.IsNotNull(file);
            }
        }

        [Test]
        public void Likes() {
            var likes = _addonDt.Rows[0]["addon_average_rating"];
            Assert.AreEqual(likes, _addon.Likes);
        }

        [Test]
        public void Name() {
            var name = _addonDt.Rows[0]["addon_name"];
            Assert.AreEqual(name, _addon.Name);
        }

        [Test(Description = "This field is not-serialized and should be null or default")]
        public void PortalName() {
            Assert.IsNull(_addon.PortalName);
        }

        [Test]
        public void PrimaryAuthorName() {
            var primaryAuthorName = _addonDt.Rows[0]["addon_primary_author"];
            Assert.AreEqual(primaryAuthorName, _addon.PrimaryAuthorName);
        }

        [Test]
        public void PrimaryCategoryId() {
            var primaryCategoryId = _addonDt.Rows[0]["addon_primary_category_id"];
            Assert.AreEqual(primaryCategoryId, _addon.PrimaryCategoryId);
        }

        [Test]
        public void Rating() {
            var rating = 0; //_addonDt.Rows[0][""];
            Assert.AreEqual(rating, _addon.Rating);
        }

        [Test(Description = "This field is not-serialized and should be null or default")]
        public void SectionName() {
            Assert.IsNull(_addon.SectionName);
        }

        [Test(Description = "This field is not-serialized and should be null or default")]
        public void Slug() {
            Assert.IsNull(_addon.Slug);
        }

        [Test]
        public void Stage() {
            var stage = (EProjectStage)(Byte)_addonDt.Rows[0]["addon_stage"];
            Assert.AreEqual(stage, _addon.Stage);
        }

        [Test]
        public void Status() {
            var status = (EProjectStatus)(Byte)_addonDt.Rows[0]["addon_status"];
            Assert.AreEqual(status, _addon.Status);
        }

        [Test]
        public void Summary() {
            var summary = _addonDt.Rows[0]["addon_summary"];
            Assert.AreEqual(summary, _addon.Summary);
        }

        [Test]
        public void WebSiteURL() {
            var portalName = _addonDt.Rows[0]["addon_portal"];
            var sectionName = _addonDt.Rows[0]["addon_section"];
            var gameSlug = _addonDt.Rows[0]["addon_game_slug"];
            var downloadName = _addonDt.Rows[0]["addon_download_name"];
            var baseDownloadUrl = string.Format(@"http://{0}/{1}/{2}/{3}", portalName, sectionName, gameSlug, downloadName);

            Assert.AreEqual(baseDownloadUrl, _addon.WebSiteURL);
        }

        [Test]
        public void PackageType()
        {
            using (var conn = new SqlConnection(RadonConnectionString))
            {
                var cmd = new SqlCommand("SELECT * FROM GameSectionPackageMap WHERE ProjectCategorySectionID = @ID", conn);
                cmd.Parameters.Add("ID", SqlDbType.Int).Value = _addonDt.Rows[0]["addon_section_id"];

                conn.Open();
                using (var reader = cmd.ExecuteReader())
                {
                    if (reader.Read())
                    {
                        var gameId = reader.GetInt32(reader.GetOrdinal("GameID"));
                        Assert.AreEqual(gameId, _addon.GameId);

                        var sectionId = reader.GetInt32(reader.GetOrdinal("ProjectCategorySectionID"));
                        Assert.AreEqual(sectionId, _addon.CategorySection.ID);
                    }
                    else Assert.Fail("Failed to pull category Section");
                }
            }
        }
        #endregion

        #region Random Methods
        private SqlDataReader GetReader(string query, SqlConnection conn) {
            var cmd = new SqlCommand(query, conn);
            cmd.Parameters.Add("ID", SqlDbType.Int).Value = ConfigurationManager.AppSettings["addonID"];

            if (conn.State != ConnectionState.Open)
                conn.Open();

            return cmd.ExecuteReader();
        }
        private SqlDataReader GetAuthorReader(SqlConnection conn) {
            return GetReader("select distinct Project.ID as addon_id, [User].Username as author_username from Project with(nolock) inner join RoleAssignment with(nolock) on RoleAssignment.EntityID = Project.ID inner join [User] with(nolock) on [User].ID = RoleAssignment.UserSurrogateID where Project.ID = @ID and RoleAssignment.RoleID = 3", conn);
        }
        private SqlDataReader GetCategoryReader(SqlConnection conn) {
            var query = "select Project.ID as addon_id, " +
                    "ProjectCategory.Name as addon_category_name, " +
                    "ProjectCategory.ID as addon_category_id, " +
                    "ProjectCategory.Slug as addon_category_slug, " +
                    "ProjectCategorySection.Slug as addon_section, " +
                    "lower(Game.Slug) as addon_game_slug " +
                "from ProjectCategory with(nolock) " +
                    "inner join ProjectCategoryMap with(nolock) " +
                    "on ProjectCategoryMap.ProjectCategoryId = ProjectCategory.ID " +
                    "inner join Project with(nolock) " +
                    "on Project.ID = ProjectCategoryMap.ProjectId " +
                    "inner join ProjectCategorySection with(nolock) " +
                    "on ProjectCategorySection.ID = ProjectCategory.SectionID " +
                    "inner join Game with(nolock) " +
                    "on Game.ID = Project.GameID " +
                "where Project.ID = @ID " +
            "union " +
                "select " +
                    "Project.ID as addon_id, " +
                    "ProjectCategory.Name as addon_category_name, " +
                    "ProjectCategory.ID as addon_category_id, " +
                    "ProjectCategory.Slug as addon_category_slug, " +
                    "ProjectCategorySection.Slug as addon_section, " +
                    "lower(Game.Slug) as addon_game_slug " +
                "from Project with(nolock) " +
                    "inner join ProjectCategory with(nolock) " +
                    "on ProjectCategory.ID = Project.PrimaryCategoryID " +
                    "inner join ProjectCategorySection with(nolock) " +
                    "on ProjectCategorySection.ID = ProjectCategory.SectionID " +
                    "inner join Game with(nolock) " +
                    "on Game.ID = Project.GameID " +
                "where Project.ID = @ID";


            return GetReader(query, conn);
        }
        private SqlDataReader GetLatestFileReader(SqlConnection conn) {
            var query = "SELECT " +
                "ProjectFile.ProjectID  as addon_id, " +
                "ProjectFile.ID as [file_id], " +
                "ProjectFile.Name as [file_name], " +
                "ProjectFile.Filename as file_filename, " +
                "ProjectFile.DateCreated as file_date, " +
                "ProjectFile.ReleaseType as file_type, " +
                "'' as file_change_log, " +
                "ProjectFile.IsAlternate as file_is_alternate, " +
                "ISNULL(ProjectFile.AlternateFileId, 0) as file_alternate_file_id, " +
                "ProjectFile.Status as file_status " +
                "FROM ProjectFile with(nolock) " +
                "inner join Project with(nolock) on Project.ID = ProjectFile.ProjectID " +
                "WHERE Project.ID = @ID " +
                "AND ProjectFile.ID in (SELECT distinct ProjectFileModuleFingerprint.FileId as [file_id] FROM ProjectFileModuleFingerprint WITH (nolock) inner join ProjectFile with(nolock) on ProjectFile.ID = ProjectFileModuleFingerprint.FileId inner join Project with(nolock) on Project.ID = ProjectFile.ProjectId WHERE Project.ID = @ID AND ProjectFileModuleFingerprint.Module IS NOT NULL) " +
                "AND (ProjectFile.DateCreated = (SELECT MAX(pf.DateCreated) FROM ProjectFile pf WHERE pf.ProjectID = Project.ID AND pf.ReleaseType = 1) " +
                "OR ProjectFile.DateCreated = (SELECT MAX(pf.DateCreated) FROM ProjectFile pf WHERE pf.ProjectID = Project.ID AND pf.ReleaseType = 2) " +
                "OR ProjectFile.DateCreated = (SELECT MAX(pf.DateCreated) FROM ProjectFile pf WHERE pf.ProjectID = Project.ID AND pf.ReleaseType = 3)) ";

            if (GetAddonPackageType() == EPackageType.Folder) {
                query += "AND ProjectFile.Status = 1";
            }
            else {
                query += "AND (ProjectFile.Status = 13 AND IsAlternate = 1)";
            }

            return GetReader(query, conn);
        }

        private Curse.AddOns.EPackageType GetAddonPackageType() {
            using (var conn = new SqlConnection(RadonConnectionString)) {
                using (var cmd = new SqlCommand("SELECT AddonPackageType FROM CurseClientGame WHERE GameId = (SELECT GameId FROM Project WHERE ID = @ID)", conn)) {
                    cmd.Parameters.Add("ID", SqlDbType.Int).Value = ConfigurationManager.AppSettings["addonID"];

                    conn.Open();
                    var packageType = cmd.ExecuteScalar();
                    if (packageType != DBNull.Value) {
                        return (Curse.AddOns.EPackageType)packageType;
                    }
                }
            }

            return Curse.AddOns.EPackageType.Folder;
        }

        private List<CAddOnFile> GetLatestFiles() {
            var files = new List<CAddOnFile>();

            using (var conn = new SqlConnection(RadonConnectionString)) {
                using (var pReader = GetLatestFileReader(conn)) {
                    while (pReader.Read()) {
                        var file = new CAddOnFile();

                        file.Id = pReader.GetInt32(pReader.GetOrdinal("file_id"));
                        file.FileName = pReader.GetString(pReader.GetOrdinal("file_name"));
                        file.FileDate = pReader.GetDateTime(pReader.GetOrdinal("file_date"));
                        file.ReleaseType = (EFileType)pReader.GetByte(pReader.GetOrdinal("file_type"));
                        file.IsAlternate = pReader.GetBoolean(pReader.GetOrdinal("file_is_alternate"));
                        file.FileStatus = (EFileStatus)pReader.GetByte(pReader.GetOrdinal("file_status"));
                        file.IsAvailable = ((file.FileStatus == EFileStatus.Normal) || (file.FileStatus == EFileStatus.ClientOnly && file.IsAlternate == true));

                        file.AlternateFileId = pReader.GetInt32(pReader.GetOrdinal("file_alternate_file_id"));
                        //file.DownloadURL = sMediaFileUrl.FormatWith(file.Id.ToModPath(), pReader["file_filename"]);
                        file.Changelog = pReader.GetString(pReader.GetOrdinal("file_change_log"));

                        files.Add(file);
                    } Assert.AreEqual(true, pReader.HasRows);
                }
            }

            return files;
        }
        private CAddOnFile GetDefaultFile() {
            var latestFiles = GetLatestFiles();

            // Release
            if (latestFiles.Exists(p => p.ReleaseType == EFileType.Release && p.IsAlternate == false && p.IsAvailable)) {
                return latestFiles.First(p => p.ReleaseType == EFileType.Release && p.IsAlternate == false);
            }
            else if (latestFiles.Exists(p => p.ReleaseType == EFileType.Release && p.IsAlternate == true && p.IsAvailable)) {
                return latestFiles.First(p => p.ReleaseType == EFileType.Release && p.IsAlternate == true);
            }

            // Beta
            if (latestFiles.Exists(p => p.ReleaseType == EFileType.Beta && p.IsAlternate == false && p.IsAvailable)) {
                return latestFiles.First(p => p.ReleaseType == EFileType.Beta && p.IsAlternate == false && p.IsAvailable);
            }
            else if (latestFiles.Exists(p => p.ReleaseType == EFileType.Beta && p.IsAlternate == true && p.IsAvailable)) {
                return latestFiles.First(p => p.ReleaseType == EFileType.Beta && p.IsAlternate == true);
            }

            // Alpha
            if (latestFiles.Exists(p => p.ReleaseType == EFileType.Alpha && p.IsAlternate == false && p.IsAvailable)) {
                return latestFiles.First(p => p.ReleaseType == EFileType.Alpha && p.IsAlternate == false);
            }
            else if (latestFiles.Exists(p => p.ReleaseType == EFileType.Alpha && p.IsAlternate == true && p.IsAvailable)) {
                return latestFiles.First(p => p.ReleaseType == EFileType.Alpha && p.IsAlternate == true);
            }

            return null;
        }
        #endregion
    }
}
