﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using ServiceUnitTests.TestFixtures;
using ServiceUnitTests.AuthService;
using NUnit.Framework;
using System.Data.SqlClient;
using System.Data;
using ServiceUnitTests.Common;
using ServiceUnitTests.Attributes;

namespace ServiceUnitTests.UnitTests.AuthService {
    [AuthUnitTestsAttribute]
    class SoapWebMethods : AuthTestFixture {
        #region Variables
        User _user;
        User _legacyUser;

        User User {
            get {
                if (_user == null) {
                    _user = AuthClient.v2GetUserProfile(CurseSiteID, Utilities.CreateUser(CurseSiteID, RandomUsername, CurseEncPassword, FirstName, LastName, BirthDate, Gender, RandomEmailAddress));
                }

                return _user;
            }
        }
        User LegacyUser {
            get {
                if (_legacyUser == null) {
                    _legacyUser = AuthClient.v2GetUserProfile(CurseSiteID, Utilities.CreateLegacyUser(CurseSiteID, LegacyID, RandomUsername, CurseEncPassword, FirstName, LastName, BirthDate, Gender, RandomEmailAddress));
                }

                return _legacyUser;
            }
        }
        #endregion

        [TestFixtureTearDown]
        public void TearDown() {
            if (_user != null) {
                Utilities.DeleteUser(CurseSiteID, _user.ID);
            }

            if (_legacyUser != null) {
                Utilities.DeleteUser(CurseSiteID, _legacyUser.ID);
            }
        }

        #region v2SetUserprofileInfo
        [Test]
        public void v2SetUserProfileInfo() {
            var result = AuthClient.v2SetUserProfileInfo(
                CurseSiteID,
                User.ID,
                FirstName,
                LastName,
                BirthDate,
                Gender,
                Country,
                Region,
                City,
                User.Profile.PrimaryEmail,
                NewsLetter
            );

            Assert.AreEqual(ESetUserProfileResult.Success, result);
        }

        [Test]
        public void v2SetUserProfileInfoInvalidEmail() {
            var result = AuthClient.v2SetUserProfileInfo(
                CurseSiteID,
                User.ID,
                FirstName,
                LastName,
                BirthDate,
                Gender,
                Country,
                Region,
                City,
                -1,
                NewsLetter
            );

            Assert.AreEqual(ESetUserProfileResult.EmailNotFound, result);
        }

        [Test]
        public void v2SetUserProfileInfoInvalidUser() {
            var result = AuthClient.v2SetUserProfileInfo(
                CurseSiteID,
                -1,
                FirstName,
                LastName,
                BirthDate,
                Gender,
                Country,
                Region,
                City,
                User.Profile.PrimaryEmail,
                NewsLetter
            );

            Assert.AreEqual(ESetUserProfileResult.UserNotFound, result);
        }

        [Test]
        public void v2SetUserProfileInfoInvalidFirstName() {
            var result = AuthClient.v2SetUserProfileInfo(
                CurseSiteID,
                User.ID,
                InvalidName,
                LastName,
                BirthDate,
                Gender,
                Country,
                Region,
                City,
                User.Profile.PrimaryEmail,
                NewsLetter
            );

            Assert.AreEqual(ESetUserProfileResult.InvalidProfileData, result);
        }

        [Test]
        public void v2SetUserProfileInfoInvalidLastName() {
            var result = AuthClient.v2SetUserProfileInfo(
                CurseSiteID,
                User.ID,
                FirstName,
                InvalidName,
                BirthDate,
                Gender,
                Country,
                Region,
                City,
                User.Profile.PrimaryEmail,
                NewsLetter
            );

            Assert.AreEqual(ESetUserProfileResult.InvalidProfileData, result);
        }

        [Test]
        public void v2SetUserProfileInfoInvalidBirthDate() {
            var result = AuthClient.v2SetUserProfileInfo(
                CurseSiteID,
                User.ID,
                FirstName,
                LastName,
                InvalidBirthDay,
                Gender,
                Country,
                Region,
                City,
                User.Profile.PrimaryEmail,
                NewsLetter
            );

            Assert.AreEqual(ESetUserProfileResult.InvalidProfileData, result);
        }

        [Test]
        public void v2SetUserProfileInfoInvalidGender() {
            var result = AuthClient.v2SetUserProfileInfo(
                CurseSiteID,
                User.ID,
                FirstName,
                LastName,
                BirthDate,
                InvalidGender,
                Country,
                Region,
                City,
                User.Profile.PrimaryEmail,
                NewsLetter
            );

            Assert.AreEqual(ESetUserProfileResult.InvalidProfileData, result);
        }
        #endregion

        #region v2SetUserPassword
        [Test]
        public void v2SetUserPassword() {
            var result = AuthClient.v2SetUserPassword(CurseSiteID, User.ID, CurseEncPassword, CurseEncPassword);
            Assert.AreEqual(ESetUserPasswordResult.Success, result);
        }

        [Test]
        public void v2SetUserPasswordWithInvalidUser() {
            var result = AuthClient.v2SetUserPassword(CurseSiteID, -1, CurseEncPassword, CurseEncPassword);
            Assert.AreEqual(ESetUserPasswordResult.UserNotFound, result);
        }

        [Test]
        public void v2SetUserPasswordWithInvalidPassword() {
            var result = AuthClient.v2SetUserPassword(CurseSiteID, User.ID, CurseEncPassword, InvalidPassword);
            Assert.AreEqual(ESetUserPasswordResult.InvalidPassword, result);
        }
        #endregion

        #region v2AdminSetUserPassword
        [Test]
        public void v2AdminSetUserPassword() {
            var result = AuthClient.v2AdminSetUserPassword(CurseSiteID, User.ID, CurseEncPassword);
            Assert.AreEqual(EAdminSetUserPasswordStatus.Success, result);
        }

        [Test]
        public void v2AdminSetUserPasswordInvalidUser() {
            var result = AuthClient.v2AdminSetUserPassword(CurseSiteID, -1, CurseEncPassword);
            Assert.AreEqual(EAdminSetUserPasswordStatus.UserNotFound, result);
        }

        [Test]
        public void v2AdminSetUserPasswordInvalidPassword() {
            var result = AuthClient.v2AdminSetUserPassword(CurseSiteID, User.ID, InvalidPassword);
            Assert.AreEqual(EAdminSetUserPasswordStatus.InvalidPassword, result);
        }

        [Test]
        public void v2AdminSetUserPasswordInsufficientPermissions() {
            var result = AuthClient.v2AdminSetUserPassword(-1, User.ID, CurseEncPassword);
            Assert.AreEqual(EAdminSetUserPasswordStatus.InsufficientPermissions, result);
        }
        #endregion

        #region v2AdminSetBinaryUserLegacyPassword
        [Test]
        public void v2AdminSetBinaryUserLegacyPassword() {
            var result = AuthClient.v2AdminSetBinaryUserLegacyPassword(CurseSiteID, LegacyID, LegacyUser.ID, CurseEncPassword);
            Assert.AreEqual(EAdminSetUserPasswordStatus.Success, result);
        }

        [Test]
        public void v2AdminSetBinaryUserLegacyPasswordInvalidUser() {
            var result = AuthClient.v2AdminSetBinaryUserLegacyPassword(CurseSiteID, LegacyID, -1, CurseEncPassword);
            Assert.AreEqual(EAdminSetUserPasswordStatus.UserNotFound, result);
        }

        [Test(Description = "There is no invalid password for this method.")]
        public void v2AdminSetBinaryUserLegacyPasswordInvalidPassword() {
            //var result = AuthClient.v2AdminSetBinaryUserLegacyPassword(CurseSiteID, LegacyID, LegacyUser.ID, InvalidPassword);
            //Assert.AreEqual(EAdminSetUserPasswordStatus.InvalidPassword, result);
            Assert.Pass("There is no invalid password for this method.");
        }

        [Test]
        public void v2AdminSetBinaryUserLegacyPasswordInsufficientPermissions() {
            var result = AuthClient.v2AdminSetBinaryUserLegacyPassword(-1, LegacyID, LegacyUser.ID, CurseEncPassword);
            Assert.AreEqual(EAdminSetUserPasswordStatus.InsufficientPermissions, result);
        }
        #endregion

        #region v2AdminCreditUserWithRename
        [Test]
        public void v2AdminCreditUserWithRename() {
            var result = AuthClient.v2AdminCreditUserWithRename(CurseSiteID, User.ID);
            Assert.AreEqual(EAdminCreditUserWithRenameStatus.Success, result);
        }

        [Test]
        public void v2AdminCreditUserWithRenameAlreadyCredited() {
            var result = AuthClient.v2AdminCreditUserWithRename(CurseSiteID, User.ID);
            Assert.AreEqual(EAdminCreditUserWithRenameStatus.AlreadyCredited, result);
        }

        [Test]
        public void v2AdminCreditUserWithRenameInsufficientPermissions() {
            var result = AuthClient.v2AdminCreditUserWithRename(-1, User.ID);
            Assert.AreEqual(EAdminCreditUserWithRenameStatus.InsufficientPermissions, result);
        }

        [Test]
        public void v2AdminCreditUserWithRenameInvalidUser() {
            var result = AuthClient.v2AdminCreditUserWithRename(CurseSiteID, -1);
            Assert.AreEqual(EAdminCreditUserWithRenameStatus.UserNotFound, result);
        }
        #endregion

        #region v2AdminClearUserCache
        [Test]
        public void v2AdminClearUserCache() {
            var result = AuthClient.v2AdminClearUserCache(CurseSiteID, User.ID);
            Assert.AreEqual(EAdminClearUserCacheStatus.Success, result);
        }

        [Test]
        public void v2AdminClearUserCacheInsifficientPermissions() {
            var result = AuthClient.v2AdminClearUserCache(-1, User.ID);
            Assert.AreEqual(EAdminClearUserCacheStatus.InsufficientPermissions, result);
        }

        [Test]
        public void v2AdminClearUserCacheInvalidUser() {
            var result = AuthClient.v2AdminClearUserCache(CurseSiteID, -1);
            Assert.AreEqual(EAdminClearUserCacheStatus.UserNotFound, result);
        }
        #endregion

        [Test(Description = "This is tested to extent in the user object validation class.")]
        public void v2GetUserProfile() {
            var result = AuthClient.v2GetUserProfile(CurseSiteID, User.ID);
            Assert.IsNotNull(result);
        }

        #region v2ValidateClientUser
        [Test]
        public void v2ValidateClientUser() {
            var result = AuthClient.v2ValidateClientUser(CurseSiteID, User.Name, CurseEncPassword);
            Assert.AreEqual(ELoginStatus.Success, result.Status, string.Format("username: {0}, password: {1}", User.Name, CurseEncPassword));
        }

        [Test]
        public void v2ValidateClientUserInvalidSite() {
            var result = AuthClient.v2ValidateClientUser(-1, User.Name, CurseEncPassword);
            Assert.AreEqual(ELoginStatus.UnauthorizedLogin, result.Status);
        }

        [Test]
        public void v2ValidateClientUserUnknownUsername() {
            var result = AuthClient.v2ValidateClientUser(CurseSiteID, InvalidUsername, CurseEncPassword);
            Assert.AreEqual(ELoginStatus.UnknownUsername, result.Status);
        }

        [Test]
        public void v2ValidateClientUserUnknownEmail() {
            var result = AuthClient.v2ValidateClientUser(CurseSiteID, RandomEmailAddress, CurseEncPassword);
            Assert.AreEqual(ELoginStatus.UnknownEmail, result.Status);
        }

        [Test]
        public void v2ValidateClientUserInvalidPassword() {
            var result = AuthClient.v2ValidateClientUser(CurseSiteID, User.Name, InvalidPassword);
            Assert.AreEqual(ELoginStatus.InvalidPassword, result.Status, string.Format("username: {0}, password: {1}", User.Name, CurseEncPassword));
        }
        #endregion

        #region v2ValidateWebUser
        [Test]
        public void v2ValidateWebUser() {
            var result = AuthClient.v2ValidateWebUser(CurseSiteID, User.Name, CurseEncPassword, SessionID);
            Assert.AreEqual(ELoginStatus.Success, result.Status, string.Format("username: {0}, password: {1}", User.Name, CurseEncPassword));
        }

        [Test]
        public void v2ValidateWebUserInvalidSite() {
            var result = AuthClient.v2ValidateWebUser(-1, User.Name, CurseEncPassword, SessionID);
            Assert.AreEqual(ELoginStatus.UnauthorizedLogin, result.Status);
        }

        [Test]
        public void v2ValidateWebUserInvalidSession() {
            var result = AuthClient.v2ValidateWebUser(CurseSiteID, User.Name, CurseEncPassword, InvalidSessionID);
            Assert.AreEqual(ELoginStatus.InvalidSession, result.Status);
        }

        [Test]
        public void v2ValidateWebUserUnknownUsername() {
            var result = AuthClient.v2ValidateWebUser(CurseSiteID, RandomUsername, CurseEncPassword, SessionID);
            Assert.AreEqual(ELoginStatus.UnknownUsername, result.Status);
        }

        [Test]
        public void v2ValidateWebUserUnknownEmail() {
            var result = AuthClient.v2ValidateWebUser(CurseSiteID, RandomEmailAddress, CurseEncPassword, SessionID);
            Assert.AreEqual(ELoginStatus.UnknownEmail, result.Status);
        }

        [Test]
        public void v2ValidateWebUserInvalidPassword() {
            var result = AuthClient.v2ValidateWebUser(CurseSiteID, User.Name, InvalidPassword, SessionID);
            Assert.AreEqual(ELoginStatus.InvalidPassword, result.Status, string.Format("username: {0}, password: {1}", User.Name, CurseEncPassword));
        }
        #endregion

        #region v2ValidateUserSession
        [Test]
        public void v2ValidateUserSession() {
            var loginResult = AuthClient.v2ValidateWebUser(CurseSiteID, User.Name, CurseEncPassword, SessionID);
            if (loginResult.Status != ELoginStatus.Success) {
                Assert.Inconclusive("Unable to log user in.");
            }

            var result = AuthClient.v2ValidateUserSession(CurseSiteID, loginResult.SessionId);
            Assert.AreEqual(result, User.ID);
        }

        [Test]
        public void v2ValidateUserSessionInvalidSite() {
            var result = AuthClient.v2ValidateUserSession(-1, SessionID);
            Assert.AreEqual(-1, result);
        }
        #endregion

        #region v2AdminRenameUser
        [Test]
        public void v2AdminRenameUser() {
            var result = AuthClient.v2AdminRenameUser(CurseSiteID, User.ID, RandomUsername);
            Assert.AreEqual(EAdminRenameUserStatus.Success, result);
        }

        [Test]
        public void v2AdminRenameUserInvalidSite() {
            var result = AuthClient.v2AdminRenameUser(-1, User.ID, RandomUsername);
            Assert.AreEqual(EAdminRenameUserStatus.InsufficientPermissions, result);
        }

        [Test]
        public void v2AdminRenameUserUnknownUser() {
            var result = AuthClient.v2AdminRenameUser(CurseSiteID, -1, RandomUsername);
            Assert.AreEqual(EAdminRenameUserStatus.UserNotFound, result);
        }

        [Test]
        public void v2AdminRenameUserExistingUsername() {
            if (string.IsNullOrEmpty(ExistingUserName)) {
                Assert.Inconclusive("Failed to pull an existing username.");
            }

            var result = AuthClient.v2AdminRenameUser(CurseSiteID, User.ID, ExistingUserName);
            Assert.AreEqual(EAdminRenameUserStatus.UsernameInUse, result, ExistingUserName);
        }

        [Test]
        public void v2AdminRenameUserInvalidUsername() {
            var result = AuthClient.v2AdminRenameUser(CurseSiteID, User.ID, InvalidUsername);
            Assert.AreEqual(EAdminRenameUserStatus.InvalidUsername, result);
        }
        #endregion

        #region v2SignOutUser
        [Test]
        public void v2SignOutUser() {
            var result = AuthClient.v2SignOutUser(CurseSiteID, User.ID);
            Assert.AreEqual(true, result);
        }

        [Test]
        public void v2SignOutUserInvalidSite() {
            var result = AuthClient.v2SignOutUser(-1, User.ID);
            Assert.AreEqual(false, result);
        }

        [Test]
        public void v2SignOutUserInvalidUser() {
            var result = AuthClient.v2SignOutUser(CurseSiteID, -1);
            Assert.AreEqual(false, result);
        }
        #endregion

        #region v2UpdatePrimaryEmailAddress
        [Test]
        public void v2UpdatePrimaryEmailAddress() {
            var result = AuthClient.v2UpdatePrimaryEmailAddress(CurseSiteID, User.ID, RandomEmailAddress);
            Assert.AreEqual(EUpdatePrimaryEmailStatus.Successful, result);
        }

        [Test]
        public void v2UpdatePrimaryEmailAddressInvalidSite() {
            var result = AuthClient.v2UpdatePrimaryEmailAddress(-1, User.ID, RandomEmailAddress);
            Assert.AreEqual(EUpdatePrimaryEmailStatus.InsufficientPermissions, result);
        }

        [Test]
        public void v2UpdatePrimaryEmailAddressInvalidEmail() {
            var result = AuthClient.v2UpdatePrimaryEmailAddress(CurseSiteID, User.ID, InvalidEmail);
            Assert.AreEqual(EUpdatePrimaryEmailStatus.InvalidEmail, result);
        }

        [Test]
        public void v2UpdatePrimaryEmailAddressUnknownUser() {
            var result = AuthClient.v2UpdatePrimaryEmailAddress(CurseSiteID, -1, RandomEmailAddress);
            Assert.AreEqual(EUpdatePrimaryEmailStatus.UnknownUser, result);
        }

        [Test]
        public void v2UpdatePrimaryEmailAddressExistingEmail() {
            if (string.IsNullOrEmpty(ExistingEmail)) {
                Assert.Inconclusive("No existing email.");
            }

            var result = AuthClient.v2UpdatePrimaryEmailAddress(CurseSiteID, User.ID, ExistingEmail);
            Assert.AreEqual(EUpdatePrimaryEmailStatus.EmailInUse, result);
        }
        #endregion

        #region v2GetuserIDFromEmail
        [Test]
        public void v2GetuserIDFromEmail() {
            var result = AuthClient.v2GetUserIDFromEmail(CurseSiteID, User.Profile.Emails[0].Address);
            Assert.AreEqual(User.ID, result);
        }

        [Test]
        public void v2GetuserIDFromEmailInvalidSite() {
            var result = AuthClient.v2GetUserIDFromEmail(-1, User.Profile.Emails[0].Address);
            Assert.AreEqual(0, result);
        }

        [Test]
        public void v2GetuserIDFromEmailUnknownEmail() {
            var result = AuthClient.v2GetUserIDFromEmail(CurseSiteID, RandomEmailAddress);
            Assert.AreEqual(0, result);
        }
        #endregion

        #region v2GetSubscriptionStatusPremium
        [Test]
        public void v2GetSubscriptionStatusPremium() {
            var result = AuthClient.v2GetSubscriptionStatus(CurseSiteID, User.ID, ESubscriptionType.Premium);
            Assert.AreEqual(User.Subscriptions.Any(p => p.Type == ESubscriptionType.Premium && p.ExpirationDate > DateTime.UtcNow), result);
        }

        [Test]
        public void v2GetSubscriptionStatusGuildHosting() {
            var result = AuthClient.v2GetSubscriptionStatus(CurseSiteID, User.ID, ESubscriptionType.GuildHosting);
            Assert.AreEqual(User.Subscriptions.Any(p => p.Type == ESubscriptionType.GuildHosting && p.ExpirationDate > DateTime.UtcNow), result);
        }

        [Test]
        public void v2GetSubscriptionStatusVentrilloHosting() {
            var result = AuthClient.v2GetSubscriptionStatus(CurseSiteID, User.ID, ESubscriptionType.VentriloHosting);
            Assert.AreEqual(User.Subscriptions.Any(p => p.Type == ESubscriptionType.VentriloHosting && p.ExpirationDate > DateTime.UtcNow), result);
        }

        [Test]
        public void v2GetSubscriptionStatusInvalidSite() {
            var result = AuthClient.v2GetSubscriptionStatus(-1, User.ID, ESubscriptionType.Premium);
            Assert.AreEqual(false, result);
        }

        [Test]
        public void v2GetSubscriptionStatusInvalidUser() {
            var result = AuthClient.v2GetSubscriptionStatus(CurseSiteID, -1, ESubscriptionType.Premium);
            Assert.AreEqual(false, result);
        }
        #endregion

        #region v2InitiateResetPassword
        [Test]
        public void v2InitiateResetPassword() {
            var result = AuthClient.v2InitiateResetPassword(CurseSiteID, User.Profile.Emails[0].Address, "www.curse.com");
            Assert.AreEqual(EResetPasswordStatus.Successful, result);
        }

        [Test]
        public void v2InitiateResetPasswordInvalidSite() {
            var result = AuthClient.v2InitiateResetPassword(-1, User.Profile.Emails[0].Address, "www.curse.com");
            Assert.AreEqual(EResetPasswordStatus.InsufficientPermissions, result);
        }

        [Test]
        public void v2InitiateResetPasswordInvalidEmail() {
            var result = AuthClient.v2InitiateResetPassword(CurseSiteID, InvalidEmail, "www.curse.com");
            Assert.AreEqual(EResetPasswordStatus.InvalidEmail, result);
        }

        [Test]
        public void v2InitiateResetPasswordUnknownEmail() {
            var result = AuthClient.v2InitiateResetPassword(CurseSiteID, RandomEmailAddress, "www.curse.com");
            Assert.AreEqual(EResetPasswordStatus.UnknownEmail, result);
        }
        #endregion

        [Test]
        public void v2RecoverAccount() {
            Assert.Inconclusive("Not Impelmented");
        }

        [Test]
        public void v2GetAccountRecoveryRequestByToken() {
            Assert.Inconclusive("Not Impelmented");
        }

        [Test]
        public void v2getAccountRecoveryRequestByUserID() {
            Assert.Inconclusive("Not Impelmented");
        }

        [Test]
        public void v2ResetPassword() {
            Assert.Inconclusive("Not Impelmented");
        }

        [Test]
        public void v2RecoverUsername() {
            Assert.Inconclusive("Not Impelmented");
        }

        #region v2AddUser
        [Test]
        public void v2AddUser() {
            AddUserResult result = AuthClient.v2AddUser(CurseSiteID, RandomUsername, CurseEncPassword, FirstName, LastName, BirthDate, Gender, Country, Region, City, RandomEmailAddress, NewsLetter, "From", "Redirect", "Session");
            Assert.AreEqual(EAddUserStatus.Successful, result.Status);

            Utilities.DeleteUser(CurseSiteID, result.UserID);
        }

        [Test]
        public void v2AddUserInvalidSite() {
            AddUserResult result = AuthClient.v2AddUser(-1, Username, CurseEncPassword, FirstName, LastName, BirthDate, Gender, Country, Region, City, RandomEmailAddress, NewsLetter, "From", "Redirect", "Session");
            Assert.AreEqual(EAddUserStatus.InsufficientPermissions, result.Status);
        }

        [Test]
        public void v2AddUserExistingUser() {
            if (string.IsNullOrEmpty(ExistingUserName)) {
                Assert.Inconclusive("Failed to pull an existing username.");
            }

            AddUserResult result = AuthClient.v2AddUser(CurseSiteID, ExistingUserName, CurseEncPassword, FirstName, LastName, BirthDate, Gender, Country, Region, City, RandomEmailAddress, NewsLetter, "From", "Redirect", "Session");
            Assert.AreEqual(EAddUserStatus.UsernameInUse, result.Status);
        }

        [Test]
        public void v2AddUserExistingEmail() {
            if (string.IsNullOrEmpty(ExistingEmail)) {
                Assert.Inconclusive("Failed to pull an existing email.");
            }

            AddUserResult result = AuthClient.v2AddUser(CurseSiteID, RandomUsername, CurseEncPassword, FirstName, LastName, BirthDate, Gender, Country, Region, City, ExistingEmail, NewsLetter, "From", "Redirect", "Session");
            Assert.AreEqual(EAddUserStatus.EmailInUse, result.Status);
        }

        [Test]
        public void v2AddUserInvalidPassword() {
            AddUserResult result = AuthClient.v2AddUser(CurseSiteID, RandomUsername, InvalidPassword, FirstName, LastName, BirthDate, Gender, Country, Region, City, RandomEmailAddress, NewsLetter, "From", "Redirect", "Session");
            Assert.AreEqual(EAddUserStatus.InvalidPassword, result.Status);
        }

        [Test]
        public void v2AddUserInvalidUsername() {
            AddUserResult result = AuthClient.v2AddUser(CurseSiteID, InvalidUsername, CurseEncPassword, FirstName, LastName, BirthDate, Gender, Country, Region, City, RandomEmailAddress, NewsLetter, "From", "Redirect", "Session");
            Assert.AreEqual(EAddUserStatus.InvalidUsername, result.Status);
        }

        [Test]
        public void v2AddUserInvalidBirthday() {
            AddUserResult result = AuthClient.v2AddUser(CurseSiteID, RandomUsername, CurseEncPassword, FirstName, LastName, InvalidBirthDay, Gender, Country, Region, City, RandomEmailAddress, NewsLetter, "From", "Redirect", "Session");
            Assert.AreEqual(EAddUserStatus.InvalidProfile, result.Status);
        }

        [Test]
        public void v2AddUserInvalidGender() {
            AddUserResult result = AuthClient.v2AddUser(CurseSiteID, RandomUsername, CurseEncPassword, FirstName, LastName, BirthDate, InvalidGender, Country, Region, City, RandomEmailAddress, NewsLetter, "From", "Redirect", "Session");
            Assert.AreEqual(EAddUserStatus.InvalidProfile, result.Status);
        }

        [Test]
        public void v2AddUserInvalidFirstname() {
            AddUserResult result = AuthClient.v2AddUser(CurseSiteID, RandomUsername, CurseEncPassword, InvalidName, LastName, BirthDate, Gender, Country, Region, City, RandomEmailAddress, NewsLetter, "From", "Redirect", "Session");
            Assert.AreEqual(EAddUserStatus.InvalidProfile, result.Status);
        }

        [Test]
        public void v2AddUserInvalidLastname() {
            AddUserResult result = AuthClient.v2AddUser(CurseSiteID, RandomUsername, CurseEncPassword, FirstName, InvalidName, BirthDate, Gender, Country, Region, City, RandomEmailAddress, NewsLetter, "From", "Redirect", "Session");
            Assert.AreEqual(EAddUserStatus.InvalidProfile, result.Status);
        }
        #endregion

        #region v2AdminDeleteUser
        [Test]
        public void v2AdminDeleteUser() {
            var userId = Utilities.CreateUser(CurseSiteID, RandomUsername, CurseEncPassword, FirstName, LastName, BirthDate, Gender, RandomEmailAddress);
            if (userId == 0) {
                Assert.Inconclusive("Failed to create a user");
            }// if there is no user id

            //Delete the user
            DeleteUserResult result = AuthClient.v2AdminDeleteUser(CurseSiteID, userId);
            Assert.AreEqual(EDeleteUserStatus.Success, result.Status);
        }

        [Test]
        public void v2AdminDeleteUserInvalidSite() {
            DeleteUserResult result = AuthClient.v2AdminDeleteUser(-1, -1);
            Assert.AreEqual(EDeleteUserStatus.InsufficientPermissions, result.Status);
        }

        [Test]
        public void v2AdminDeleteUserInvalidUser() {
            DeleteUserResult result = AuthClient.v2AdminDeleteUser(CurseSiteID, -1);
            Assert.AreEqual(EDeleteUserStatus.UserNotFound, result.Status);
        }
        #endregion

        #region v2AdminAddLegacyUser
        [Test]
        public void v2AdminAddLegacyUser() {
            var result = AuthClient.v2AdminAddLegacyUser(
                CurseSiteID,
                LegacyID,
                RandomUsername,
                CurseEncPassword,
                FirstName,
                LastName,
                BirthDate,
                Gender,
                Country,
                Region,
                City,
                RandomEmailAddress,
                NewsLetter,
                "From",
                "Redirect",
                null
            );

            Assert.AreEqual(EAddUserStatus.Successful, result.Status);

            if (result.Status == EAddUserStatus.Successful) {
                Utilities.DeleteUser(CurseSiteID, result.UserID);
            }
        }

        [Test]
        public void v2AdminAddLegacyUserInvalidSite() {
            var result = AuthClient.v2AdminAddLegacyUser(
                -1,
                LegacyID,
                RandomUsername,
                CurseEncPassword,
                FirstName,
                LastName,
                BirthDate,
                Gender,
                Country,
                Region,
                City,
                RandomEmailAddress,
                NewsLetter,
                "From",
                "Redirect",
                null
            );

            Assert.AreEqual(EAddUserStatus.InsufficientPermissions, result.Status);
        }

        [Test]
        public void v2AdminAddLegacyUserExistingUser() {
            if (string.IsNullOrEmpty(ExistingUserName)) {
                Assert.Inconclusive("Failed to pull an existing username.");
            }

            var result = AuthClient.v2AdminAddLegacyUser(
                CurseSiteID,
                LegacyID,
                ExistingUserName,
                CurseEncPassword,
                FirstName,
                LastName,
                BirthDate,
                Gender,
                Country,
                Region,
                City,
                RandomEmailAddress,
                NewsLetter,
                "From",
                "Redirect",
                null
            );

            Assert.AreEqual(EAddUserStatus.UsernameInUse, result.Status);
        }

        [Test]
        public void v2AdminAddLegacyUserExistingEmail() {
            if (string.IsNullOrEmpty(ExistingEmail)) {
                Assert.Inconclusive("Failed to pull an existing email.");
            }

            var result = AuthClient.v2AdminAddLegacyUser(
                CurseSiteID,
                LegacyID,
                RandomUsername,
                CurseEncPassword,
                FirstName,
                LastName,
                BirthDate,
                Gender,
                Country,
                Region,
                City,
                ExistingEmail,
                NewsLetter,
                "From",
                "Redirect",
                "Session"
            );

            Assert.AreEqual(EAddUserStatus.EmailInUse, result.Status);
        }

        [Test(Description = "There is no invalid password for this method.")]
        public void v2AdminAddLegacyUserInvalidPassword() {
            //var result = AuthClient.v2AdminAddLegacyUser(
            //    CurseSiteID,
            //    LegacyID,
            //    RandomUsername,
            //    InvalidPassword,
            //    FirstName,
            //    LastName,
            //    BirthDate,
            //    Gender,
            //    Country,
            //    Region,
            //    City,
            //    RandomEmailAddress,
            //    NewsLetter,
            //    "From",
            //    "Redirect",
            //    null
            //);

            //Assert.AreEqual(EAddUserStatus.InvalidPassword, result.Status);
            Assert.Pass("There is no invalid password for this method.");
        }

        [Test]
        public void v2AdminAddLegacyUserInvalidUsername() {
            var result = AuthClient.v2AdminAddLegacyUser(
                CurseSiteID,
                LegacyID,
                InvalidUsername,
                CurseEncPassword,
                FirstName,
                LastName,
                BirthDate,
                Gender,
                Country,
                Region,
                City,
                RandomEmailAddress,
                NewsLetter,
                "From",
                "Redirect",
                null
            );

            Assert.AreEqual(EAddUserStatus.InvalidUsername, result.Status);
        }

        [Test]
        public void v2AdminAddLegacyUserInvalidBirthDay() {
            var result = AuthClient.v2AdminAddLegacyUser(
                CurseSiteID,
                LegacyID,
                RandomUsername,
                CurseEncPassword,
                FirstName,
                LastName,
                InvalidBirthDay,
                Gender,
                Country,
                Region,
                City,
                RandomEmailAddress,
                NewsLetter,
                "From",
                "Redirect",
                null
            );

            Assert.AreEqual(EAddUserStatus.InvalidProfile, result.Status);
        }

        [Test]
        public void v2AdminAddLegacyUserInvalidGender() {
            var result = AuthClient.v2AdminAddLegacyUser(
                CurseSiteID,
                LegacyID,
                RandomUsername,
                CurseEncPassword,
                FirstName,
                LastName,
                BirthDate,
                InvalidGender,
                Country,
                Region,
                City,
                RandomEmailAddress,
                NewsLetter,
                "From",
                "Redirect",
                null
            );

            Assert.AreEqual(EAddUserStatus.InvalidProfile, result.Status);
        }

        [Test]
        public void v2AdminAddLegacyUserInvalidFirstname() {
            var result = AuthClient.v2AdminAddLegacyUser(
                CurseSiteID,
                LegacyID,
                RandomUsername,
                CurseEncPassword,
                InvalidName,
                LastName,
                BirthDate,
                Gender,
                Country,
                Region,
                City,
                RandomEmailAddress,
                NewsLetter,
                "From",
                "Redirect",
                null
            );

            Assert.AreEqual(EAddUserStatus.InvalidProfile, result.Status);
        }

        [Test]
        public void v2AdminAddLegacyUserInvalidLastname() {
            var result = AuthClient.v2AdminAddLegacyUser(
                CurseSiteID,
                LegacyID,
                RandomUsername,
                CurseEncPassword,
                FirstName,
                InvalidName,
                BirthDate,
                Gender,
                Country,
                Region,
                City,
                RandomEmailAddress,
                NewsLetter,
                "From",
                "Redirect",
                null
            );

            Assert.AreEqual(EAddUserStatus.InvalidProfile, result.Status);
        }
        #endregion

        [Test]
        public void v2ValidateUsername() {
            Assert.Inconclusive("Not Impelmented");
        }

        [Test]
        public void v2IsUSernameAvailable() {
            Assert.Inconclusive("Not Impelmented");
        }

        [Test]
        public void v2GetUserIDFromUsername() {
            Assert.Inconclusive("Not Impelmented");
        }

        [Test]
        public void v2GetDeletedUsersFromTime() {
            var mark = DateTime.Parse("01/01/2011");
            var results = AuthClient.v2GetDeletedUsersFromTime(mark);

            using (var conn = new SqlConnection(AuthConnectionString)) {
                using (var cmd = new SqlCommand("select count(_uid) from userdeletion where _date >= @mark", conn)) {
                    cmd.Parameters.Add("mark", SqlDbType.DateTime).Value = mark;

                    conn.Open();
                    var count = (int)cmd.ExecuteScalar();

                    Assert.AreEqual(count, results.Length);
                }//using cmd
            }//using conn
        }

        [Test]
        public void v2AddSubscriptionCredit() {
            Assert.Inconclusive("Not Impelmented");
        }

        [Test]
        public void v2SetPremiumExpiration() {
            Assert.Inconclusive("Not Impelmented");
        }

        [Test]
        public void v2ActivateEmailAddress() {
            Assert.Inconclusive("Not Impelmented");
        }

        [Test]
        public void v2AddNetworkSiteIP()
        {
            var result = AuthClient.v2AddNetworkSiteIP("A", 7777788, "10.0.0.120");
            Assert.AreEqual(EGenericAuthenticationStatus.Successful , result.Status);
        }

        //[Test]
        //public void TestGeneratedAccounts()
        //{
        //    var offset = 1001;
        //    var failures = string.Empty;
        //    for (int i = 0; i < 350; i++)
        //    {
        //        var userName = string.Format("EuPayment{0}", (i + offset).ToString().PadLeft(3, '0'));
        //        var result = AuthClient.v2ValidateWebUser(CurseSiteID, userName, CurseEncPassword, SessionID);
        //        if (result.Status != ELoginStatus.Success)
        //        {
        //            failures += string.Format("Failed to LoginUser user: {0}", userName) + Environment.NewLine;
        //            Assert.Fail(userName);
        //        }// log errors
        //    }

        //    if (!string.IsNullOrEmpty(failures))
        //    {
        //        Assert.Fail(failures);
        //    }
        //    else Assert.Pass();
        //}

        //[Test]
        //public void GenerateAccountsForEUTesting()
        //{
        //    var offset = 1001;
        //    var failures = string.Empty;
        //    for (int i = 0; i < 350; i++)
        //    {
        //        var userName = string.Format("EuPayment{0}", (i + offset).ToString().PadLeft(3, '0'));
        //        //Assert.Fail(userName);
        //        AddUserResult result = AuthClient.v2AddUser(CurseSiteID, userName, CurseEncPassword, FirstName, LastName, BirthDate, Gender, "US", "AL", "Huntsville", userName + "@curse.com", false, "From", "Redirect", "Session");
        //        if (result.Status != EAddUserStatus.Successful)
        //        {
        //            failures += string.Format("Failed to create user: {0}", userName) + Environment.NewLine;
        //            Assert.Fail(result.Status + " " + result.Message);
        //        }// log errors
        //    }

        //    if (!string.IsNullOrEmpty(failures))
        //    {
        //        Assert.Fail(failures);
        //    }
        //    else Assert.Pass();
        //}

        //[Test]
        //public void DeleteTestUsers()
        //{
        //    var failures = string.Empty;
        //    int[] userIds = { 9293763, 9293764, 9293765, 9293766, 9293767, 9293768, 9293769, 9293770, 9293771, 9293772, 9293773, 9293774, 9293775, 9293776, 9293777, 9293778, 9293779, 9293780, 9293781, 9293782, 9293783, 9293784, 9293785, 9293786, 9293787, 9293788, 9293789, 9293791, 9293792, 9293793, 9293794, 9293795, 9293796, 9293797, 9293798, 9293799, 9293800, 9293801, 9293802, 9293803, 9293804, 9293805, 9293806, 9293807, 9293808, 9293809, 9293810, 9293811, 9293812, 9293813, 9293814, 9293815, 9293816, 9293817, 9293818, 9293819, 9293820, 9293821, 9293822, 9293823, 9293824, 9293825, 9293826, 9293827, 9293828, 9293829, 9293830, 9293831, 9293832, 9293833, 9293834, 9293835, 9293836, 9293837, 9293838, 9293839, 9293840, 9293841, 9293842, 9293843, 9293844, 9293845, 9293846, 9293848, 9293849, 9293850, 9293851, 9293852, 9293853, 9293854, 9293855, 9293856, 9293857, 9293858, 9293859, 9293860, 9293861, 9293862, 9293863, 9293864, 9293865, 9293866, 9293867, 9293868, 9293869, 9293870, 9293871, 9293872, 9293873, 9293874, 9293875, 9293876, 9293877, 9293878, 9293879, 9293881, 9293882, 9293883, 9293884, 9293885, 9293886, 9293887, 9293888, 9293889, 9293890, 9293891, 9293892, 9293893, 9293894, 9293895, 9293896, 9293897, 9293898, 9293899, 9293900, 9293901, 9293902, 9293903, 9293904, 9293944 };
        //    foreach (var userId in userIds)
        //    {
        //        DeleteUserResult result = AuthClient.v2AdminDeleteUser(CurseSiteID, userId);
        //        if (result.Status != EDeleteUserStatus.Success)
        //        {
        //            failures += string.Format("Failed to delete userid: {0}", userId) + Environment.NewLine;
        //        }
        //    }
            
        //    if (!string.IsNullOrEmpty(failures))
        //    {
        //        Assert.Fail(failures);
        //    }
        //    else Assert.Pass();
        //}
    }
}
