﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using ServiceUnitTests.Attributes;
using ServiceUnitTests.TestFixtures;
using NUnit.Framework;
using Curse.CurseClient.Common.ClientService;
using Curse.AddOns;
using System.Data.SqlClient;
using System.Data;
using ServiceUnitTests.Extensions;
using Curse.DownloadSecurity.Tokens;
using System.Net;
using System.ServiceModel;
using Curse.ClientService;

namespace ServiceUnitTests.UnitTests.ClientService {
    [ClientUnitTests]
    class AddonTests : ClientServiceTestFixture {
        private static int _tokenFileId = 552969;
        private static string _tokenAddonUrl = @"http://addons.curse.cursecdn.com/files/552/969/Dragonscale_Green_Complete.zip?v3={0}&p2=1622694999";

        [Test]
        public void GetAddon() {
            var addonId = Addons.Random().Key;
            var result = Proxy.TryCall<CAddOn>(delegate(CClientServiceClient svc) {
                return svc.GetAddOn(addonId);
            });

            if (result != null) {
                using (var conn = new SqlConnection(RadonConnectionString)) {
                    var cmd = new SqlCommand("SELECT * FROM Project WHERE ID = @id", conn);
                    cmd.Parameters.Add("id", SqlDbType.Int).Value = addonId;

                    conn.Open();
                    using (var reader = cmd.ExecuteReader()) {
                        reader.Read();
                        Assert.AreEqual((string)reader["Name"], result.Name);
                    }
                }

                Addons[addonId] = result;
            }
            else Assert.IsNotNull(result);
        }

        [Test]
        public void GetAddOnDescription() {
            var addonId = Addons.Random().Key;
            var result = Proxy.TryCall<string>(delegate(CClientServiceClient svc) {
                return svc.GetAddOnDescription(addonId);
            });

            if (result != null) {
                using (var conn = new SqlConnection(RadonConnectionString)) {
                    var cmd = new SqlCommand("SELECT [Text] FROM ProjectDescription WHERE ProjectID = @id", conn);
                    cmd.Parameters.Add("id", SqlDbType.Int).Value = addonId;

                    conn.Open();
                    var description = cmd.ExecuteScalar();
                    if (description != DBNull.Value) {
                        Assert.AreEqual((string)description, result);
                    }
                }
            }
            else Assert.IsNotNull(result);
        }

        [Test]
        public void GetAddOnFile() {
            var addonId = Addons.Random().Key;
            var fileId = FileIdsByAddon[addonId].Random();

            var result = Proxy.TryCall<CAddOnFile>(delegate(CClientServiceClient svc) {
                return svc.GetAddOnFile(addonId, fileId);
            });

            if (result != null) {
                using (var conn = new SqlConnection(RadonConnectionString)) {
                    var cmd = new SqlCommand("SELECT Name FROM ProjectFile WHERE ID = @id", conn);
                    cmd.Parameters.Add("id", SqlDbType.Int).Value = fileId;

                    conn.Open();
                    var filename = cmd.ExecuteScalar();
                    if (filename != DBNull.Value) {
                        Assert.AreEqual((string)filename, result.FileName);
                    }
                }
            }
            else Assert.IsNotNull(result);
        }

        [Test]
        public void GetAddOns() {
            var addons = Proxy.TryCall<List<CAddOn>>(delegate(CClientServiceClient svc) {
                return svc.GetAddOns(Addons.Keys.ToList());
            });

            if (addons != null) {
                var addonDictionary = addons.ToDictionary(p => p.Id);
                var missing = Addons.Where(p => !addonDictionary.Keys.Contains(p.Key));
                Assert.AreEqual(Addons.Count, addons.Count, string.Join(",\r\n", missing.ToArray()));
            }
            else Assert.IsNotNull(addons);
        }

        [Test]
        public void GetChangeLog() {
            var addonId = Addons.Random().Key;
            var fileId = FileIdsByAddon[addonId].Random();

            var result = Proxy.TryCall<string>(delegate(CClientServiceClient svc) {
                return svc.GetChangeLog(addonId, fileId);
            });

            if (result != null) {
                using (var conn = new SqlConnection(RadonConnectionString)) {
                    var cmd = new SqlCommand("SELECT ChangeLog FROM ProjectFile WHERE ID = @ID AND ProjectID = @ProjectID", conn);
                    cmd.Parameters.Add("ID", SqlDbType.Int).Value = fileId;
                    cmd.Parameters.Add("ProjectID", SqlDbType.Int).Value = addonId;

                    conn.Open();
                    var changeLog = cmd.ExecuteScalar();
                    if (changeLog != DBNull.Value) {
                        Assert.AreEqual(changeLog, result);
                    }
                    else Assert.Fail(string.Format("Could not pull the change log from the DB for ProjectID: {0} and FileID: {1}", addonId, fileId));
                }
            }
            else Assert.IsNotNull(result);
        }

        [Test]
        public void GetDownloadToken() {
            var result = Proxy.TryCall<CDownloadToken>(delegate(CClientServiceClient svc) {
                return svc.GetDownloadToken(_tokenFileId);
            });

            if (result != null) {
                var hash = SecurityToken.GetHashCode(_tokenFileId, result.Timestamp);
                var request = (HttpWebRequest)WebRequest.Create(string.Format(_tokenAddonUrl, hash));

                var response = request.GetResponse();
                var contentType = response.Headers["Content-Type"];
                Assert.AreEqual(@"application/x-zip-compressed", contentType);
            }
            else Assert.IsNotNull(result);
        }

        [Test]
        public void GetFingerprintMatches() {
            var addonId = Addons.Random().Key;

            var fingerprints = new List<long>();
            using (var conn = new SqlConnection(RadonConnectionString)) {
                var cmd = new SqlCommand(string.Format("SELECT Fingerprint FROM ProjectFileModuleFingerprint WHERE FileID in ({0})", string.Join(",", FileIdsByAddon[addonId].ToArray())), conn);

                conn.Open();
                using (var reader = cmd.ExecuteReader()) {
                    while (reader.Read()) {
                        fingerprints.Add(reader.GetInt64(0));
                    }
                }
            }


            var result = Proxy.TryCall<CFingerprintMatchResult>(delegate(CClientServiceClient svc) {
                return svc.GetFingerprintMatches(fingerprints);
            });

            if (result != null) {
                foreach (var match in result.ExactMatches) {
                    Assert.AreEqual(addonId, match.Id);
                }
            }
            else Assert.IsNotNull(result);
        }

        [Test]
        public void GetFingerprintMatchesTest()
        {
            List<int> projectIds = new int[] { 40316, 40565, 41373, 31695, 32123 }.ToList();
            List<long> fingerprints = new long[] { 4154450938, 3411086108, 1566421540, 49002412, 262184644 }.ToList();
            var result = Proxy.TryCall<CFingerprintMatchResult>(delegate(CClientServiceClient svc)
            {
                return svc.GetFingerprintMatches(fingerprints);
            });

            foreach (var match in result.ExactMatches)
            {
                if (projectIds.Contains(match.Id))
                {
                    projectIds.Remove(match.Id);
                }
            }

            if (projectIds.Count > 0)
            {
                Assert.Fail(string.Format("Following Addons not found: {0}", string.Join(", ", projectIds.ToArray())));
            }
        }

        [Test]
        public void GetFuzzyMatches() {
            //var result = Proxy.TryCall<CAddOnFile>(delegate(CClientServiceClient svc) {
            //    return svc.GetFuzzyMatches(1,
            //});

            //if (result != null) {

            //}
            //else Assert.IsNotNull(result);
        }

        [Test]
        public void GetRepositoryMatchFromSlug() {
            var addonId = Addons.Random().Key;
            string gameSlug, addonSlug;

            using (var conn = new SqlConnection(RadonConnectionString)) {
                var cmd = new SqlCommand("SELECT Game.Slug, Project.Slug FROM Project INNER JOIN Game ON Game.ID = Project.GameID WHERE Project.ID = @ProjectID", conn);
                cmd.Parameters.Add("ProjectID", SqlDbType.Int).Value = addonId;

                conn.Open();
                using (var reader = cmd.ExecuteReader()) {
                    reader.Read();

                    gameSlug = reader.GetString(0);
                    addonSlug = reader.GetString(1);
                }
            }

            var result = Proxy.TryCall<CRepositoryMatch>(delegate(CClientServiceClient svc) {
                return svc.GetRepositoryMatchFromSlug(gameSlug.ToLower(), addonSlug.ToLower());
            });

            if (result != null) {
                Assert.AreEqual(addonId, result.Id);
            }
            else Assert.IsNotNull(result);
        }

        [Test]
        public void GetSecureDownloadToken() {
            var subscriptionToken = CClientAuthentication.GetSubscriptionToken(Proxy.UserID, Proxy.IsPremium);

            var result = Proxy.TryCall<CDownloadToken>(delegate(CClientServiceClient svc) {
                return svc.GetSecureDownloadToken(_tokenFileId, Proxy.UserID, Proxy.IsPremium, subscriptionToken);
            });

            if (result != null) {
                var hash = SecurityToken.GetHashCode(_tokenFileId, result.Timestamp);
                var request = (HttpWebRequest)WebRequest.Create(string.Format(_tokenAddonUrl, hash));

                var response = request.GetResponse();
                var contentType = response.Headers["Content-Type"];
                Assert.AreEqual(@"application/x-zip-compressed", contentType);
            }
            else Assert.IsNotNull(result);
        }

        [Test]
        [ExpectedException(typeof(FaultException))]
        public void GetSecureDownloadTokenNegative() {
            var result = Proxy.TryCall<CDownloadToken>(delegate(CClientServiceClient svc) {
                return svc.GetSecureDownloadToken(_tokenFileId, Proxy.UserID, Proxy.IsPremium, 0);
            });
        }
    }
}
