﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Data.SqlClient;
using System.Configuration;
using System.Data;

namespace VoiceSubscriptionMiner.Models
{
	class VoiceSubInvoiceData
	{
		public int Month, Year;

		public long ServiceID { get; set; }
		public Dictionary<int, int> ChannelsByDay { get; set; }
		public string FirstChannelDay
		{
			get
			{
				return DateTime.Parse(string.Format("{0}/{1}/{2}", Month, ChannelsByDay.First(p => p.Value > 0).Key, Year)).ToShortDateString();
			}
		}
		public string LastChannelDay
		{
			get
			{
				if (ChannelsByDay.Last().Value == 0)
				{
					return DateTime.Parse(string.Format("{0}/{1}/{2}", Month, ChannelsByDay.First(p => p.Value == 0 && p.Key > ChannelsByDay.First(r => r.Value > 0).Key).Key, Year)).ToShortDateString();
				}
				else return DateTime.Parse(string.Format("{0}/{1}/{2}", Month, DateTime.DaysInMonth(Year, Month), Year)).ToShortDateString();
			}
		}

		public List<SubscriptionData> SubscriptionData { get; set; }
		public int SubscriptionCount
		{
			get
			{
				if (SubscriptionData != null)
				{
					return SubscriptionData.Count;
				}
				else return 0;
			}
		}
		public SubscriptionData ActiveSubsciption
		{
			get
			{
				if (SubscriptionData == null)
				{
					return null;
				}

				return SubscriptionData.FirstOrDefault(p => p.Status == SubscriptionStatus.Active);
			}
		}
		public SubscriptionData LastActiveSubscription
		{
			get
			{
				if (SubscriptionData == null)
				{
					return null;
				}

				var inactiveSubs = SubscriptionData.Where(p => p.Status != SubscriptionStatus.Active);

				SubscriptionData lastSub = null;
				foreach (var sub in inactiveSubs)
				{
					if (lastSub == null)
					{
						lastSub = sub;
					}
					else
					{
						if (sub.DateStarted > lastSub.DateStarted)
						{
							lastSub = sub;
						}
					}
				}

				return lastSub;
			}
		}
		public bool HasActiveSubscription
		{
			get
			{
				return (ActiveSubsciption != null);
			}
		}
		public string ActiveSubDateStarted
		{
			get
			{
				if (ActiveSubsciption == null)
				{
					return string.Empty;
				}

				return ActiveSubsciption.DateStarted.ToShortDateString();
			}
		}
		public string ActiveSubDateExpires
		{
			get
			{
				if (ActiveSubsciption == null)
				{
					return string.Empty;
				}

				return ActiveSubsciption.DateExpires.ToShortDateString();
			}
		}
		public string LastActiveDateStarted
		{
			get
			{
				if (LastActiveSubscription == null)
				{
					return string.Empty;
				}

				return LastActiveSubscription.DateStarted.ToShortDateString();
			}
		}
		public string LastActiveDateExpired
		{
			get
			{
				if (LastActiveSubscription == null)
				{
					return string.Empty;
				}

				return LastActiveSubscription.DateExpires.ToShortDateString();
			}
		}
		public bool DoChannelsMatch
		{
			get
			{
				foreach (var day in ChannelsByDay)
				{
					var sub = GetSubByDay(day.Key);
					if (sub == null)
					{
						if (day.Value != 0)
							return false;
					}
					else if (sub.Channels != day.Value)
					{
						return false;
					}
				}

				return true;
			}
		}
		public Dictionary<int, int> SubChannelsByDay
		{
			get
			{
				var subChannelByDay = new Dictionary<int, int>();

				foreach (var day in ChannelsByDay)
				{
					var sub = GetSubByDay(day.Key);

					if (sub == null)
					{
						subChannelByDay.Add(day.Key, 0);
					}
					else subChannelByDay.Add(day.Key, sub.Channels);
				}

				return subChannelByDay;
			}
		}

		public List<AccountEntitlementData> Entitlements { get; set; }
		public bool HasActiveEntitlement
		{
			get
			{
				if (Entitlements != null)
				{
					return (Entitlements.Count(p => p.IsActive) > 0);
				}
				else return false;
			}
		}

		public VoiceSubInvoiceData() { }
		public VoiceSubInvoiceData(string inputData, string month, string year)
		{
			Month = Int32.Parse(month);
			Year = Int32.Parse(year);

			string[] data = inputData.Split('\t');

			ServiceID = Int32.Parse(data[0]);
			ChannelsByDay = new Dictionary<int, int>();
			for (int i = 1; i <= (data.Length - 1); i++)
			{
				ChannelsByDay.Add(i, Int32.Parse(data[i]));
			}
		}

		private SubscriptionData GetSubByDay(int day)
		{
			if (SubscriptionData == null)
			{
				return null;
			}

			var date = DateTime.Parse(string.Format("{0}/{1}/{2}", Month, day, Year));
			var sub = SubscriptionData.FirstOrDefault(p => p.DateStarted <= date && p.DateExpires > date);
			return sub;
		}

		#region Static Methods
		public static VoiceSubInvoiceDataCollection ParseInvoiceFile(string fileName)
		{
			var voiceSubs = new VoiceSubInvoiceDataCollection();

			var sr = new StreamReader(fileName);
			var lines = sr.ReadToEnd().Replace("\r", "").Split('\n');
			sr.Close();

			var month = lines[0].Split('\t')[0];
			var year = lines[0].Split('\t')[1];

			for (int i = 1; i < lines.Length; i++)
			{
				var sub = new VoiceSubInvoiceData(lines[i], month, year);
				voiceSubs.Add(sub);
			}

			PopulateSubscriptions(voiceSubs);
			PopulateEntitlements(voiceSubs);
			return voiceSubs;
		}
		public static void GenerateOverview(VoiceSubInvoiceDataCollection voiceSubs, string outputFileName)
		{
			var sw = new StreamWriter(outputFileName);
			#region Lightspeed
			var thierChannelCount = voiceSubs.Sum(p => p.ChannelsByDay.Select(v => v.Value).Sum());

			sw.WriteLine("Lightspeed Totals");
			sw.WriteLine("Service Count: " + voiceSubs.Count);
			sw.WriteLine("Channels Billed: " + thierChannelCount);
			sw.WriteLine();
			#endregion

			#region Curse
			var servicesWithSubs = voiceSubs.Where(p => p.SubscriptionData != null);
			var ourChannelCount = servicesWithSubs.Sum(p => p.SubChannelsByDay.Select(v => v.Value).Sum());

			sw.WriteLine("Curse Totals");
			sw.WriteLine("Service Count: " + servicesWithSubs.Count());
			sw.WriteLine("Active Entitlement Count: " + servicesWithSubs.Where(p => p.HasActiveEntitlement == true).Count());
			sw.WriteLine("Channels Used: " + ourChannelCount);
			sw.WriteLine();
			#endregion

			#region Differences
			sw.WriteLine("Discrepancies");
			sw.WriteLine("Services: " + (voiceSubs.Count - servicesWithSubs.Count()));
			sw.WriteLine("Channels: " + (thierChannelCount - ourChannelCount));
			sw.WriteLine();
			#endregion
			sw.Close();
		}
		public static void GenerateServiceWithoutSubscriptionReport(VoiceSubInvoiceDataCollection voiceSubs, string outputFileName)
		{
			var servicesWithoutSubs = voiceSubs.Where(p => p.SubscriptionData == null);
			var sw = new TabWriter(outputFileName);
			#region Header
			sw.Write("Service ID");
			sw.Write("Daily Channels");
			sw.Write("Total Channels");
			sw.EndLine();
			#endregion

			#region Body
			foreach (var service in servicesWithoutSubs)
			{
				sw.Write(service.ServiceID);
				sw.Write(service.ChannelsByDay[1]);
				sw.Write(service.ChannelsByDay.Select(p => p.Value).Sum());
				sw.EndLine();
			}
			#endregion

			#region Footer
			sw.Write("Total: " + servicesWithoutSubs.Count());
			sw.Write(servicesWithoutSubs.Select(p => p.ChannelsByDay[1]).Sum());
			sw.Write(servicesWithoutSubs.Sum(p => p.ChannelsByDay.Select(v => v.Value).Sum()));
			#endregion
			sw.Close();
		}
		public static void GenerateChannelsByDayOverview(VoiceSubInvoiceDataCollection voiceSubs, string outputFileName)
		{
			var sw = new TabWriter(outputFileName);

			#region Header
			sw.Write("Service ID");
			sw.Write("Used Channels");
			sw.Write("Billed Channels");
			sw.Write("Delta");
			sw.EndLine();
			#endregion

			#region Body
			int usedTotal = 0, billedTotal = 0;
			var subs = voiceSubs.Where(p => p.DoChannelsMatch == false && p.SubscriptionData != null);
			foreach (var sub in subs)
			{
				sw.Write(sub.ServiceID);

				var usedCount = sub.SubChannelsByDay.Select(p => p.Value).Sum();
				var billedCount = sub.ChannelsByDay.Select(p => p.Value).Sum();
				sw.Write(usedCount);
				sw.Write(billedCount);
				sw.Write(billedCount - usedCount);
				sw.EndLine();

				usedTotal += usedCount;
				billedTotal += billedCount;
			}
			#endregion

			#region Footer
			sw.Write("Totals:");
			sw.Write(usedTotal);
			sw.Write(billedTotal);
			sw.Write(billedTotal - usedTotal);
			#endregion

			sw.Close();
		}

		public static void GenerateValidationReport(VoiceSubInvoiceDataCollection voiceSubs, string outputFileName)
		{
			var sw = new TabWriter(outputFileName);
			#region Header
			sw.Write("Service ID");
			sw.Write("Num Subs");
			sw.Write("Active Subs");
			sw.Write("Active Entitlement");

			sw.Write("FirstChannelDay");
			sw.Write("LastChannelDay");

			sw.Write("DateStarted");
			sw.Write("DateExpires");

			sw.Write("LastDateStarted");
			sw.Write("LastDateExpired");

			sw.Write("ChannelsMatch");
			sw.EndLine();
			#endregion

			foreach (var sub in voiceSubs)
			{
				sw.Write(sub.ServiceID);
				sw.Write(sub.SubscriptionCount);
				sw.Write(sub.HasActiveSubscription);
				sw.Write(sub.HasActiveEntitlement);

				sw.Write(sub.FirstChannelDay);
				sw.Write(sub.LastChannelDay);

				sw.Write(sub.ActiveSubDateStarted);
				sw.Write(sub.ActiveSubDateExpires);

				sw.Write(sub.LastActiveDateStarted);
				sw.Write(sub.LastActiveDateExpired);

				sw.Write(sub.DoChannelsMatch);

				sw.EndLine();
			}
			sw.Close();
		}
		public static void GenerateChannelsByDayReport(VoiceSubInvoiceDataCollection voiceSubs, string outputFileName)
		{
			var sw = new TabWriter(outputFileName);

			#region Header
			var daysInMonth = DateTime.DaysInMonth(voiceSubs.First(p => p != null).Year, voiceSubs.First(p => p != null).Month);

			sw.Write(string.Empty);
			for (int i = 1; i <= daysInMonth; i++)
			{
				sw.Write("Day " + i);
				sw.Write(string.Empty);
			}
			sw.EndLine();

			sw.Write("Service ID");
			for (int i = 1; i <= daysInMonth; i++)
			{
				sw.Write("Them");
				sw.Write("Us");
			}
			sw.EndLine();
			#endregion

			#region Body
			var subs = voiceSubs.Where(p => p.DoChannelsMatch == false);
			foreach (var sub in subs)
			{
				sw.Write(sub.ServiceID);

				var subChannelsByDay = sub.SubChannelsByDay;
				for (int i = 1; i <= DateTime.DaysInMonth(sub.Year, sub.Month); i++)
				{
					sw.Write(sub.ChannelsByDay[i]);
					sw.Write(subChannelsByDay[i]);
				}
				sw.EndLine();
			}
			#endregion

			sw.Close();
		}
		public static void GenerateMissingSubscriptionReport(VoiceSubInvoiceDataCollection voiceSubs, string outputFileName)
		{
			using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["BillingService"].ConnectionString))
			{
				var cmd = new SqlCommand("SELECT ID FROM AccountVoiceService WHERE ServiceID = @ServiceID", conn);
				cmd.Parameters.Add("ServiceID", SqlDbType.Int);

				if (conn.State != ConnectionState.Open)
				{
					conn.Open();
				}

				var sw = new StreamWriter(outputFileName);
				foreach (var sub in voiceSubs)
				{
					cmd.Parameters["ServiceID"].Value = sub.ServiceID;
					Console.WriteLine(sub.ServiceID);

					if (cmd.ExecuteScalar() == null)
					{
						sw.WriteLine(sub.ServiceID.ToString().PadLeft(10, '0'));
					}
				}
				sw.Close();
			}
		}
		public static void GenerateTotalChannelReport(VoiceSubInvoiceDataCollection voiceSubs, string outputFileName)
		{
			var sw = new TabWriter(outputFileName);

			#region Header
			var daysInMonth = DateTime.DaysInMonth(voiceSubs.First(p => p != null).Year, voiceSubs.First(p => p != null).Month);

			sw.Write(string.Empty);
			for (int i = 1; i <= daysInMonth; i++)
			{
				sw.Write("Day " + i);
			}
			sw.EndLine();
			#endregion

			#region Body
			List<CountData> countData = new List<CountData>();
			for (int i = 1; i <= daysInMonth; i++)
			{
				var count = new CountData
				{
					Day = i,
					TheirCount = voiceSubs.Sum(p => p.ChannelsByDay[i]),
					OurCount = voiceSubs.Sum(p => p.SubChannelsByDay[i])

				};
				countData.Add(count);
			}

			for (int i = 0; i < 3; i++)
			{
				#region left header
				if (i == 0)
				{
					sw.Write("Their Count");
				}
				else if (i == 1)
				{
					sw.Write("Our Count");
				}
				else sw.Write("Difference");
				#endregion

				for (int x = 0; x < countData.Count; x++)
				{
					if (i == 0)
					{
						sw.Write(countData[x].TheirCount);
					}
					else if (i == 1)
					{
						sw.Write(countData[x].OurCount);
					}
					else sw.Write(countData[x].Difference);
				}
				sw.EndLine();
			}
			#endregion

			sw.Close();
		}
		#endregion

		class CountData
		{
			public int Day { get; set; }
			public int TheirCount { get; set; }
			public int OurCount { get; set; }
			public int Difference
			{
				get
				{
					return TheirCount - OurCount;
				}
			}
		}

		private static void PopulateSubscriptions(VoiceSubInvoiceDataCollection voiceSubs)
		{
			#region Query
			var query = "SELECT " +
							"VoiceService.ServiceID, " +
							"Subscription.ProductID, " +
							"Subscription.BillingPlanID, " +
							"CASE " +
								"WHEN Subscription.BillingPlanID LIKE 'GHV%' " +
									"THEN SUBSTRING(Subscription.BillingPlanID, 4, PATINDEX('%-%', Subscription.BillingPlanID) - 4) " +
								"WHEN Subscription.BillingPlanID LIKE 'VH%' " +
									"THEN SUBSTRING(Subscription.BillingPlanID, 3, PATINDEX('%-%', Subscription.BillingPlanID) - 3) " +
								"ELSE 0 " +
							"END AS Channels, " +
							"CASE " +
								"WHEN Subscription.DateStarted > @MonthStart " +
									"THEN DATEPART(DD, Subscription.DateStarted) " +
								"WHEN Subscription.DateStarted > @MonthEnd " +
									"THEN 31 " +
								"ELSE 1 " +
							"END AS StartDay, " +
							"CASE " +
								"WHEN Subscription.DateExpires < @MonthEnd " +
									"THEN DATEPART(DD, Subscription.DateExpires) " +
								"WHEN Subscription.DateExpires < @MonthStart " +
									"THEN 31 " +
								"ELSE 30 " +
							"END AS EndDay, " +
							"Subscription.DateStarted, " +
							"Subscription.DateExpires, " +
							"Subscription.[Status] " +
						"FROM BillingService.dbo.AccountVoiceService AS VoiceService " +
							"INNER JOIN BillingService.dbo.AccountVoiceClient AS VoiceClient ON VoiceClient.ID = VoiceService.AccountVoiceClientID " +
							"INNER JOIN BillingService.dbo.Subscription AS Subscription ON Subscription.AccountID = VoiceClient.AccountID " +
						"WHERE Subscription.ProductID IN ('GHV', 'VH') " +
							"AND VoiceService.ServiceID IN ({0}) " +
							"AND Subscription.DateStarted < @MonthEnd";
			//"AND (Subscription.[Status] = 0 OR (Subscription.[Status] = 1 AND Subscription.DateExpires > @MonthStart))" +
			//"AND Subscription.DateExpires > @MonthStart";
			#endregion

			using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["BillingService"].ConnectionString))
			{
				var cmd = new SqlCommand(string.Format(query, string.Join(",", voiceSubs.ServiceIDs)), conn);
				cmd.Parameters.Add("MonthStart", SqlDbType.DateTime).Value = DateTime.Parse(voiceSubs.First(p => p != null).FirstChannelDay);
				cmd.Parameters.Add("MonthEnd", SqlDbType.DateTime).Value = DateTime.Parse(voiceSubs.First(p => p != null).LastChannelDay);

				if (conn.State != ConnectionState.Open)
				{
					conn.Open();
				}

				using (var reader = cmd.ExecuteReader())
				{
					while (reader.Read())
					{
						var serviceId = reader.GetInt64(0);

						if (voiceSubs.ContainsServiceID(serviceId))
						{
							if (voiceSubs[serviceId].SubscriptionData == null)
							{
								voiceSubs[serviceId].SubscriptionData = new List<SubscriptionData>();
							}

							voiceSubs[serviceId].SubscriptionData.Add(new SubscriptionData(reader));
						}
					}
				}
			}
		}
		private static void PopulateEntitlements(VoiceSubInvoiceDataCollection voiceSubs)
		{
			var entitlements = AccountEntitlementData.GetAllEntitlements();
			foreach (var voiceSub in voiceSubs)
			{
				voiceSub.Entitlements = entitlements.Where(p => p.ServiceID == voiceSub.ServiceID).ToList();
			}
		}
	}
}
