﻿using Curse;
using Curse.WAR;
using System;
using System.Data.SqlClient;
using System.Collections.Generic;
using System.Data;
using System.Linq;
namespace WARDataService
{
    public static partial class DBItems
    {
        private const Byte NUM_BONUSES = 10;
        private const Byte NUM_CRAFTING_BONUSES = 25;
        private const Byte NUM_ALT_CURRENCY = 3;

        private static HashSet<CustomKey> sKnownItems =
            new HashSet<CustomKey>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, Int32> sItemNameToId = new Dictionary<CustomKey, Int32>(new CustomKey.CustomKeyComparer());                

        private static Dictionary<CustomKey, DateTime> sHistory =
            new Dictionary<CustomKey, DateTime>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, Confirmation> sConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());

        private static String sUpdateQuery = null;

        private static String[] sUpdateIgnoredColumns = { "id", "version_id", "posted", "posted_id" };
        private static Dictionary<string, string> sUpdateCustomColumns = new Dictionary<string, string>();

        private static DataTable sTableSchema = null;

        public static Int32 GetItemId(String pItemName, String pLocale)
        {
            return DB.GetIdFromName(sItemNameToId, pItemName, pLocale);
        }

        public static void AddItemId(String pItemName, String pLocale, Int32 pItemId)
        {
            DB.AddNameFromId(sItemNameToId, pLocale, pItemName, pItemId);
        }

        public static void Initialize()
        {
            sKnownItems.Clear();
            sHistory.Clear();
            sItemNameToId.Clear();       

            foreach (Int32 version in Config.Instance.Versions)
            {                                
                DB.PopulateId(sKnownItems, "item", version);
            }

            foreach (String locale in DB.sLocalizations)
            {
                DB.PopulateNameToID(sItemNameToId, "item", locale);
            }

            DB.LoadSchema(ref sTableSchema, "item");

            sUpdateCustomColumns.Add("is_quest_item", "item.is_quest_item=dbo.Highest(item.is_quest_item,tmp.is_quest_item)");
            sUpdateCustomColumns.Add("buy_price", "item.buy_price=dbo.Highest(item.buy_price,tmp.buy_price)");

            for (Byte i = 0; i < NUM_ALT_CURRENCY; i++)
            {
                string iconColName = "alt_currency_icon" + i;
                string amountColName = "alt_currency_amount" + i;
                sUpdateCustomColumns.Add(iconColName, "item." + iconColName + " = COALESCE(tmp." + iconColName + ",tmp." + iconColName + ",item." + iconColName + ")");
                sUpdateCustomColumns.Add(amountColName, "item." + amountColName + " = COALESCE(tmp." + amountColName + ",tmp." + amountColName + ",item." + amountColName + ")");
            }

            sUpdateQuery = "update item set ";
            sUpdateQuery = sUpdateQuery + DB.GetUpdateSQL(sTableSchema, sUpdateIgnoredColumns, sUpdateCustomColumns);
            sUpdateQuery = sUpdateQuery + " FROM item,#item tmp" +
                " WHERE item.id=tmp.id AND item.version_id=tmp.version_id;";

        }

        public static void Save(Update pUpdate, SqlConnection pConn)
        {

            if (pUpdate.Items.Count == 0)
            {
                return;
            }
            
            DataRow dr = null;
            DataTable dtInsert = sTableSchema.Clone();
            DataTable dtUpdate = sTableSchema.Clone();

            dtInsert.BeginLoadData();
            dtUpdate.BeginLoadData();

            CustomKey key;
            String locale = pUpdate.Language.ToString();           

            bool isNewRow = false;
            foreach (Item item in pUpdate.Items)
            {
                if (!DB.ReadyForDB(sHistory,
                                sConfirmations,
                                item.Id,
                                Config.Instance.ItemExpiration,
                                Config.Instance.MinConfirmations,
                                pUpdate.UserId,
                                pUpdate.ClientVersion.Value,
                                (ELocale)pUpdate.Language,
                                pUpdate.IsTrustedUser))
                {
                    continue;
                }

                key = new CustomKey(pUpdate.ClientVersion.Value,
                                    item.Id);

                isNewRow = !sKnownItems.Contains(key);
                if (isNewRow)
                {
                    dr = dtInsert.NewRow();
                    AddItemId(item.Name, locale, item.Id); 
                }
                else
                {
                    dr = dtUpdate.NewRow();
                }

                dr["id"] = item.Id;
                dr["display_id"] = item.DisplayId;
                dr["version_id"] = pUpdate.ClientVersion.Value;
                dr["name_" + locale] = item.Name;
                dr["description_" + locale] = item.Description;
                dr["type"] = item.Type;
                dr["level"] = item.Level;
                dr["ilevel"] = item.ILevel;
                dr["renown"] = item.Renown;
                dr["rarity"] = item.Rarity;
                dr["sell_price"] = item.SellPrice;              
                dr["buy_price"] = item.BuyPrice;              
                dr["itemset"] = item.ItemSet;
                dr["binding"] = item.Binding;
                dr["stack_count"] = item.StackCount;
                dr["equip_slot"] = item.EquipSlot;
                dr["enchant_slots"] = item.EnchantSlots;
                dr["charges"] = item.Charges;
                dr["cultivation_type"] = item.CultivationType;
                dr["trophy_index"] = item.TrophyIndex;
                dr["trophy_location"] = item.TrophyLocation;
                dr["required_crafting_skill"] = item.CraftingSkillRequired;
                dr["speed"] = item.Speed;
                dr["block_rating"] = item.BlockRating;
                dr["icon"] = item.Icon;
                dr["armor"] = item.Armor;
                dr["dps"] = item.DPS;
                dr["is_quest_item"] = item.IsQuestItem;
                
                // Flags
                Int32 flagBits = 0;
                foreach (ItemFlag flag in item.Flags)
                {
                    if(flag.Value == 1)
                    {
                        Utility.AddBitValue(ref flagBits, flag.Id);                            
                    }
                }
                dr["flags"] = flagBits;


                Utility.SetBitField(item.RequiredSkills, dr, "required_skills");
                Utility.SetBitField(item.RequiredRaces, dr, "required_races");
                Utility.SetBitField(item.RequiredCareers, dr, "required_careers");

                //Bonuses
                for (Int32 index = 0; index < NUM_BONUSES; ++index)                    
                {
                    if (index < item.Bonuses.Count)
                    {
                        dr["bonus_id"+index] = item.Bonuses[index].Id;
                        dr["bonus_type"+index] = item.Bonuses[index].Type;
                        dr["bonus_value"+index] = item.Bonuses[index].Value;
                        dr["bonus_cooldown"+index] = item.Bonuses[index].TotalCooldownTime;
                    }
                    else
                    {
                        dr["bonus_id" + index] = 0;
                        dr["bonus_type" + index] = 0;
                        dr["bonus_value" + index] = 0;
                        dr["bonus_cooldown" + index] = 0;
                    }
                }

                //Crafting Bonuses
                for (Int32 index = 0; index < NUM_CRAFTING_BONUSES; ++index)
                {
                    if (index < item.CraftingBonuses.Count)
                    {
                        dr["crafting_bonus_id" + index] = item.CraftingBonuses[index].Id;
                        dr["crafting_bonus_type" + index] = item.CraftingBonuses[index].Type;
                        dr["crafting_bonus_value" + index] = item.CraftingBonuses[index].Value;                            
                    }
                    else
                    {
                        dr["crafting_bonus_id" + index] = 0;
                        dr["crafting_bonus_type" + index] = 0;
                        dr["crafting_bonus_value" + index] = 0;
                    }
                }

                item.AltCurrency.Sort(delegate(ItemAltCurrency x, ItemAltCurrency y)
                {
                    return x.DisplayOrder.CompareTo(y.DisplayOrder);
                });

                // Alternate Currency
                for (Byte index = 0; index < NUM_ALT_CURRENCY; index++)
                {
                    if (index < item.AltCurrency.Count)
                    {
                        dr["alt_currency_icon" + index] = item.AltCurrency[index].Icon;
                        dr["alt_currency_amount" + index] = item.AltCurrency[index].Amount;
                     
                    }
                    else
                    {
                        dr["alt_currency_icon" + index] = System.DBNull.Value;
                        dr["alt_currency_amount" + index] = System.DBNull.Value;                     
                    }

                }
                                                                                         
                // Meta Data
                dr["updated_id"] = pUpdate.UserId;
                dr["posted_id"] = pUpdate.UserId;
                dr["posted"] = DateTime.UtcNow;
                dr["updated"] = DateTime.UtcNow;

                if (isNewRow)
                {                        
                    dtInsert.Rows.Add(dr);
                    sKnownItems.Add(key);
                }
                else
                {
                    dtUpdate.Rows.Add(dr);
                }
                
            }

            // New Items
            if (dtInsert.Rows.Count > 0)
            {
                using (SqlBulkCopy bulk = new SqlBulkCopy(pConn,
                                                    SqlBulkCopyOptions.UseInternalTransaction | SqlBulkCopyOptions.TableLock,
                                                    null))
                {
                    bulk.BatchSize = 5000;
                    bulk.DestinationTableName = "item";
                    bulk.WriteToServer(dtInsert);
                    bulk.Close();
                }
                
            }

            // Item Updates
            if (dtUpdate.Rows.Count > 0)
            {
                SqlCommand cmd = pConn.CreateCommand();
                cmd.CommandText = "SELECT TOP 0 * INTO #item FROM item;";
                cmd.ExecuteNonQuery();
                using (SqlBulkCopy bulk = new SqlBulkCopy(pConn,
                                                    SqlBulkCopyOptions.UseInternalTransaction | SqlBulkCopyOptions.TableLock,
                                                    null))
                {
                    bulk.BatchSize = 5000;
                    bulk.DestinationTableName = "#item";
                    bulk.WriteToServer(dtUpdate);
                    bulk.Close();
                }                               

                cmd.CommandText = String.Format(sUpdateQuery, locale);
                cmd.ExecuteNonQuery();
                cmd.CommandText = "DROP TABLE #item;";
                cmd.ExecuteNonQuery();
            }
        }
            
    }

}

