﻿using Curse;
using Curse.WAR;
using System;
using System.Text;
using System.Data.SqlClient;
using System.Collections.Generic;
using System.Data;
using System.Linq;
namespace WARDataService
{
    public static partial class DBPublicQuests
    {

        private const Byte NUM_CONDITIONS = 6;

        private static HashSet<CustomKey> sKnownPublicQuests =
            new HashSet<CustomKey>(new CustomKey.CustomKeyComparer());

        private static HashSet<CustomKey> sKnownStages =
            new HashSet<CustomKey>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, DateTime> sPublicQuestLootExpirations =
            new Dictionary<CustomKey, DateTime>(new CustomKey.CustomKeyComparer());
       
        private static Dictionary<CustomKey, DateTime> sHistory =
            new Dictionary<CustomKey, DateTime>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, Confirmation> sConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());

        private static String sPublicQuestLootQueryFormat = "UPDATE public_quest_loot SET " +                                                   
                                                   "reward_total=reward_total+1," +
                                                   "careers=careers|{3}," +
                                                   "updated_id={4}," +
                                                   "updated=GETUTCDATE()" +
                                                   " WHERE public_quest_id={0} AND container_item_id={1} and reward_item_id={2}" +
                                                   " IF @@ROWCOUNT = 0 " +
                                                   "INSERT INTO public_quest_loot(public_quest_id," +
                                                   "container_item_id,reward_item_id," +
                                                   "container_total,reward_total,careers,posted_id,updated_id) VALUES(" +
                                                   "{0},{1},{2},0,1,{3},{4},{4});";

        private static String sUpdateQuery = null;
        private static String sStageUpdateQuery = null;        

        private static String[] sUpdateIgnoredColumns = { "id", "version_id", "posted", "posted_id" };
        private static String[] sInsertIgnoredColumns = {  };

        private static Dictionary<String, String> sUpdateCustomColumns = new Dictionary<string, string>();
        private static Dictionary<String, String> sStageUpdateColumns = new Dictionary<string, string>();

        private static DataTable sTableSchema = null;
        private static DataTable sStageSchema = null;

        public static void Initialize()
        {
            sKnownPublicQuests.Clear();
            sKnownStages.Clear();
            sHistory.Clear();
            sUpdateCustomColumns.Clear();
            

            foreach (Int32 version in Config.Instance.Versions)
            {             
                DB.PopulateId(sKnownPublicQuests, "public_quest", version);                
            }

            foreach (String locale in DB.sLocalizations)
            {
                PopulateKnownStages(locale);
            }
            
            DB.LoadSchema(ref sTableSchema, "public_quest");
            DB.LoadSchema(ref sStageSchema, "public_quest_stage");

            sUpdateCustomColumns.Add("given_to_races", "public_quest.given_to_races=public_quest.given_to_races|tmp.given_to_races");
            sUpdateCustomColumns.Add("influence_type", "public_quest.influence_type=DBO.Highest(public_quest.influence_type,tmp.influence_type)");
            sUpdateCustomColumns.Add("level", "public_quest.level=DBO.Lowest(public_quest.level,tmp.level)");

            sUpdateQuery = "update public_quest set ";
            sUpdateQuery = sUpdateQuery + DB.GetUpdateSQL(sTableSchema, sUpdateIgnoredColumns, sUpdateCustomColumns);
            sUpdateQuery = sUpdateQuery + " FROM public_quest,#public_quest tmp" +
                " WHERE public_quest.id=tmp.id AND public_quest.version_id=tmp.version_id;";

            for (int i = 0; i < NUM_CONDITIONS; i++)
            {
                sStageUpdateColumns.Add("condition_name" + i, "public_quest_stage.condition_name" + i + "=COALESCE(tmp.condition_name" + i + ",tmp.condition_name" + i + ",public_quest_stage.condition_name" + i + ")");
                sStageUpdateColumns.Add("condition_count" + i, "public_quest_stage.condition_count" + i + "=COALESCE(tmp.condition_count" + i + ",tmp.condition_count" + i + ",public_quest_stage.condition_count" + i + ")");
                sStageUpdateColumns.Add("condition_type" + i, "public_quest_stage.condition_type" + i + "=COALESCE(tmp.condition_type" + i + ",tmp.condition_type" + i + ",public_quest_stage.condition_type" + i + ")");
                sStageUpdateColumns.Add("condition_id" + i, "public_quest_stage.condition_id" + i + "=COALESCE(tmp.condition_id" + i + ",tmp.condition_id" + i + ",public_quest_stage.condition_id" + i + ")");
            }
            sStageUpdateColumns.Add("timer", "public_quest_stage.timer=DBO.Highest(public_quest_stage.timer,tmp.timer)");
            sStageUpdateColumns.Add("timer_name", "public_quest_stage.timer_name=COALESCE(tmp.timer_name,tmp.timer_name,public_quest_stage.timer_name)");
            
            sStageUpdateQuery = "update public_quest_stage set ";
            sStageUpdateQuery = sStageUpdateQuery + DB.GetUpdateSQL(sStageSchema, sUpdateIgnoredColumns, sStageUpdateColumns);
            sStageUpdateQuery = sStageUpdateQuery + " FROM public_quest_stage,#public_quest_stage tmp" +
                " WHERE public_quest_stage.public_quest_id=tmp.public_quest_id AND public_quest_stage.locale=tmp.locale and public_quest_stage.name = tmp.name;";

            

        }

        public static bool StageExists(String pStageName, Int32 pQuestId, String pLocale)
        {            
            return sKnownStages.Contains(new CustomKey(pStageName, pQuestId, pLocale));            
        }

        public static void AddStage(String pStageName, Int32 pQuestId, String pLocale)
        {
            CustomKey key = new CustomKey(pStageName, pQuestId, pLocale);
            sKnownStages.Add(key);
        }

        public static void PopulateKnownStages(String pLocale)
        {
            String query = "select name, public_quest_id from public_quest_stage where locale='" + pLocale + "'";
                      
            Int32 count = 0;
            using (SqlConnection conn = new SqlConnection(Config.Instance.ConnectionString))
            {
                conn.Open();

                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = query;
                using (SqlDataReader dr = cmd.ExecuteReader())
                {                    
                    while (dr.Read())
                    {
                        AddStage((String)dr[0], (Int32)dr[1], pLocale);
                        ++count;
                    }
                }
            }          
        }

        private static void SetPublicQuestLootQuery(Int32 pPublicQuestId, Update pUpdate, PackableList<PublicQuestLoot> pPublicQuestLoot, StringBuilder pBuilder)
        {

            Int32 playerCareerBits = Utility.GetBitValue(pUpdate.Players[0].Career);

            foreach (PublicQuestLoot loot in pPublicQuestLoot)
            {
                foreach (Int32 rewardItemId in loot.RewardItems)
                {
                    pBuilder.AppendFormat(sPublicQuestLootQueryFormat,
                    pPublicQuestId,
                    loot.ContainerItemId,
                    rewardItemId,
                    playerCareerBits,                    
                    pUpdate.UserId);
                }
                pBuilder.AppendFormat("update public_quest_loot set container_total=container_total+1 where public_quest_id={0} and container_item_id={1};", pPublicQuestId, loot.ContainerItemId);
            }

        }

        public static void Save(Update pUpdate, SqlConnection pConn)
        {

            if (pUpdate.PublicQuests.Count == 0)
            {
                return;
            }
            
            DataRow dr = null;
            DataRow drStage = null;

            DataTable dtInsert = sTableSchema.Clone();
            DataTable dtUpdate = sTableSchema.Clone();
            DataTable dtStageInsert = sStageSchema.Clone();
            DataTable dtStageUpdate = sStageSchema.Clone();

            dtInsert.BeginLoadData();
            dtUpdate.BeginLoadData();
            dtStageUpdate.BeginLoadData();

            CustomKey key;
            CustomKey rewardKey;
            String locale = pUpdate.Language.ToString();

            bool isNewRow = false;
            StringBuilder sql = new StringBuilder();
            foreach (PublicQuest publicQuest in pUpdate.PublicQuests)
            {

                // Expirations keyed on ID and Career:
                rewardKey = new CustomKey(publicQuest.Id, pUpdate.Players[0].Career, pUpdate.ClientVersion.Value);
                if (!DB.IsEntityExpired(sPublicQuestLootExpirations, rewardKey, Config.Instance.QuestExpiration, pUpdate.IsTrustedUser))
                {
                    continue;
                }

                // Save Loot
                SetPublicQuestLootQuery(publicQuest.Id, pUpdate, publicQuest.Loot, sql);
                if (publicQuest.Loot.Count > 0)
                {
                    sql.AppendFormat("update public_quest set updated = GetUtcDate() where id = {0};", publicQuest.Id);
                }
            }

            foreach (PublicQuest publicQuest in pUpdate.PublicQuests)
            {

                if (!DB.ReadyForDB(sHistory,
                                sConfirmations,
                                publicQuest.Id,
                                Config.Instance.QuestExpiration,
                                Config.Instance.MinConfirmations,
                                pUpdate.UserId,
                                pUpdate.ClientVersion.Value,
                                (ELocale)pUpdate.Language,
                                pUpdate.IsTrustedUser))
                {
                    continue;
                }
                

                key = new CustomKey(pUpdate.ClientVersion.Value,
                                    publicQuest.Id);

                isNewRow = !sKnownPublicQuests.Contains(key);
                if (isNewRow)
                {
                    dr = dtInsert.NewRow();
                }
                else
                {
                    dr = dtUpdate.NewRow();
                }

                dr["id"] = publicQuest.Id;
                dr["version_id"] = pUpdate.ClientVersion.Value;
                dr["name_" + locale] = publicQuest.Name.Value;
                dr["zone_id"] = publicQuest.ZoneId;
                dr["is_capture_point"] = publicQuest.IsCapturePoint;
                if (publicQuest.Stages.Count > 0)
                {
                    dr["influence_type"] = publicQuest.Stages[0].InfluenceType;
                }
                else
                {
                    dr["influence_type"] = -1;
                }
                Int32 raceBits = 0;
                Utility.AddBitValue(ref raceBits, pUpdate.Players[0].Race);                
                dr["given_to_races"] = raceBits;
                dr["level"] = pUpdate.Players[0].Level;

                // Meta Data
                dr["updated_id"] = pUpdate.UserId;
                dr["posted_id"] = pUpdate.UserId;
                dr["posted"] = DateTime.UtcNow;
                dr["updated"] = DateTime.UtcNow;

                if (isNewRow)
                {
                    dtInsert.Rows.Add(dr);
                    sKnownPublicQuests.Add(key);
                }
                else
                {
                    dtUpdate.Rows.Add(dr);
                }



                List<String> handledStages = new List<string>();

                foreach (PublicQuestStage stage in publicQuest.Stages)
                {
                    if (handledStages.Contains(stage.Name))
                    {
                        continue;
                    }

                    handledStages.Add(stage.Name);

                    bool isNewStage = !StageExists(stage.Name, publicQuest.Id, locale);

                    if (isNewStage)
                    {
                        drStage = dtStageInsert.NewRow();
                    }
                    else
                    {
                        drStage = dtStageUpdate.NewRow();
                    }
                    
                    bool setConditions = !stage.Conditions.Exists(delegate(QuestCondition x) { return (x.Name.Length == 0); });


                    stage.Conditions.Sort(delegate(QuestCondition x, QuestCondition y)
                    {
                        return x.Name.CompareTo(y.Name);
                    });

                    //Conditions
                    if (isNewStage || setConditions)
                    {
                        for (Int32 index = 0; index < NUM_CONDITIONS; ++index)
                        {
                            if (setConditions && index < stage.Conditions.Count)
                            {
                                drStage["condition_name" + index] = stage.Conditions[index].Name;
                                switch (stage.Conditions[index].ActionType)
                                {
                                    case (Byte)EQuestConditionType.CreatureKill:
                                        stage.Conditions[index].ActionId = DBNPCs.GetNpcId(stage.Conditions[index].ActionName.Value, locale);
                                        break;
                                    case (Byte)EQuestConditionType.CollectGameObject:
                                        stage.Conditions[index].ActionId = DBGameObjects.GetGameObjectId(stage.Conditions[index].ActionName.Value, locale);
                                        break;
                                }
                                if (stage.Conditions[index].ActionId <= 0)
                                {
                                    drStage["condition_id" + index] = System.DBNull.Value;
                                    drStage["condition_type" + index] = System.DBNull.Value;

                                }
                                else
                                {
                                    drStage["condition_id" + index] = stage.Conditions[index].ActionId;
                                    drStage["condition_type" + index] = stage.Conditions[index].ActionType;
                                }
                                Utility.SetNullIfZero(drStage, "condition_count" + index, stage.Conditions[index].Count);

                            }
                            else
                            {
                                drStage["condition_name" + index ] = System.DBNull.Value;
                                drStage["condition_count" + index] = System.DBNull.Value;
                                drStage["condition_type" + index] = System.DBNull.Value;
                                drStage["condition_id" + index] = System.DBNull.Value;
                            }
                        }
                    }

                  

                    drStage["public_quest_id"] = publicQuest.Id;
                    drStage["locale"] = locale;                    
                    drStage["timer"] = stage.MaxTimer;
                    drStage["influence_type"] = stage.InfluenceType;
                    drStage["name"] = stage.Name;
                    if (stage.TimerName.Length > 0)
                    {
                        drStage["timer_name"] = stage.TimerName;
                    }
                    drStage["description"] = stage.Description;

                    // Meta Data
                    drStage["updated_id"] = pUpdate.UserId;
                    drStage["posted_id"] = pUpdate.UserId;
                    drStage["posted"] = DateTime.UtcNow;
                    drStage["updated"] = DateTime.UtcNow;
                  
                    if (isNewStage)
                    {
                        dtStageInsert.Rows.Add(drStage);
                        AddStage(stage.Name, publicQuest.Id, locale);
                    }
                    else
                    {
                        dtStageUpdate.Rows.Add(drStage);
                    }
                }

            }

            SqlCommand cmd = pConn.CreateCommand();

            // New Stages:
            if (dtStageInsert.Rows.Count > 0)
            {
                using (SqlBulkCopy bulk = new SqlBulkCopy(pConn,
                                         SqlBulkCopyOptions.UseInternalTransaction | SqlBulkCopyOptions.TableLock,
                                         null))
                {
                    bulk.BatchSize = 5000;
                    bulk.DestinationTableName = "public_quest_stage";
                    bulk.WriteToServer(dtStageInsert);                    
                }
                
            }

            // Updated Stages:
            if (dtStageUpdate.Rows.Count > 0)
            {
                cmd.CommandText = "SELECT TOP 0 * INTO #public_quest_stage FROM public_quest_stage;";
                cmd.ExecuteNonQuery();
                using (SqlBulkCopy bulkStageUpdate = new SqlBulkCopy(pConn,
                                                     SqlBulkCopyOptions.UseInternalTransaction | SqlBulkCopyOptions.TableLock,
                                                     null))
                {
                    bulkStageUpdate.BatchSize = 5000;
                    bulkStageUpdate.DestinationTableName = "#public_quest_stage";
                    bulkStageUpdate.WriteToServer(dtStageUpdate);                
                }

                
                cmd.CommandText = String.Format(sStageUpdateQuery, locale);
                cmd.ExecuteNonQuery();
                cmd.CommandText = "DROP TABLE #public_quest_stage;";
                cmd.ExecuteNonQuery();
            }

            // Loot:
            if (sql.Length > 0)
            {
                cmd.CommandText = sql.ToString();
                cmd.ExecuteNonQuery();
            }

            // New Public Quests
            if (dtInsert.Rows.Count > 0)
            {
                using (SqlBulkCopy bulkInsert = new SqlBulkCopy(pConn,
                                                  SqlBulkCopyOptions.UseInternalTransaction | SqlBulkCopyOptions.TableLock,
                                                  null))
                {
                    bulkInsert.BatchSize = 5000;
                    bulkInsert.DestinationTableName = "public_quest";
                    bulkInsert.WriteToServer(dtInsert);                    
                }
                
            }

            // Public Quest Updates
            if (dtUpdate.Rows.Count > 0)
            {                
                cmd.CommandText = "SELECT TOP 0 * INTO #public_quest FROM public_quest;";
                cmd.ExecuteNonQuery();

                using (SqlBulkCopy bulk = new SqlBulkCopy(pConn,
                                                     SqlBulkCopyOptions.UseInternalTransaction | SqlBulkCopyOptions.TableLock,
                                                     null))
                {
                    bulk.BatchSize = 5000;
                    bulk.DestinationTableName = "#public_quest";
                    bulk.WriteToServer(dtUpdate);                    
                }
                
                cmd.CommandText = String.Format(sUpdateQuery, locale);
                cmd.ExecuteNonQuery();
                cmd.CommandText = "DROP TABLE #public_quest;";
                cmd.ExecuteNonQuery();
            }

           
        }
    }

}
