﻿using Curse;
using Curse.Auth;
using Curse.WAR;
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Data.SqlClient;
using System.Threading;
using System.Text;
using System.Web;
using System.Web.Services;
using System.Web.Services.Protocols;

namespace WARDataService
{
    /// <summary>
    /// Summary description for Service1
    /// </summary>
    [WebService(Namespace = "http://warservice.curse.com/")]
    [WebServiceBinding(ConformsTo = WsiProfiles.BasicProfile1_1)]
    [ToolboxItem(false)]
    public sealed class WARDataService : AuthenticatableWebService
    {        
        private static Queue<Update> sUpdateQueue = new Queue<Update>();
        private static Object sRebootLock = new Object();
        private static Thread sUpdateThread = null;
        private static Byte[] sDLL = null;
        private static String sDLLHash = null;        
        private static Dictionary<Int32, DateTime> sUpdateTimes = new Dictionary<Int32, DateTime>();
        private static Thread sServerUpdateThread = null;
        private static DateTime sLastServersUpdate = DateTime.MinValue;
        private static Int32 sGameServerUpdateInterval = 60000;
        private static Random rnd = new Random(DateTime.UtcNow.Month);
        private static Boolean sServiceStarted = false;
        
        static WARDataService()
        {
            Reload();
        }

          /**
         * Support method for reloading the state of the service
         * Reloads config file, log file, DLL, SQL and DBC records
         * Restarts the update thread and server update thread
         */
        private static void Reload()
        {
            sServiceStarted = false;
            lock (sRebootLock)
            {                
                if (sUpdateThread != null)
                {
                    sUpdateThread.Abort();
                    sUpdateThread.Join();
                    sUpdateThread = null;
                }
                if (sServerUpdateThread != null)
                {
                    sServerUpdateThread.Abort();
                    sServerUpdateThread.Join();
                    sServerUpdateThread = null;
                }

                sUpdateQueue.Clear();

                CPUMonitor monitorTotal = new CPUMonitor("Reload");
                if (!Config.Load())
                {
                    return;
                }
                Logger.SetAutoGrowthMegabytes = 50;
                Logger.SetLogLevel = Config.Instance.LogLevel;
                Logger.SetLogPath = Config.Instance.LogPath;

                if (!DB.Load())
                {
                    return;
                }
                sClientKey = Utility.HexStringToByteArray(Config.Instance.ClientKey);
                sClientCipher = new StringCipher(Config.Instance.ClientKey);
                sAuthentication = new Authentication(Config.Instance.AuthenticationService,
                                                     Config.Instance.AuthenticationId,
                                                     Config.Instance.AuthenticationKey,
                                                     Config.Instance.SessionExpiration);

                sUpdateThread = new Thread(ProcessUpdateThread);
                sUpdateThread.Priority = ThreadPriority.Highest;
                sUpdateThread.Start();

                Logger.Log(ELogLevel.Debug,
                    null,
                    monitorTotal.ToString());

                Logger.Log(ELogLevel.Info,
                           null,
                           "Service Started");
            }
            sServiceStarted = true;

        }

        private static bool IsTrustedUser(Int32 pUserId)
        {
            return Array.Exists<int>(Config.Instance.Trusted, i => i == pUserId);
        }

        private void QueueUpdate(Package pBuf, Int32 pUserId)
        {

            Update update = new Update(Config.Instance.MinProfilerVersion, 
                                        Config.Instance.Versions,
                                        Config.Instance.SupportedLocales,
                                        pUserId,
                                        IsTrustedUser(pUserId));
            if (!update.Read(pBuf))
            {

                switch (update.UpdateStatus)
                {
                    case StatusCode.InvalidStream:
                        Logger.Log(ELogLevel.Error,
                              Context.Request.UserHostAddress,
                              "Invalid update file stream");
                        break;
                    case StatusCode.InvalidGameClientVersion:
                        Logger.Log(ELogLevel.Error,
                            Context.Request.UserHostAddress,
                            "Unsupported game client version: " + update.ClientVersion.Value);
                        break;
                    case StatusCode.InvalidProfilerVersion:
                        Logger.Log(ELogLevel.Error,
                            Context.Request.UserHostAddress,
                            "Out of date profiler version: " + update.ProfilerVersion);
                        break;

                }
                Context.Response.OutputStream.WriteByte((Byte)update.UpdateStatus);
                return;
            }
            
            update.UserHost = Context.Request.UserHostAddress;
            Logger.Log(ELogLevel.Debug,
                 update.UserHost,
                "Processing update with locale: {0}, profiler version: {1} and game client version: {2}.",
                update.Language,
                update.ProfilerVersion,
                update.ClientVersion.Value);
            
            lock (sUpdateQueue)
            {
                if (sUpdateQueue.Count >= Config.Instance.MaxQueuedUpdates && !update.IsTrustedUser)
                {
                    update.Items.Clear();
                    update.ItemSets.Clear();
                    update.NPCs.Clear();
                    update.Abilities.Clear();
                    update.Quests.Clear();
                    update.PublicQuests.Clear();
                }
                sUpdateQueue.Enqueue(update);
            }
            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);
            return;
        }

        [WebMethod]
        public void GetStats(String pAdminKey)
        {
            Byte[] buf;
            if (pAdminKey.ToLower() != Config.Instance.AdminKey.ToLower())
            {
                Logger.Log(ELogLevel.Error,
                           Context.Request.UserHostAddress,
                           "GetStats attempted with invalid admin key");
                buf = Encoding.ASCII.GetBytes("Invalid admin key, attempt has been logged");
                Context.Response.OutputStream.Write(buf, 0, buf.Length);
                return;
            }
            int queueCount = 0;

            lock (sUpdateQueue)
            {
                queueCount = sUpdateQueue.Count;
            }

            int sessionCount = sAuthentication.GetSessionCount();
            long memoryUsed = System.Diagnostics.Process.GetCurrentProcess().WorkingSet64 / (1024 * 1024);            
            Context.Response.Write(String.Format("Queue Count: {0}\n\nMemory Usage: {1}\n\nSession Count: {2}",
                queueCount.ToString(),
                memoryUsed.ToString(),
                sessionCount.ToString()));


        }

        private static void ProcessUpdateThread()
        {
            Boolean aborted = false;
            Update update;
            while (!aborted)
            {
                try
                {
                    update = null;
                    lock (sUpdateQueue)
                    {
                        if (sUpdateQueue.Count > 0)
                        {
                            update = sUpdateQueue.Dequeue();
                        }
                    }

                    if (update == null)
                    {
                        Thread.Sleep(1000);
                        continue;
                    }

                    lock (sRebootLock)
                    {                                                                        
                        // DO stuff
                        if (IsRateLimitExceeded(update))
                        {
                            continue;
                        }

                        CPUMonitor monitorTotal = new CPUMonitor("ProcessUpdateThread for User {0}", update.UserId);
                        CPUMonitor subMonitor = null;

                        using (SqlConnection conn = new SqlConnection(Config.Instance.ConnectionString))
                        {
                            conn.Open();
                                                        
                            // Save Servers, Guilds and Players for all languages:
                            DBServers.Save(update, conn);
                                                        
                            DBGuilds.Save(update, conn);
                                                        
                            DBPlayers.Save(update, conn);


                            // Only save the other for supported languages:
                            if (update.IsSupportedLanguage && DBServers.IsSupportedServer(update.ServerName, Config.Instance.SupportedLocales))
                            {

                                DBItems.Save(update, conn);

                                DBNPCs.Save(update, conn);

                                DBGameObjects.Save(update, conn);

                                DBPublicQuests.Save(update, conn);

                                DBQuests.Save(update, conn);

                                DBInfluence.Save(update, conn);

                                DBAdvances.Save(update, conn);

                                DBAbilities.Save(update, conn);

                                DBItemSets.Save(update, conn);

                                DBBattlefieldStatus.Save(update, conn);
                            }
                            else
                            {
                                Logger.Log(ELogLevel.Debug, update.UserHost, "Unsupported Server:"+update.ServerName);
                            }
                            
                        }
                        Logger.Log(ELogLevel.Debug,
                                update.UserHost,
                                monitorTotal.ToString());

                    }
                }
                catch (ThreadAbortException tae)
                {
                    aborted = true;                    
                    Logger.Log(ELogLevel.Debug,
                               null,
                               "ProcessUpdateThread Aborted: {0}", tae.Message + "\n" + tae.StackTrace);
                }
                catch (Exception exc)
                {
                    Logger.Log(ELogLevel.Debug,
                               null,
                               "ProcessUpdateThread Exception: {0}",
                               exc.Message);
                    Logger.Log(ELogLevel.Debug,
                               null,
                               exc.StackTrace);
                }
            }
        }

        private static bool IsRateLimitExceeded(Update pUpdate)
        {
            if (pUpdate.IsTrustedUser)
            {
                return false;
            }
            DateTime now = DateTime.UtcNow;
            DateTime lastupdate;

            if (!sUpdateTimes.TryGetValue(pUpdate.UserId, out lastupdate) ||
                            now.Subtract(lastupdate).TotalSeconds >
                            Config.Instance.RateLimit)
            {
                sUpdateTimes[pUpdate.UserId] = DateTime.UtcNow;
                return false;
            }
            else
            {
                Logger.IgnoredAction(pUpdate.UserId, pUpdate.UserHost, "Update", "Exceeded update rate limit");
                return true;
            }            
        }

        /**
         * Publically exposed web method for obtaining a hash of the latest DLL
         * Parameters are taken through form fields, expects a form posted file
         * Returns a StatusCode in the response stream
         */
        [WebMethod]
        public void UploadUpdate()
        {
            if (!sServiceStarted)
            {
                Context.Response.OutputStream.WriteByte((Byte)StatusCode.Unknown);
                return;
            }

            try
            {

                String session = Context.Request.Form["pSession"];

                if (Context.Request.Files.Count == 0)
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "No file to read from the stream");
                    Context.Response.OutputStream.WriteByte((Byte)StatusCode.InvalidStream);
                    return;
                }

                Int32 userId;
                StatusCode status = sAuthentication.ValidateSessionKeyToUserId(session, out userId);

                if (status != StatusCode.Ok)
                {
                    Logger.Log(ELogLevel.Access,
                               Context.Request.UserHostAddress,
                               "ValidateSessionKeyToUserId failed with status {0}",
                               status);
                    Context.Response.OutputStream.WriteByte((Byte)status);
                    return;
                }

                Logger.Log(ELogLevel.Debug,
                           Context.Request.UserHostAddress,
                           "Received {0} bytes for update file stream",
                           Context.Request.Files[0].InputStream.Length);

                Package buf = new Package(Encoding.UTF8, Context.Request.Files[0].InputStream);


                if (!buf.Decompress())
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "Invalid compressed update file stream");
                    Context.Response.OutputStream.WriteByte((Byte)StatusCode.InvalidStream);
                    return;
                }
                QueueUpdate(buf, userId);


            }
            catch (Exception exc)
            {
                Logger.Log(ELogLevel.Error,
                           Context.Request.UserHostAddress,
                           "UploadUpdate Exception: {0}",
                           exc.Message);
                Logger.Log(ELogLevel.Error,
                           Context.Request.UserHostAddress,
                           exc.StackTrace);
            }

            return;
        }


        [WebMethod]
        public void TestUpdate(string test)
        {
            byte[] testbuf = Convert.FromBase64String(test);                        
            Int32 userId = 1;            
            Package buf = new Package(Encoding.UTF8, testbuf);
            QueueUpdate(buf, userId);
        }

        /**
        * Internal class for passing update thread parameters
        */
        private class UpdateThreadParam
        {            
            public String Host;
            public Update Update;

            /**
             * Initialization constructor
             */
            public UpdateThreadParam(String pHost,
                                     Update pUpdate)
            {            
                Host = pHost;
                Update = pUpdate;
            }
        }
    }
}
