using Curse;
using Curse.WoW;
using Curse.WoW.WDB;
using WoWDataCenter.DBC;
using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Text;

namespace WoWDataCenter
{
    /**
     * Handles all the internal details of managing data pushes to a database
     *
     * @author Shane Bryldt
     */
    public static partial class DB
    {
        /**
         * Prepares the DB class for use
         * Clears validation maps
         * Clears history maps
         * Clears confirmation maps
         * Reloads validation maps
         * 
         * @return  true if loading was successful, false otherwise
         */
        public static Boolean Load()
        {
            CPUMonitor monitorTotal = new CPUMonitor("DB Loading");
            Boolean ok = true;
            try
            {
                sCreatureHash.Clear();
                sGameObjectHash.Clear();
                sGameObjectPositionToId.Clear();
                sItemHash.Clear();
                sPageTextHash.Clear();
                sQuestHash.Clear();

                sServerKeyToId.Clear();
                sGuildKeyToId.Clear();
                sPlayerKeyToId.Clear();

                sCreatureHistory.Clear();
                sGameObjectHistory.Clear();
                sItemHistory.Clear();
                sPageTextHistory.Clear();
                sQuestHistory.Clear();

                sCreatureConfirmations.Clear();
                sCreatureLocationConfirmations.Clear();
                sCreatureLootConfirmations.Clear();
                sCreatureMerchandiseConfirmations.Clear();
                sCreatureLimitedMerchandiseConfirmations.Clear();
                sCreatureMerchandiseCostsConfirmations.Clear();
                sCreatureTrainConfirmations.Clear();
                sGameObjectConfirmations.Clear();
                sGameObjectLocationConfirmations.Clear();
                sGameObjectLootConfirmations.Clear();
                sItemConfirmations.Clear();
                sItemLootConfirmations.Clear();
                sPageTextConfirmations.Clear();
                sQuestConfirmations.Clear();
                sQuestExperienceConfirmations.Clear();
                sQuestFactionConfirmations.Clear();

                PopulatePositionToId(sGameObjectPositionToId,
                                         "gameobject",
                                         "gameobject_location",
                                         "object_id");

                foreach (Int32 ver in Config.Instance.Versions)
                {
                    UInt16 version = Convert.ToUInt16(ver);
                    Logger.Log(ELogLevel.Debug,
                               null,
                               "Loading for version {0}",
                               version);
                    PopulateId(sCreatureHash,
                               "creature",
                               version);
                    PopulateId(sGameObjectHash,
                               "gameobject",
                               version);                    

                    PopulateId(sItemHash,
                               "item",
                               version);
                    PopulateId(sPageTextHash,
                               "pagetext",
                               version);
                    PopulateId(sQuestHash,
                               "quest",
                               version);
                }
                PopulateServerKeyToId();
                PopulateGuildKeyToId();
                PopulatePlayerKeyToId();

                LoadSchema(ref sCreatureSchema,
                           "creature");
                LoadSchema(ref sGameObjectSchema,
                           "gameobject");
                LoadSchema(ref sItemSchema,
                           "item");
                LoadSchema(ref sPageTextSchema,
                           "pagetext");
                LoadSchema(ref sQuestSchema,
                           "quest");
                LoadCreatureSchemaOrdinals();
                LoadGameObjectSchemaOrdinals();
                LoadItemSchemaOrdinals();
                LoadPageTextSchemaOrdinals();
                LoadQuestSchemaOrdinals();
            }
            catch (Exception exc)
            {
                ok = false;
                Logger.Log(ELogLevel.Debug,
                           null,
                           "DB Load Exception: {0}",
                           exc.Message);
            }

            Logger.Log(ELogLevel.Debug,
                       null,
                       monitorTotal.ToString());

            return ok;
        }

        /**
         * Loads validation mapping for obtaining ID based on position and name
         * Does NOT clear the sightings map, allowing multiple versions to be loaded into the same map
         * 
         * @param pSightings      the sightings map to populate
         * @param pEntityTable    the name of the table which entity id's are obtained
         * @param pPositionTable  the name of the table which positions are obtained
         * @param pIdColumn       the name of the column in the position table relating to the entity id
         * @param pVersion        the wow version to filter the results on
         */
        private static void PopulatePositionToId(Dictionary<CustomKey, List<IdAndName>> pSightings,
                                                 String pEntityTable,
                                                 String pPositionTable,
                                                 String pIdColumn)
        {
            StringBuilder sql = new StringBuilder();
            sql.Append("SELECT e.id,p.area_id,p.pos_x,p.pos_y,");

            String[] locales = Enum.GetNames(typeof(ELocale));
            foreach (String locale in locales)
            {
                sql.AppendFormat("e.name_{0},", locale);
            }
            sql.Remove(sql.Length - 1, 1);

            sql.Append(" FROM ");
            sql.AppendFormat("{0} AS e INNER JOIN {1} AS p ", pEntityTable, pPositionTable);
            sql.AppendFormat("ON e.id = p.{0};", pIdColumn);

            CustomKey key;
            Int32 index;
            Int32 id;
            Int16 area;
            Int16 x;
            Int16 y;
            String name;
            List<IdAndName> sightings;
            IdAndName value;
            Int32 count = 0;
            using (SqlConnection conn = new SqlConnection(Config.Instance.WoWDB))
            {
                conn.Open();
                SqlTransaction trans = conn.BeginTransaction();
                SqlCommand cmd = new SqlCommand(sql.ToString(), conn, trans);
                using (SqlDataReader dr = cmd.ExecuteReader())
                {
                    while (dr.Read())
                    {
                        index = 0;
                        id = (Int32)dr[index++];
                        area = (Int16)dr[index++];
                        x = (Int16)dr[index++];
                        y = (Int16)dr[index++];
                        for (Int32 locale = 0; locale < Config.LocaleCount; ++locale, ++index)
                        {
                            name = null;
                            if (!dr.IsDBNull(index))
                            {
                                name = (String)dr[index];
                            }

                            if (name != null)
                            {
                                key = new CustomKey((ELocale)locale,
                                                    area,
                                                    x,
                                                    y);
                                sightings = null;
                                if (!pSightings.TryGetValue(key, out sightings))
                                {
                                    value = new IdAndName(id, name);
                                    sightings = new List<IdAndName>();
                                    sightings.Add(value);
                                    pSightings[key] = sightings;
                                }
                                else
                                {
                                    value = sightings.Find(v => v.Id == id);
                                    if (value == null)
                                    {
                                        value = new IdAndName(id, name);
                                        sightings.Add(value);
                                    }
                                    else
                                    {
                                        value.Name = name;
                                    }
                                }
                                ++count;
                            }
                        }
                    }
                }
                trans.Commit();
            }
            Logger.Log(ELogLevel.Debug,
                       null,
                       "Populated {0} name and position to id mappings from tables {1} and {2}",
                       count,
                       pEntityTable,
                       pPositionTable);
        }

        /**
         * Loads validation mapping for known IDs
         * Does NOT clear the hash map, allowing multiple versions to be loaded into the same map
         * 
         * @param pHash         the hash map to populate
         * @param pEntityTable  the name of the table which entity id's are obtained
         * @param pVersion      the wow version to filter the results on
         */
        private static void PopulateId(HashSet<CustomKey> pHash,
                                       String pEntityTable,
                                       UInt16 pVersion)
        {
            StringBuilder sql = new StringBuilder();
            sql.AppendFormat("SELECT id FROM {0} WHERE version_id={1};",
                             pEntityTable,
                             pVersion);

            CustomKey key;
            Int32 id;
            Int32 count = 0;
            using (SqlConnection conn = new SqlConnection(Config.Instance.WoWDB))
            {
                conn.Open();
                SqlTransaction trans = conn.BeginTransaction();
                SqlCommand cmd = new SqlCommand(sql.ToString(), conn, trans);
                using (SqlDataReader dr = cmd.ExecuteReader())
                {
                    while (dr.Read())
                    {
                        id = (Int32)dr[0];
                        key = new CustomKey(pVersion,
                                            id);
                        pHash.Add(key);
                        ++count;
                    }
                }
                trans.Commit();
            }
            Logger.Log(ELogLevel.Debug,
                       null,
                       "Populated {0} ids from table {1} for version {2}",
                       count,
                       pEntityTable,
                       pVersion);
        }

        /**
         * Loads validation mapping for known servers
         */
        private static void PopulateServerKeyToId()
        {
            StringBuilder sql = new StringBuilder();
            sql.Append("SELECT id,name,region FROM server");

            CustomKey key;
            Int32 index;
            Int32 id;
            String name;
            Byte region;
            using (SqlConnection conn = new SqlConnection(Config.Instance.WoWDB))
            {
                conn.Open();
                SqlTransaction trans = conn.BeginTransaction();
                SqlCommand cmd = new SqlCommand(sql.ToString(), conn, trans);
                using (SqlDataReader dr = cmd.ExecuteReader())
                {
                    while (dr.Read())
                    {
                        index = 0;
                        id = (Int32)dr[index++];
                        name = (String)dr[index++];
                        region = (Byte)dr[index++];

                        key = new CustomKey(name.ToLowerInvariant(),
                                            region);
                        sServerKeyToId[key] = id;
                    }
                }
                trans.Commit();
            }
        }

        /**
         * Loads validation mapping for known guilds
         */
        private static void PopulateGuildKeyToId()
        {
            StringBuilder sql = new StringBuilder();
            sql.Append("SELECT id,name,server_id FROM guild");

            CustomKey key;
            Int32 index;
            Int32 id;
            String name;
            Int32 serverId;
            using (SqlConnection conn = new SqlConnection(Config.Instance.WoWDB))
            {
                conn.Open();
                SqlTransaction trans = conn.BeginTransaction();
                SqlCommand cmd = new SqlCommand(sql.ToString(), conn, trans);
                using (SqlDataReader dr = cmd.ExecuteReader())
                {
                    while (dr.Read())
                    {
                        index = 0;
                        id = (Int32)dr[index++];
                        name = (String)dr[index++];
                        serverId = (Int32)dr[index++];

                        key = new CustomKey(name,
                                            serverId);
                        sGuildKeyToId[key] = id;
                    }
                }
                trans.Commit();
            }
        }
        /**
         * Loads validation mapping for known players
         */
        private static void PopulatePlayerKeyToId()
        {
            StringBuilder sql = new StringBuilder();
            sql.Append("SELECT id,name,server_id FROM player");

            CustomKey key;
            Int32 index;
            Int32 id;
            String name;
            Int32 serverId;
            using (SqlConnection conn = new SqlConnection(Config.Instance.WoWDB))
            {
                conn.Open();
                SqlTransaction trans = conn.BeginTransaction();
                SqlCommand cmd = new SqlCommand(sql.ToString(), conn, trans);
                using (SqlDataReader dr = cmd.ExecuteReader())
                {
                    while (dr.Read())
                    {
                        index = 0;
                        id = (Int32)dr[index++];
                        name = (String)dr[index++];
                        serverId = (Int32)dr[index++];

                        key = new CustomKey(name,
                                            serverId);
                        sPlayerKeyToId[key] = id;
                    }
                }
                trans.Commit();
            }
        }

        /**
         * Utility method to escape a string containing single quotes with double quotes, and quote the return
         * 
         * @param  pValue  the string to escape and quote
         * @return         the escaped and quoted result
         */
        private static String EscapeAndQuote(String pValue)
        {
            return "N'" + pValue.Replace("'", "''") + "'";
        }

        /**
         * Private support method used to check if a given key has been confirmed and is ready to push to the DB
         * 
         * @param  pConfirmations      the map of confirmations to check in
         * @param  pKey                the key to check for
         * @param  pConfirmationCount  the minimum number of confirmations to require before allowing a database push
         * @param  pUserId             the id of the user sending the update
         * @param  pVersion            the wow version to check for skipping confirmation on
         * @return                     true if the data is ready to be pushed to the database, false otherwise
         */

        private static Boolean IsEntityExpired( Dictionary<CustomKey, DateTime> pHistory,
                                                CustomKey pKey,
                                                Int32 pExpiration,
                                                Int32 pUserId)
        {

            DateTime found;
            DateTime now = DateTime.UtcNow;

            if (!Array.Exists<Int32>(Config.Instance.Trusted, i => pUserId == i) &&
               pHistory.TryGetValue(pKey, out found))
            {
                if (now.Subtract(found).TotalSeconds < pExpiration)
                {
                    return false;
                }
            }
            lock (pHistory)
            {
                pHistory[pKey] = now;
            }
            return true;

        }

        private static Boolean IsEntityExpiredNoUpdate(Dictionary<CustomKey, DateTime> pHistory,
                                               CustomKey pKey,
                                               Int32 pExpiration,
                                               Int32 pUserId)
        {

            DateTime found;
            DateTime now = DateTime.UtcNow;

            if (!Array.Exists<Int32>(Config.Instance.Trusted, i => pUserId == i) &&
               pHistory.TryGetValue(pKey, out found))
            {
                if (now.Subtract(found).TotalSeconds < pExpiration)
                {
                    return false;
                }
            }            
            return true;

        }

        private static Boolean ReadyForDB(Dictionary<CustomKey, Confirmation> pConfirmations,
                                          CustomKey pKey,
                                          Int32 pConfirmationCount,
                                          Int32 pUserId,
                                          Int32 pVersion)
        {
            Boolean IsReady = false;           
            Confirmation confirmation = null;
            if (!pConfirmations.TryGetValue(pKey, out confirmation))
            {
                confirmation = new Confirmation(pUserId);

                lock (pConfirmations)
                {
                    pConfirmations[pKey] = confirmation;
                }
                IsReady = false;
            }
            else
            {
                ++confirmation.Count;
            }
            if (pVersion == Config.Instance.SkipConfirmationForVersion ||
               Array.Exists<Int32>(Config.Instance.Trusted, i => pUserId == i) ||
               pConfirmationCount < 2)
            {
                return true;
            }           
            IsReady = confirmation.Count >= pConfirmationCount;
            return IsReady;
        }
        /**
         * Private support method used to check if a given key has been confirmed and is ready to push to the DB
         * Adds additional history check for minimum expiration time between allowing updates
         * 
         * @param  pHistory            the map of history to check in
         * @param  pConfirmations      the map of confirmations to check in
         * @param  pKey                the id to check for
         * @param  pExpiration         the minimum number of seconds to require before allowing another update to the same key
         * @param  pConfirmationCount  the minimum number of confirmations to require before allowing a database push
         * @param  pUserId             the id of the user sending the update
         * @param  pVersion            the wow version to check for skipping confirmation on, and building the custom key
         * @param  pLocale             the locale used to build the custom key
         * @return                     true if the data is ready to be pushed to the database, false otherwise
         */
        private static Boolean ReadyForDB(Dictionary<CustomKey, DateTime> pHistory,
                                          Dictionary<CustomKey, Confirmation> pConfirmations,
                                          Int32 pKey,
                                          Int32 pExpiration,
                                          Int32 pConfirmationCount,
                                          Int32 pUserId,
                                          UInt16 pVersion,
                                          ELocale pLocale)
        {
            CustomKey key = new CustomKey(pVersion,
                                          pLocale,
                                          pKey);

            if (!ReadyForDB(pConfirmations,
                            key,
                            pConfirmationCount,
                            pUserId,
                            pVersion))
            {
                return false;
            }
            /*
            DateTime now = DateTime.UtcNow;
            DateTime found;
            if (!Array.Exists<Int32>(Config.Instance.Trusted, i => pUserId == i) &&
                pHistory.TryGetValue(key, out found))
            {
                if (now.Subtract(found).TotalSeconds < pExpiration)
                {
                    return false;
                }
            }
            pHistory[key] = now;
             * return true;
             */


            return IsEntityExpired(pHistory, key, pExpiration, pUserId);
            
        }

        /**
         * Private support method used to convert a name and coordinates into an ID
         * 
         * @param  pSightings  the sightings map to check in
         * @param  pVersion    the wow version used to build the custom key
         * @param  pLocale     the locale used to build the custom key
         * @param  pName       the name of the entity to lookup
         * @param  pAreaId     the area id of the coordinates to build the custom key
         * @param  pX          the X coordinate to build the custom key
         * @param  pY          the Y coordinate to build the custom key
         * @param  pId         the out value assigned to the id if found
         * @return             true if the id could be resolved, false otherwise
         */
        private static Boolean ResolveNameToId(Dictionary<CustomKey, List<IdAndName>> pSightings,                                               
                                               ELocale pLocale,
                                               String pName,
                                               UInt16 pAreaId,
                                               Int16 pX,
                                               Int16 pY,
                                               out Int32 pId)
        {
            CustomKey key;
            List<IdAndName> subjects;
            pId = 0;
            key = new CustomKey(pLocale,
                                pAreaId,
                                pX,
                                pY);
            if (!pSightings.TryGetValue(key, out subjects))
            {
                return false;
            }
            foreach (IdAndName subject in subjects)
            {
                if (subject.Name == pName)
                {
                    pId = subject.Id;
                    return true;
                }
            }
            return false;
        }
        /**
         * Private support method used to add new coordinate locations for an entity
         * 
         * @param  pSightings  the sightings map to add to
         * @param  pVersion    the wow version used to build the custom key
         * @param  pLocale     the locale used to build the custom key
         * @param  pName       the name of the entity
         * @param  pAreaId     the area id of the coordinates to build the custom key
         * @param  pX          the X coordinate to build the custom key
         * @param  pY          the Y coordinate to build the custom key
         * @param  pId         the id of the entity
         */
        private static void AddLocation(Dictionary<CustomKey, List<IdAndName>> pSightings,                                        
                                        ELocale pLocale,
                                        String pName,
                                        UInt16 pAreaId,
                                        Int16 pX,
                                        Int16 pY,
                                        Int32 pId)
        {
            CustomKey key;
            List<IdAndName> subjects;
            IdAndName entity;
            key = new CustomKey(pLocale,
                                pAreaId,
                                pX,
                                pY);
            if (!pSightings.TryGetValue(key, out subjects))
            {
                entity = new IdAndName(pId, pName);
                subjects = new List<IdAndName>();
                subjects.Add(entity);
                lock (pSightings)
                {
                    pSightings[key] = subjects;
                }
            }
            else
            {
                Boolean found = false;
                foreach (IdAndName subject in subjects)
                {
                    if (subject.Id == pId)
                    {
                        if (subject.Name == null ||
                            subject.Name.Length == 0)
                        {
                            subject.Name = pName;
                        }
                        found = true;
                        break;
                    }
                }

                if (!found)
                {
                    entity = new IdAndName(pId, pName);
                    subjects.Add(entity);
                }
            }
        }

        private static void LoadSchema(ref DataTable pSchema,
                                       String pTable)
        {
            pSchema = new DataTable();

            using (SqlConnection conn = new SqlConnection(Config.Instance.WoWDB))
            {
                conn.Open();
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "SELECT TOP 1 * FROM " + pTable;
                using (SqlDataReader dr = cmd.ExecuteReader(CommandBehavior.SchemaOnly))
                {
                    pSchema.Load(dr);
                }
            }
        }

        public static void AddServers(List<KeyValuePair<String, Byte>> pServers)
        {
            CustomKey key = null;
            Int32 serverId = 0;
            using (SqlConnection conn = new SqlConnection(Config.Instance.WoWDB))
            {
                conn.Open();
                SqlCommand cmd = conn.CreateCommand();
                lock (sServerKeyToId)
                {
                    foreach (KeyValuePair<String, Byte> server in pServers)
                    {

                        //The server key always uses lower case
                        key = new CustomKey(server.Key.ToLowerInvariant(),
                                            server.Value);

                        if (!sServerKeyToId.ContainsKey(key))
                        {
                            cmd.CommandText = String.Format("INSERT INTO server(name,region) OUTPUT INSERTED.id VALUES({0},{1});",
                                                            EscapeAndQuote(server.Key),
                                                            server.Value);
                            serverId = (Int32)cmd.ExecuteScalar();
                            sServerKeyToId[key] = serverId;
                        }
                    }
                }
            }
        }

        public static Boolean ValidateServer(ref String pServer, Byte pRegion)
        {
            //For euro regions, strip accents:
            if (pRegion == 2)
            {
                byte[] b = Encoding.GetEncoding(1251).GetBytes(pServer); // 8 bit characters
                pServer = Encoding.ASCII.GetString(b); // 7 bit characters 
            }
            //The server key always uses lower case
            return sServerKeyToId.ContainsKey(new CustomKey(pServer.ToLowerInvariant(),
                                                            pRegion));
        }

        /**
         * Publically exposed method for updating player profiles
         * 
         * @param  pUserId    the id of the user sending the update
         * @param  pHost      the host of the user updating in IP format
         * @param  pVersion   the wow version
         * @param  pLocale    the locale
         * @param  pPlayers   the list of player profiles contained in the update
         * @param  pFactions  the list of strings for faction indexes used in the update
         */
        public static void UpdatePlayers(Int32 pUserId,
                                         String pHost,
                                         UInt16 pVersion,
                                         ELocale pLocale,
                                         PackableList<Player> pPlayers,
                                         PackableStringList pFactions,
                                         Byte pRegion)
        {
            
#if ALPHA
            CPUMonitor monitorTotal = new CPUMonitor("UpdatePlayers for {0} entries", pPlayers.Count);
#endif


            if (pPlayers.Count == 0)
            {
                return;
            }            

            Boolean skipPlayer;
            CustomKey key;
            Int32 serverId;
            Int32 guildId;
            Int32 playerId;
            Int32 friendId;
            ItemReference item;
            Byte slot;
            String factionName;
            UInt16 factionId;
            UInt16 lineId;
            StringBuilder sql = new StringBuilder(SQL_INITIAL_CAPACITY);
            using (SqlConnection conn = new SqlConnection(Config.Instance.WoWDB))
            {
                conn.Open();
                using (SqlTransaction trans = conn.BeginTransaction())
                {
                    SqlCommand cmd = new SqlCommand();
                    cmd.Connection = conn;
                    cmd.Transaction = trans;

                    foreach (Player player in pPlayers)
                    {
                        skipPlayer = false;
                        foreach (PlayerFaction faction in player.Factions)
                        {
                            if (faction.Index > pFactions.Count)
                            {
                                Logger.IgnoredAction(pUserId,
                                                             pHost,
                                                             "Update Player",
                                                             "Invalid faction {0}, from list: {1}",
                                                             faction.Index,
                                                             string.Join(",", pFactions.ToArray()));
                                skipPlayer = true;
                                break;
                            }
                        }

                        if (skipPlayer)
                        {
                            continue;
                        }

                        if (!ChrRacesDBC.Exists(pVersion,
                                                pLocale,
                                                player.RaceId))
                        {
                            Logger.IgnoredAction(pUserId,
                                                 pHost,
                                                 "Update Player",
                                                 "Invalid race {0}",
                                                 player.RaceId);
                            continue;
                        }
                        if (!ChrClassesDBC.Exists(pVersion,
                                                  pLocale,
                                                  player.ClassId))
                        {
                            Logger.IgnoredAction(pUserId,
                                                 pHost,
                                                 "Update Player",
                                                 "Invalid class {0}",
                                                 player.ClassId);
                            continue;
                        }
                                                                    
                        

                        key = new CustomKey(player.Server.ToLowerInvariant(),
                                            pRegion);
                        serverId = sServerKeyToId[key];

                        guildId = 0;
                        if (player.GuildName.Length > 0)
                        {
                            key = new CustomKey(player.GuildName,
                                                serverId);
                            if (!sGuildKeyToId.TryGetValue(key, out guildId))
                            {
                                cmd.CommandText = String.Format("INSERT INTO guild(name,server_id)  OUTPUT INSERTED.id VALUES({0},{1});",
                                                                EscapeAndQuote(player.GuildName),
                                                                serverId);
                                guildId = (Int32)cmd.ExecuteScalar();                                
                                sGuildKeyToId[key] = guildId;                                
                            }
                        }

                        key = new CustomKey(player.Name,
                                            serverId);
                        if (!sPlayerKeyToId.TryGetValue(key, out playerId))
                        {
                            cmd.CommandText = String.Format(sPlayerInsertQueryFormat,
                                                            EscapeAndQuote(player.Name),
                                                            serverId,
                                                            pUserId,
                                                            player.CombatRatings[0].Amount,
                                                            player.CombatRatings[1].Amount,
                                                            player.CombatRatings[2].Amount,
                                                            player.CombatRatings[3].Amount,
                                                            player.CombatRatings[4].Amount,
                                                            player.CombatRatings[5].Amount,
                                                            player.CombatRatings[6].Amount,
                                                            player.CombatRatings[7].Amount,
                                                            player.CombatRatings[8].Amount,
                                                            player.CombatRatings[9].Amount,
                                                            player.CombatRatings[10].Amount,
                                                            player.CombatRatings[11].Amount,
                                                            player.CombatRatings[12].Amount,
                                                            player.CombatRatings[13].Amount,
                                                            player.CombatRatings[14].Amount,
                                                            player.CombatRatings[15].Amount,
                                                            player.CombatRatings[16].Amount,
                                                            player.CombatRatings[17].Amount,
                                                            player.CombatRatings[18].Amount,
                                                            player.CombatRatings[19].Amount,
                                                            player.CombatRatings[20].Amount,
                                                            player.CombatRatings[21].Amount,
                                                            player.CombatRatings[22].Amount,
                                                            player.CombatRatings[23].Amount,
                                                            player.MeleeStats.MainHandWeaponSkill,
                                                            player.MeleeStats.OffHandWeaponSkill,
                                                            player.MeleeStats.MainHandMinDamage,
                                                            player.MeleeStats.MainHandMaxDamage,
                                                            player.MeleeStats.OffHandMinDamage,
                                                            player.MeleeStats.OffHandMaxDamage,
                                                            (Single)player.MeleeStats.MainHandSpeed / 100,
                                                            (Single)player.MeleeStats.OffHandSpeed / 100,
                                                            player.MeleeStats.AttackPower,
                                                            player.MeleeStats.AttackPowerBonus,
                                                            (Single)player.MeleeStats.Critical / 100,
                                                            player.DefenseStats.Defense,
                                                            player.DefenseStats.DefenseBonus,
                                                            player.DefenseStats.Dodge,
                                                            player.DefenseStats.Parry,
                                                            player.DefenseStats.Block,
                                                            player.MaxHealth,
                                                            player.MaxPower,
                                                            player.Money,
                                                            player.Gender,
                                                            player.Level,
                                                            player.RaceId,
                                                            player.ClassId,
                                                            guildId,
                                                            EscapeAndQuote(player.GuildRank),
                                                            player.TotalExperienceForLevel - player.Experience,
                                                            player.SpellStats.HolyBonus,
                                                            player.SpellStats.FireBonus,
                                                            player.SpellStats.NatureBonus,
                                                            player.SpellStats.FrostBonus,
                                                            player.SpellStats.ShadowBonus,
                                                            player.SpellStats.ArcaneBonus,
                                                            player.SpellStats.HealingBonus,
                                                            (Single)player.SpellStats.HolyCritical / 100,
                                                            (Single)player.SpellStats.FireCritical / 100,
                                                            (Single)player.SpellStats.NatureCritical / 100,
                                                            (Single)player.SpellStats.FrostCritical / 100,
                                                            (Single)player.SpellStats.ShadowCritical / 100,
                                                            (Single)player.SpellStats.ArcaneCritical / 100,
                                                            player.SpellStats.Penetration,
                                                            player.SpellStats.MP5NotCasting,
                                                            player.SpellStats.MP5Casting,
                                                            player.RangedStats.WeaponSkill,
                                                            player.RangedStats.MinDamage,
                                                            player.RangedStats.MaxDamage,
                                                            (Single)player.RangedStats.Speed / 100,
                                                            player.RangedStats.AttackPower,
                                                            player.RangedStats.AttackPowerBonus,
                                                            (Single)player.RangedStats.Critical / 100,
                                                            player.Statistics[0].Base,
                                                            player.Statistics[0].Modifier,
                                                            player.Statistics[1].Base,
                                                            player.Statistics[1].Modifier,
                                                            player.Statistics[2].Base,
                                                            player.Statistics[2].Modifier,
                                                            player.Statistics[3].Base,
                                                            player.Statistics[3].Modifier,
                                                            player.Statistics[4].Base,
                                                            player.Statistics[4].Modifier,
                                                            player.Resistances[0].Base,
                                                            player.Resistances[0].Modifier,
                                                            player.Resistances[1].Base,
                                                            player.Resistances[1].Modifier,
                                                            player.Resistances[2].Base,
                                                            player.Resistances[2].Modifier,
                                                            player.Resistances[3].Base,
                                                            player.Resistances[3].Modifier,
                                                            player.Resistances[4].Base,
                                                            player.Resistances[4].Modifier,
                                                            player.Resistances[5].Base,
                                                            player.Resistances[5].Modifier,
                                                            player.Resistances[6].Base,
                                                            player.Resistances[6].Modifier);
                            playerId = (Int32)cmd.ExecuteScalar();                            
                            sPlayerKeyToId[key] = playerId;                            
                        }
                        else
                        {
                            sql.AppendFormat(sPlayerUpdateQueryFormat,
                                             pUserId,
                                             player.CombatRatings[0].Amount,
                                             player.CombatRatings[1].Amount,
                                             player.CombatRatings[2].Amount,
                                             player.CombatRatings[3].Amount,
                                             player.CombatRatings[4].Amount,
                                             player.CombatRatings[5].Amount,
                                             player.CombatRatings[6].Amount,
                                             player.CombatRatings[7].Amount,
                                             player.CombatRatings[8].Amount,
                                             player.CombatRatings[9].Amount,
                                             player.CombatRatings[10].Amount,
                                             player.CombatRatings[11].Amount,
                                             player.CombatRatings[12].Amount,
                                             player.CombatRatings[13].Amount,
                                             player.CombatRatings[14].Amount,
                                             player.CombatRatings[15].Amount,
                                             player.CombatRatings[16].Amount,
                                             player.CombatRatings[17].Amount,
                                             player.CombatRatings[18].Amount,
                                             player.CombatRatings[19].Amount,
                                             player.CombatRatings[20].Amount,
                                             player.CombatRatings[21].Amount,
                                             player.CombatRatings[22].Amount,
                                             player.CombatRatings[23].Amount,
                                             player.MeleeStats.MainHandWeaponSkill,
                                             player.MeleeStats.OffHandWeaponSkill,
                                             player.MeleeStats.MainHandMinDamage,
                                             player.MeleeStats.MainHandMaxDamage,
                                             player.MeleeStats.OffHandMinDamage,
                                             player.MeleeStats.OffHandMaxDamage,
                                             (Single)player.MeleeStats.MainHandSpeed / 100,
                                             (Single)player.MeleeStats.OffHandSpeed / 100,
                                             player.MeleeStats.AttackPower,
                                             player.MeleeStats.AttackPowerBonus,
                                             (Single)player.MeleeStats.Critical / 100,
                                             player.DefenseStats.Defense,
                                             player.DefenseStats.DefenseBonus,
                                             player.DefenseStats.Dodge,
                                             player.DefenseStats.Parry,
                                             player.DefenseStats.Block,
                                             player.MaxHealth,
                                             player.MaxPower,
                                             player.Money,
                                             player.Gender,
                                             player.Level,
                                             player.RaceId,
                                             player.ClassId,
                                             guildId,
                                             EscapeAndQuote(player.GuildRank),
                                             player.TotalExperienceForLevel - player.Experience,
                                             player.SpellStats.HolyBonus,
                                             player.SpellStats.FireBonus,
                                             player.SpellStats.NatureBonus,
                                             player.SpellStats.FrostBonus,
                                             player.SpellStats.ShadowBonus,
                                             player.SpellStats.ArcaneBonus,
                                             player.SpellStats.HealingBonus,
                                             (Single)player.SpellStats.HolyCritical / 100,
                                             (Single)player.SpellStats.FireCritical / 100,
                                             (Single)player.SpellStats.NatureCritical / 100,
                                             (Single)player.SpellStats.FrostCritical / 100,
                                             (Single)player.SpellStats.ShadowCritical / 100,
                                             (Single)player.SpellStats.ArcaneCritical / 100,
                                             player.SpellStats.Penetration,
                                             player.SpellStats.MP5NotCasting,
                                             player.SpellStats.MP5Casting,
                                             player.RangedStats.WeaponSkill,
                                             player.RangedStats.MinDamage,
                                             player.RangedStats.MaxDamage,
                                             (Single)player.RangedStats.Speed / 100,
                                             player.RangedStats.AttackPower,
                                             player.RangedStats.AttackPowerBonus,
                                             (Single)player.RangedStats.Critical / 100,
                                             player.Statistics[0].Base,
                                             player.Statistics[0].Modifier,
                                             player.Statistics[1].Base,
                                             player.Statistics[1].Modifier,
                                             player.Statistics[2].Base,
                                             player.Statistics[2].Modifier,
                                             player.Statistics[3].Base,
                                             player.Statistics[3].Modifier,
                                             player.Statistics[4].Base,
                                             player.Statistics[4].Modifier,
                                             player.Resistances[0].Base,
                                             player.Resistances[0].Modifier,
                                             player.Resistances[1].Base,
                                             player.Resistances[1].Modifier,
                                             player.Resistances[2].Base,
                                             player.Resistances[2].Modifier,
                                             player.Resistances[3].Base,
                                             player.Resistances[3].Modifier,
                                             player.Resistances[4].Base,
                                             player.Resistances[4].Modifier,
                                             player.Resistances[5].Base,
                                             player.Resistances[5].Modifier,
                                             player.Resistances[6].Base,
                                             player.Resistances[6].Modifier,
                                             playerId);
                        }

                        sql.AppendFormat("DELETE FROM player_friend WHERE player_id={0};", playerId);

                        foreach (String friend in player.Friends)
                        {
                            key = new CustomKey(friend, serverId);
                            friendId = 0;
                            if (sPlayerKeyToId.TryGetValue(key, out friendId))
                            {
                                sql.AppendFormat("INSERT INTO player_friend(player_id,friend_id) VALUES({0},{1});",
                                                 playerId,
                                                 friendId);
                            }
                        }

                        sql.AppendFormat("DELETE FROM player_talent WHERE player_id={0};", playerId);

                        foreach (PlayerTalent talent in player.Talents)
                        {
                            sql.AppendFormat("INSERT INTO player_talent(player_id,spell_id,rank) VALUES({0},{1},{2});",
                                             playerId,
                                             talent.Id,
                                             talent.Rank );
                        }

                        sql.AppendFormat("DELETE FROM player_equipment WHERE player_id={0};",
                                         playerId);

                        for (Int32 index = 0;
                             index < player.Equipment.Count;
                             ++index)
                        {
                            item = player.Equipment[index];

                            if (item.ItemId == 0)
                            {
                                continue;
                            }

                            // Slots are not a 0 based index, so we add one to the index to get the correct slot
                            slot = (Byte)(index + 1);

                            sql.AppendFormat(sPlayerEquipmentInsertQueryFormat,
                                             playerId,
                                             slot,
                                             item.ItemId,
                                             item.EnchantId,
                                             item.Jewel1Id,
                                             item.Jewel2Id,
                                             item.Jewel3Id,
                                             item.Jewel4Id,
                                             item.Suffix);
                        }

                        sql.AppendFormat("DELETE FROM player_tradeskill WHERE player_id={0};",
                                         playerId);

                        List<Int32> processedRecipes = new List<Int32>();
                        foreach (PlayerTradeSkill tradeSkill in player.TradeSkills)
                        {
                            foreach (PlayerRecipe recipe in tradeSkill.Recipes)
                            {
                                if (!SpellDBC.Exists(pVersion,
                                                     pLocale,
                                                     recipe.RecipeId))
                                {
                                    Logger.IgnoredAction(pUserId,
                                                         pHost,
                                                         "Update Player Trade Skill",
                                                         "Invalid recipe spell {0}",
                                                         recipe.RecipeId);
                                    continue;
                                }
                                if (processedRecipes.Contains(recipe.RecipeId))
                                {
                                    continue;
                                }
                                sql.AppendFormat("INSERT INTO player_tradeskill(player_id,spell_id) VALUES({0},{1});",
                                                 playerId,
                                                 recipe.RecipeId);
                                processedRecipes.Add(recipe.RecipeId);
                            }
                        }

                        sql.AppendFormat("DELETE FROM player_faction WHERE player_id={0};",
                                         playerId);

                        foreach (PlayerFaction faction in player.Factions)
                        {
                            factionId = 0;
                            if (faction.Index > 0)
                            {
                                factionName = pFactions[faction.Index - 1];
                                if (!FactionDBC.LookupId(pVersion,
                                                         pLocale,
                                                         factionName,
                                                         out factionId))
                                {
                                    Logger.IgnoredAction(pUserId,
                                                         pHost,
                                                         "Update Player Faction",
                                                         "Invalid faction {0}",
                                                         factionName);
                                    continue;
                                }
                            }

                            sql.AppendFormat(sPlayerFactionInsertQueryFormat,
                                             playerId,
                                             factionId,
                                             faction.Standing,
                                             faction.Rating,
                                             faction.AtWar ? "1" : "0");
                        }

                        sql.AppendFormat("DELETE FROM player_skill WHERE player_id={0};",
                                         playerId);

                        foreach (PlayerSkill skill in player.Skills)
                        {
                            if (!SkillLineDBC.LookupId(pVersion,
                                                       pLocale,
                                                       skill.Name,
                                                       player.RaceId,
                                                       player.ClassId,
                                                       out lineId))
                            {
                                Logger.IgnoredAction(pUserId,
                                                     pHost,
                                                     "Update Player Skill",
                                                     "Invalid skill {0} for race {1}, class {2}",
                                                     skill.Name,
                                                     player.RaceId,
                                                     player.ClassId);
                                continue;
                            }

                            sql.AppendFormat(sPlayerSkillInsertQueryFormat,
                                             playerId,
                                             lineId,
                                             skill.Level,
                                             skill.MaxLevel);
                        }

                        sql.AppendFormat(sPlayerRecentItemDeleteQueryFormat,
                                         playerId,
                                         player.RecentItems.Count);
                        foreach (UInt16 itemId in player.RecentItems)
                        {
                            sql.AppendFormat(sPlayerRecentItemInsertQueryFormat,
                                             playerId,
                                             itemId);
                        }

                        // Player Mounts and Pets
                        sql.Append("delete from player_pet where player_id  = " + playerId + ";");

                        foreach (Int32 mountSpellId in player.Mounts)
                        {
                            sql.AppendFormat(sPlayerPetInsertQueryFormat, playerId, mountSpellId, 1);
                        }
                        foreach (Int32 critterSpellId in player.Critters)
                        {
                            sql.AppendFormat(sPlayerPetInsertQueryFormat, playerId, critterSpellId, 2);
                        }

                        //Achievements
                        sql.Append("delete from player_achievement where player_id  = " + playerId + ";");
                        foreach (UInt16 achievementId in player.Achievements)
                        {
                            sql.AppendFormat(sPlayerAchievementInsertQueryFormat, playerId, achievementId);
                        }

                        if (sql.Length >= SQL_INITIAL_CAPACITY)
                        {
                            cmd.CommandText = sql.ToString();
                            cmd.ExecuteNonQuery();
                            sql.Length = 0;
                        }
                    }
                    if (sql.Length > 0)
                    {
                        cmd.CommandText = sql.ToString();
                        cmd.ExecuteNonQuery();
                        sql.Length = 0;
                    }
                    trans.Commit();
                }
            }

#if ALPHA
            Logger.Log(ELogLevel.Debug,
                       pHost,
                       monitorTotal.ToString());
#endif
        }

        /**
         * Publically exposed method for updating NPC information
         * 
         * @param  pUserId    the id of the user sending the update
         * @param  pHost      the host of the user updating in IP format
         * @param  pVersion   the wow version
         * @param  pLocale    the locale
         * @param  pNPCs      the list of NPCs contained in the update
         * @param  pAreas     the list of strings for area indexes used in the update
         * @param  pFactions  the list of strings for faction indexes used in the update
         */
        public static void UpdateNPCs(Int32 pUserId,
                                      String pHost,
                                      UInt16 pVersion,
                                      ELocale pLocale,
                                      PackableList<NPC> pNPCs,
                                      PackableStringList pAreas,
                                      PackableStringList pFactions)
        {
#if ALPHA
            CPUMonitor monitorTotal = new CPUMonitor("UpdateNPCs for {0} entries", pNPCs.Count);
#endif

            if (pNPCs.Count == 0)
            {
                return;
            }

            Byte levelMin;
            Byte levelMax;
            Int32 levelCount;
            Int32 levelSum;
            Int32 hpMin;
            Int32 hpMax;
            Int64 hpCount;
            Int64 hpSum;
            DateTime updated;
            CustomKey key;
            UInt16 factionId;
            String factionName;
            String areaName;
            UInt16 areaId;
            Boolean insert;
            UInt16 lineId;
            Int32 spellId;
            List<Int32> trains = new List<Int32>();
            StringBuilder sql = new StringBuilder(SQL_INITIAL_CAPACITY);
            Boolean isCreatureExpired = false;

            using (SqlConnection conn = new SqlConnection(Config.Instance.WoWDB))
            {
                conn.Open();
                using (SqlTransaction trans = conn.BeginTransaction())
                {
                    SqlCommand cmd = new SqlCommand();
                    cmd.Connection = conn;
                    cmd.Transaction = trans;

                    foreach (NPC npc in pNPCs)
                    {
                        if (!sCreatureHash.Contains(new CustomKey(pVersion,
                                                                  npc.Id)) && !Array.Exists<Int32>(Config.Instance.Trusted, i => pUserId == i))
                        {
                            Logger.IgnoredAction(pUserId,
                                                 pHost,
                                                 "Update NPC",
                                                 "Unconfirmed id {0}",
                                                 npc.Id);
                            continue;
                        }

                        // Check if this creature is expired:
                        isCreatureExpired = IsEntityExpired(sCreatureLuaHistory,                                
                                new CustomKey(pVersion, npc.Id),
                                Config.Instance.CreatureExpiration,
                                pUserId);


                        // If the creature is not expired, do not update:
                        // Level
                        // HP
                        // Faction
                        // Reaction

                        if (isCreatureExpired)
                        {
                            if (npc.Statistics[0].Observations.Count > 0)
                            {
                                levelMin = 0;
                                levelMax = 0;
                                levelCount = 0;
                                levelSum = 0;
                                hpMin = 0;
                                hpMax = 0;
                                hpCount = 0;
                                hpSum = 0;

                                foreach (NPCObservation observation in npc.Statistics[0].Observations)
                                {
                                    if (observation.Level > 0)
                                    {
                                        levelCount++;
                                        levelSum += observation.Level;
                                        if (levelMin == 0)
                                        {
                                            levelMin = observation.Level;
                                        }
                                        else
                                        {
                                            levelMin = Math.Min(levelMin, observation.Level);
                                        }
                                        if (levelMax == 0)
                                        {
                                            levelMax = observation.Level;
                                        }
                                        else
                                        {
                                            levelMax = Math.Max(levelMax, observation.Level);
                                        }
                                    }
                                    if (observation.HitPoints > 0)
                                    {
                                        hpCount++;
                                        hpSum += observation.HitPoints;
                                        if (hpMin == 0)
                                        {
                                            hpMin = observation.HitPoints;
                                        }
                                        else
                                        {
                                            hpMin = Math.Min(hpMin, observation.HitPoints);
                                        }
                                        if (hpMax == 0)
                                        {
                                            hpMax = observation.HitPoints;
                                        }
                                        else
                                        {
                                            hpMax = Math.Max(hpMax, observation.HitPoints);
                                        }
                                    }
                                }

                                factionId = 0;
                                if (npc.Faction > 0)
                                {
                                    factionName = pFactions[npc.Faction - 1];
                                    FactionDBC.LookupId(pVersion,
                                                        pLocale,
                                                        factionName,
                                                        out factionId);
                                }
                                sql.AppendFormat(sNPCUpdateQueryFormat,
                                                 factionId,
                                                 levelCount == 0 ? "NULL" : levelMin.ToString(),
                                                 levelCount == 0 ? "NULL" : levelMax.ToString(),
                                                 levelSum,
                                                 levelCount,
                                                 hpCount == 0 ? "NULL" : hpMin.ToString(),
                                                 hpCount == 0 ? "NULL" : hpMax.ToString(),
                                                 hpSum,
                                                 hpCount,
                                                 npc.Statistics[0].Observations.Count,
                                                 pUserId,
                                                 npc.Id,
                                                 pVersion);

                            }

                            foreach (NPCReaction reaction in npc.Reactions)
                            {
                                factionId = 0;
                                if (reaction.Faction > 0)
                                {
                                    factionName = pFactions[reaction.Faction - 1];
                                    if (!FactionDBC.LookupId(pVersion,
                                                             pLocale,
                                                             factionName,
                                                             out factionId))
                                    {
                                        continue;
                                    }
                                }

                                sql.AppendFormat(sNPCReactionQueryFormat,
                                                 reaction.Reaction,
                                                 pUserId,
                                                 npc.Id,
                                                 factionId);
                            }

                          
                        }

                        // Creature Locations
                        foreach (Locations location in npc.Seen)
                        {
                            areaId = 0;
                            if (location.Area > 0)
                            {
                                areaName = pAreas[location.Area - 1];
                                if (!AreaTableDBC.LookupId(pVersion,
                                                           pLocale,
                                                           areaName,
                                                           out areaId))
                                {
                                    continue;
                                }
                            }

                            foreach (Coordinate coord in location.Coordinates)
                            {
                                key = new CustomKey(npc.Id,
                                                    areaId,
                                                    coord.X / 100,
                                                    coord.Y / 100);

                                if (!ReadyForDB(sCreatureLocationConfirmations,
                                                key,
                                                Config.Instance.MinConfirmations,
                                                pUserId,
                                                pVersion))
                                {
                                    continue;
                                }

                                sql.AppendFormat(sNPCLocationQueryFormat,
                                                 pUserId,
                                                 npc.Id,                                                 
                                                 areaId,
                                                 coord.X / 100,
                                                 coord.Y / 100,
                                                 (Single)coord.X / 100,
                                                 (Single)coord.Y / 100);
                            }
                        }

                        // Creature Loot
                        for (Byte mode = 0; mode < npc.Statistics.Count; ++mode)
                        {                            
                            foreach (NPCLoot npcloot in npc.Statistics[mode].Loots)
                            {
                                if (npcloot.MethodId > 0 &&
                                    !SpellDBC.Exists(pVersion,
                                                     pLocale,
                                                     npcloot.MethodId))
                                {
                                    continue;
                                }

                                insert = true;
                                foreach (PackableList<Loot> lootlist in npcloot.Loots)
                                {
                                    foreach (Loot loot in lootlist)
                                    {
                                        key = new CustomKey(npc.Id,
                                                            loot.Id,
                                                            mode);
                                        insert = insert &&
                                                 ReadyForDB(sCreatureLootConfirmations,
                                                            key,
                                                            Config.Instance.MinConfirmations,
                                                            pUserId,
                                                            pVersion);
                                    }
                                }

                                if (!insert)
                                {
                                    continue;
                                }

                                sql.AppendFormat(sNPCLootCountQueryFormat,
                                                 pUserId,
                                                 npc.Id,
                                                 npcloot.MethodId,
                                                 1 << mode,
                                                 npcloot.Loots.Count);

                                foreach (PackableList<Loot> lootlist in npcloot.Loots)
                                {
                                    foreach (Loot loot in lootlist)
                                    {
                                        if (loot.Id == 0)
                                        {
                                            sql.AppendFormat(sNPCCoinQueryFormat,
                                                             loot.Quantity,
                                                             pUserId,
                                                             npc.Id,                                                             
                                                             npcloot.MethodId);

                                        }
                                        else
                                        {
                                            sql.AppendFormat(sNPCLootQueryFormat,
                                                             1 << mode,
                                                             loot.Quantity,
                                                             pUserId,
                                                             npc.Id,                                                             
                                                             loot.Id,
                                                             npcloot.MethodId);
                                        }
                                    }
                                }
                            }

                            //Creature Spells
                            if (isCreatureExpired)
                            {
                                foreach (NPCSpell spell in npc.Statistics[mode].Spells)
                                {
                                    if (!SpellDBC.Exists(pVersion,
                                                         pLocale,
                                                         spell.SpellId))
                                    {
                                        continue;
                                    }

                                    sql.AppendFormat(sNPCSpellsQueryFormat,
                                                     1 << mode,
                                                     pUserId,
                                                     npc.Id,                                                     
                                                     spell.SpellId);
                                }
                            }
                        }


                        // Merchandise Costs
                        if(npc.MerchandiseCosts.Count > 0)
                        {
                            key = new CustomKey(pVersion,
                                                   pLocale,
                                                   npc.Id);                            

                            if (Array.Exists<Int32>(Config.Instance.Trusted, i => pUserId == i) ||
                                   !sNPCMerchandiseCostHistory.TryGetValue(key, out updated) ||
                                   DateTime.UtcNow.Subtract(updated).TotalSeconds >
                                   Config.Instance.MerchandiseExpiration)
                            {
                                sNPCMerchandiseCostHistory[key] = DateTime.UtcNow;
                                //Merchandise Costs
                                insert = true;
                                foreach (NPCMerchandiseCost item in npc.MerchandiseCosts)
                                {
                                    insert = insert &&
                                             ReadyForDB(sCreatureMerchandiseCostsConfirmations,
                                                        new CustomKey(pVersion,
                                                                      pLocale,
                                                                      npc.Id,
                                                                      item.Id),
                                                        Config.Instance.MinConfirmations,
                                                        pUserId,
                                                        pVersion);
                                }

                                if (insert)
                                {
                                    sql.AppendFormat("DELETE FROM creature_merchandise_cost WHERE creature_id={0};",
                                                     npc.Id);
                                    
                                    foreach (NPCMerchandiseCost item in npc.MerchandiseCosts)
                                    {
                                        sql.AppendFormat(sNPCMerchandiseCostInsertQueryFormat,
                                                         npc.Id,                                                         
                                                         item.Id,
                                                         item.Arena,
                                                         item.Honor,
                                                         item.Rating,
                                                         item.Item1,
                                                         item.Item1Amount,
                                                         item.Item2,
                                                         item.Item2Amount,
                                                         item.Item3,
                                                         item.Item3Amount,
                                                         pUserId);
                                    }
                                    
                                }

                            }

                        }

                        // Merchandise                       
                        if (npc.Merchandise.Count > 0 || npc.LimitedMerchandise.Count > 0)
                        {
                            if(IsEntityExpired(sNPCMerchandiseHistory,
                                new CustomKey(pVersion, pLocale,npc.Id), Config.Instance.MerchandiseExpiration, pUserId))
                            {                                                                        
                                sql.AppendFormat("DELETE FROM creature_merchandise WHERE creature_id={0};",
                                                 npc.Id);

                                sql.AppendFormat("DELETE FROM creature_limited_merchandise WHERE creature_id={0};",
                                                 npc.Id);

                                foreach (NPCMerchandise item in npc.Merchandise)
                                {
                                    sql.AppendFormat(sNPCMerchandiseInsertQueryFormat,
                                                     npc.Id,
                                                     item.Id,
                                                     item.Quantity,
                                                     pUserId);
                                }

                                foreach (NPCMerchandise item in npc.LimitedMerchandise)
                                {
                                    sql.AppendFormat(sNPCLimitedMerchandiseInsertQueryFormat,
                                                     npc.Id,                                      
                                                     pVersion,
                                                     item.Id,
                                                     item.Quantity,
                                                     pUserId);
                                }                                
                            }
                        } 
                        

                        // Trained Abilities            
                        if (npc.Teaches.Count > 0)
                        {                            
                            trains.Clear();
                            if (IsEntityExpired(sNPCTrainingHistory,
                                new CustomKey(pVersion, pLocale, npc.Id), Config.Instance.MerchandiseExpiration, pUserId))
                            {
                                foreach (NPCTrainSkill skill in npc.Teaches)
                                {
                                    if (!SkillLineDBC.LookupId(pVersion,
                                                               pLocale,
                                                               skill.Line,
                                                               skill.RaceId,
                                                               skill.ClassId,
                                                               out lineId))
                                    {
                                        Logger.Log(ELogLevel.Debug, pHost, "Unknown Skill line. Name: {0}, Race: {1}, Class: {2}", skill.Line, skill.RaceId, skill.ClassId);
                                        continue;
                                    }
                                    if (!SpellDBC.LookupId(pVersion,
                                                           pLocale,
                                                           skill.Name,
                                                           skill.Rank,
                                                           lineId,
                                                           out spellId))
                                    {
                                        Logger.Log(ELogLevel.Debug, pHost, "Unknown Skill. Name: {0}, Rank: {1}", skill.Name, skill.Rank);
                                        continue;
                                    }
                                    if (!trains.Contains(spellId))
                                    {
                                        trains.Add(spellId);
                                    }
                                }

                                Logger.Log(ELogLevel.Debug, pHost, "Saving {0} creature trained skills!", trains.Count);

                                if (trains.Count > 0)
                                {
                                    sql.AppendFormat("DELETE FROM creature_trained_skill WHERE creature_id={0};",
                                                     npc.Id);
                                    
                                    foreach (Int32 id in trains)
                                    {
                                        sql.AppendFormat(sNPCTrainedSkillInsertQueryFormat,
                                                         npc.Id,
                                                         id,
                                                         pUserId);
                                    }
                                }

                            }
                            else
                            {
                                Logger.Log(ELogLevel.Debug, pHost, "Creature trained skill not expired!");
                            }
                        }
                        

                        if (sql.Length >= SQL_INITIAL_CAPACITY)
                        {
                            cmd.CommandText = sql.ToString();
                            cmd.ExecuteNonQuery();
                            sql.Length = 0;
                        }
                    }
                    if (sql.Length > 0)
                    {
                        cmd.CommandText = sql.ToString();
                        cmd.ExecuteNonQuery();
                        sql.Length = 0;
                    }
                    trans.Commit();
                }
            }
#if ALPHA
            Logger.Log(ELogLevel.Debug,
                       pHost,
                       monitorTotal.ToString());
#endif
        }

        /**
         * Publically exposed method for updating game objects that give quests or are lootable
         * 
         * @param  pUserId               the id of the user sending the update
         * @param  pHost                 the host of the user updating in IP format
         * @param  pVersion              the wow version
         * @param  pLocale               the locale
         * @param  pQuestGameObjects     the list of game objects giving quests contained in the update
         * @param  pLootableGameObjects  the list of game objects which had loot contained in the update
         * @param  pCache                the cache for last chance validation of 
         * @param  pAreas                the list of strings for area indexes used in the update
         */
        public static void UpdateGameObjects(Int32 pUserId,
                                             String pHost,
                                             UInt16 pVersion,
                                             ELocale pLocale,
                                             PackableList<QuestGameObject> pQuestGameObjects,
                                             PackableList<LootableGameObject> pLootableGameObjects,
                                             PackableList<WGOBObject> pCache,
                                             PackableStringList pAreas)
        {

#if ALPHA
            CPUMonitor monitorTotal = new CPUMonitor("UpdateGameObjects for {0} quest entries, and {1} lootable entries",
                                                     pQuestGameObjects.Count,
                                                     pLootableGameObjects.Count);
#endif            

            if (pQuestGameObjects.Count == 0 &&
                pLootableGameObjects.Count == 0)
            {
                return;
            }

            String areaName;
            UInt16 areaId;
            Int32 objId;
            Boolean insert;
            StringBuilder sql = new StringBuilder(SQL_INITIAL_CAPACITY);
            using (SqlConnection conn = new SqlConnection(Config.Instance.WoWDB))
            {
                conn.Open();
                using (SqlTransaction trans = conn.BeginTransaction())
                {
                    SqlCommand cmd = new SqlCommand();
                    cmd.Connection = conn;
                    cmd.Transaction = trans;
                    foreach (QuestGameObject obj in pQuestGameObjects)
                    {
                        areaId = 0;
                        if (obj.Area > 0)
                        {
                            areaName = pAreas[obj.Area - 1];
                            if (!AreaTableDBC.LookupId(pVersion,
                                                       pLocale,
                                                       areaName,
                                                       out areaId))
                            {
                                Logger.IgnoredAction(pUserId,
                                                    pHost,
                                                    "Update LootableGameObject",
                                                    "Unknown area \"{0}\".",
                                                    areaName);
                                continue;
                            }
                        }
                        if (!ReadyForDB(sGameObjectLocationConfirmations,
                                        new CustomKey(pVersion,
                                                      pLocale,
                                                      obj.Id,
                                                      areaId,
                                                      obj.X / 100,
                                                      obj.Y / 100),
                                        Config.Instance.MinConfirmations,
                                        pUserId,
                                        pVersion))
                        {

                            continue;
                        }

                        sql.AppendFormat(sGameObjectLocationQueryFormat,
                                         pUserId,
                                         obj.Id,                                         
                                         areaId,
                                         obj.X / 100,
                                         obj.Y / 100,
                                         (Single)obj.X / 100,
                                         (Single)obj.Y / 100);

                        if (sql.Length >= SQL_INITIAL_CAPACITY)
                        {
                            cmd.CommandText = sql.ToString();
                            cmd.ExecuteNonQuery();
                            sql.Length = 0;
                        }
                    }
                    foreach (LootableGameObject obj in pLootableGameObjects)
                    {
                        areaId = 0;
                        if (obj.Area > 0)
                        {
                            areaName = pAreas[obj.Area - 1];
                            if (!AreaTableDBC.LookupId(pVersion,
                                                       pLocale,
                                                       areaName,
                                                       out areaId))
                            {
                                Logger.IgnoredAction(pUserId,
                                                     pHost,
                                                     "Update LootableGameObject",
                                                     "Unknown area \"{0}\".",
                                                     areaName);
                                continue;
                            }
                        }

                        if (!ResolveNameToId(sGameObjectPositionToId,                                             
                                             pLocale,
                                             obj.Name,
                                             areaId,
                                             (Int16)(obj.X / 100),
                                             (Int16)(obj.Y / 100),
                                             out objId))
                        {
                            WGOBObject found = pCache.Find(c => c.Name == obj.Name);
                            if (found == null)
                            {
                                Logger.IgnoredAction(pUserId,
                                                     pHost,
                                                     "Update LootableGameObject",
                                                     "Unconfirmed object name (also not found in cache file) \"{0}\".",
                                                     obj.Name);
                                continue;
                            }
                            objId = found.Id;
                        }
                        if (!ReadyForDB(sGameObjectLocationConfirmations,
                                        new CustomKey(pVersion,
                                                      pLocale,
                                                      objId,
                                                      areaId,
                                                      obj.X / 100,
                                                      obj.Y / 100),
                                        Config.Instance.MinConfirmations,
                                        pUserId,
                                        pVersion))
                        {
                            continue;
                        }

                        AddLocation(sGameObjectPositionToId,                                    
                                    pLocale,
                                    obj.Name,
                                    areaId,
                                    (Int16)(obj.X / 100),
                                    (Int16)(obj.Y / 100),
                                    objId);

                        sql.AppendFormat(sGameObjectLocationQueryFormat,
                                         pUserId,
                                         objId,
                                         areaId,
                                         obj.X / 100,
                                         obj.Y / 100,
                                         (Single)obj.X / 100,
                                         (Single)obj.Y / 100);

                        if (obj.MethodId > 0 &&
                            !SpellDBC.Exists(pVersion,
                                             pLocale,
                                             obj.MethodId))
                        {
                            continue;
                        }

                        insert = true;
                        foreach (Loot loot in obj.Loots)
                        {
                            insert = insert &&
                                     ReadyForDB(sGameObjectLootConfirmations,
                                                new CustomKey(pVersion,
                                                              pLocale,
                                                              objId,
                                                              loot.Id),
                                                Config.Instance.MinConfirmations,
                                                pUserId,
                                                pVersion);
                        }

                        if (!insert)
                        {
                            continue;
                        }

                        sql.AppendFormat(sGameObjectLootCountQueryFormat,
                                         pUserId,
                                         objId,                                         
                                         obj.MethodId);
                        
                        foreach (Loot loot in obj.Loots)
                        {
                            if (loot.Id == 0)
                            {
                                sql.AppendFormat(sGameObjectCoinQueryFormat,
                                                 loot.Quantity,
                                                 pUserId,
                                                 objId,                                                 
                                                 obj.MethodId);
                            }
                            else
                            {
                                sql.AppendFormat(sGameObjectLootQueryFormat,
                                                 loot.Quantity,
                                                 pUserId,
                                                 objId,                                                 
                                                 loot.Id,
                                                 obj.MethodId);
                            }
                        }

                        if (sql.Length >= SQL_INITIAL_CAPACITY)
                        {
                            cmd.CommandText = sql.ToString();
                            cmd.ExecuteNonQuery();
                            sql.Length = 0;
                        }
                    }
                    if (sql.Length > 0)
                    {
                        cmd.CommandText = sql.ToString();
                        cmd.ExecuteNonQuery();
                        sql.Length = 0;
                    }
                    trans.Commit();
                }
            }

#if ALPHA
            Logger.Log(ELogLevel.Debug,
                       pHost,
                       monitorTotal.ToString());
#endif
        }

        /**
         * Publically exposed method for updating items that give loot
         * 
         * @param  pUserId         the id of the user sending the update
         * @param  pHost           the host of the user updating in IP format
         * @param  pVersion        the wow version
         * @param  pLocale         the locale
         * @param  pLootableItems  the list of items which had loot contained in the update
         */
        public static void UpdateItems(Int32 pUserId,
                                       String pHost,
                                       UInt16 pVersion,
                                       ELocale pLocale,
                                       PackableList<LootableItem> pLootableItems)
        {
#if ALPHA
            CPUMonitor monitorTotal = new CPUMonitor("UpdateItems for {0} lootable entries", pLootableItems.Count);
#endif

            if (pLootableItems.Count == 0)
            {
                return;
            }

            Boolean insert;
            StringBuilder sql = new StringBuilder(SQL_INITIAL_CAPACITY);
            using (SqlConnection conn = new SqlConnection(Config.Instance.WoWDB))
            {
                conn.Open();
                using (SqlTransaction trans = conn.BeginTransaction())
                {
                    SqlCommand cmd = new SqlCommand();
                    cmd.Connection = conn;
                    cmd.Transaction = trans;

                    foreach (LootableItem item in pLootableItems)
                    {
                        if (item.MethodId > 0 &&
                            !SpellDBC.Exists(pVersion,
                                             pLocale,
                                             item.MethodId))
                        {
                            continue;
                        }

                        insert = true;
                        foreach (Loot loot in item.Loots)
                        {
                            insert = insert &&
                                     ReadyForDB(sItemLootConfirmations,
                                                new CustomKey(pVersion,
                                                              pLocale,
                                                              item.Id,
                                                              loot.Id),
                                                Config.Instance.MinConfirmations,
                                                pUserId,
                                                pVersion);
                        }

                        if (!insert)
                        {
                            continue;
                        }

                        sql.AppendFormat(sItemLootCountQueryFormat,
                                         pUserId,
                                         item.Id,                                         
                                         item.MethodId);

                        foreach (Loot loot in item.Loots)
                        {
                            if (loot.Id == 0)
                            {
                                sql.AppendFormat(sItemCoinQueryFormat,
                                                 loot.Quantity,
                                                 pUserId,
                                                 item.Id,
                                                 item.MethodId);
                            }
                            else
                            {
                                sql.AppendFormat(sItemLootQueryFormat,
                                                 loot.Quantity,
                                                 pUserId,
                                                 item.Id,                                                 
                                                 loot.Id,
                                                 item.MethodId);
                            }
                        }

                        if (sql.Length >= SQL_INITIAL_CAPACITY)
                        {
                            cmd.CommandText = sql.ToString();
                            cmd.ExecuteNonQuery();
                            sql.Length = 0;
                        }
                    }
                    if (sql.Length > 0)
                    {
                        cmd.CommandText = sql.ToString();
                        cmd.ExecuteNonQuery();
                        sql.Length = 0;
                    }
                    trans.Commit();
                }
            }
#if ALPHA
            Logger.Log(ELogLevel.Debug,
                       pHost,
                       monitorTotal.ToString());
#endif
        }

        /**
         * Publically exposed method for updating quests
         * 
         * @param  pUserId    the id of the user sending the update
         * @param  pHost      the host of the user updating in IP format
         * @param  pVersion   the wow version
         * @param  pLocale    the locale
         * @param  pQuests    the list of quests
         * @param  pFactions  the list of indexed factions to use for relating
         */
        public static void UpdateQuests(Int32 pUserId,
                                        String pHost,
                                        UInt16 pVersion,
                                        ELocale pLocale,
                                        PackableList<Quest> pQuests,
                                        PackableStringList pFactions)
        {

#if ALPHA
            CPUMonitor monitorTotal = new CPUMonitor("UpdateQuests for {0} entries", pQuests.Count);
#endif

            if (pQuests.Count == 0)
            {
                return;
            }

            String locale = pLocale.ToString();
            Int32 raceMask;
            Int32 classMask;
            String factionName;
            UInt16 factionId;
            StringBuilder sql = new StringBuilder(SQL_INITIAL_CAPACITY);
            using (SqlConnection conn = new SqlConnection(Config.Instance.WoWDB))
            {
                conn.Open();
                using (SqlTransaction trans = conn.BeginTransaction())
                {
                    SqlCommand cmd = new SqlCommand();
                    cmd.Connection = conn;
                    cmd.Transaction = trans;

                    foreach(Quest quest in pQuests)
                    {
                        if (!sQuestHash.Contains(new CustomKey(pVersion,
                                                               quest.Id)))
                        {
                            Logger.IgnoredAction(pUserId,
                                                 pHost,
                                                 "Update Quest",
                                                 "Unconfirmed id {0}",
                                                 quest.Id);
                            continue;
                        }

                        raceMask = 0;
                        classMask = 0;
                        quest.Races.ForEach(r => raceMask |= 1 << r);
                        quest.Classes.ForEach(c => classMask |= 1 << c);

                        string updated = "updated";
                        //Note: This intentionally does not update the "updated" column in the quest table unless it is actually expired:
                        if (IsEntityExpiredNoUpdate(sQuestHistory, new CustomKey(pVersion, pLocale, quest.Id), Config.Instance.QuestExpiration, pUserId))
                        {
                            updated = "GetUtcDate()";
                        }
                        
                        sql.AppendFormat(sQuestUpdateQueryFormat,
                                         locale,
                                         quest.MidText.Length == 0 ? "NULL" : EscapeAndQuote(quest.MidText),
                                         quest.EndText.Length == 0 ? "NULL" : EscapeAndQuote(quest.EndText),
                                         quest.LowestLevel == 0 ? "NULL" : quest.LowestLevel.ToString(),
                                         raceMask,
                                         classMask,
                                         quest.StartingIdType == 0 ? "NULL" : quest.StartingIdType.ToString(),
                                         quest.StartingId == 0 ? "NULL" : quest.StartingId.ToString(),
                                         quest.FinishingIdType == 0 ? "NULL" : quest.FinishingIdType.ToString(),
                                         quest.FinishingId == 0 ? "NULL" : quest.FinishingId.ToString(),
                                         quest.TimeToComplete,                                         
                                         pUserId,
                                         updated,
                                         quest.Id,
                                         pVersion);

                        if (quest.ExperienceGained > 0)
                        {
                            if (ReadyForDB(sQuestExperienceConfirmations,
                                           new CustomKey(pVersion,
                                                         pLocale,
                                                         quest.Id,
                                                         quest.LowestLevel,
                                                         quest.ExperienceGained),
                                           Config.Instance.MinConfirmations,
                                           pUserId,
                                           pVersion))
                            {
                                sql.AppendFormat(sQuestExperienceQueryFormat,
                                                 quest.ExperienceGained,
                                                 pUserId,
                                                 quest.Id,
                                                 pVersion,
                                                 quest.LowestLevel);
                            }
                        }

                        foreach (FactionChange change in quest.FactionChanges)
                        {
                            factionId = 0;
                            if (change.Faction > 0)
                            {
                                factionName = pFactions[change.Faction - 1];
                                if (!FactionDBC.LookupId(pVersion,
                                                         pLocale,
                                                         factionName,
                                                         out factionId))
                                {
                                    continue;
                                }
                            }
                            if (ReadyForDB(sQuestFactionConfirmations,
                                           new CustomKey(pVersion,
                                                         pLocale,
                                                         quest.Id,
                                                         factionId,
                                                         change.Amount),
                                           Config.Instance.MinConfirmations,
                                           pUserId,
                                           pVersion))
                            {
                                sql.AppendFormat(sQuestFactionQueryFormat,
                                                 change.Amount,
                                                 classMask,
                                                 raceMask,
                                                 pUserId,
                                                 quest.Id,
                                                 pVersion,
                                                 factionId);
                            }
                        }
                        if (sql.Length >= SQL_INITIAL_CAPACITY)
                        {
                            cmd.CommandText = sql.ToString();
                            cmd.ExecuteNonQuery();
                            sql.Length = 0;
                        }
                    }
                    if (sql.Length > 0)
                    {
                        cmd.CommandText = sql.ToString();
                        cmd.ExecuteNonQuery();
                        sql.Length = 0;
                    }
                    trans.Commit();
                }
            }

#if ALPHA
            Logger.Log(ELogLevel.Debug,
                       pHost,
                       monitorTotal.ToString());
#endif
        }

        /**
         * Private constants
         */
        private const Int32 SQL_INITIAL_CAPACITY = 1024 * 1024;

        /**
         * Static member data
         */
        private static HashSet<CustomKey> sCreatureHash =
            new HashSet<CustomKey>(new CustomKey.CustomKeyComparer());
        private static HashSet<CustomKey> sGameObjectHash =
            new HashSet<CustomKey>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, List<IdAndName>> sGameObjectPositionToId =
            new Dictionary<CustomKey, List<IdAndName>>(new CustomKey.CustomKeyComparer());
        private static HashSet<CustomKey> sItemHash =
            new HashSet<CustomKey>(new CustomKey.CustomKeyComparer());
        private static HashSet<CustomKey> sPageTextHash =
            new HashSet<CustomKey>(new CustomKey.CustomKeyComparer());
        private static HashSet<CustomKey> sQuestHash =
            new HashSet<CustomKey>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, Int32> sServerKeyToId =
            new Dictionary<CustomKey, Int32>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, Int32> sGuildKeyToId =
            new Dictionary<CustomKey, Int32>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, Int32> sPlayerKeyToId =
            new Dictionary<CustomKey, Int32>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, DateTime> sCreatureHistory =
            new Dictionary<CustomKey, DateTime>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, DateTime> sCreatureLuaHistory =
            new Dictionary<CustomKey, DateTime>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, DateTime> sGameObjectHistory =
            new Dictionary<CustomKey, DateTime>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, DateTime> sItemHistory =
            new Dictionary<CustomKey, DateTime>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, DateTime> sPageTextHistory =
            new Dictionary<CustomKey, DateTime>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, DateTime> sQuestHistory =
            new Dictionary<CustomKey, DateTime>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, DateTime> sNPCMerchandiseHistory =
            new Dictionary<CustomKey, DateTime>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, DateTime> sNPCMerchandiseCostHistory =
            new Dictionary<CustomKey, DateTime>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, DateTime> sNPCTrainingHistory =
            new Dictionary<CustomKey, DateTime>(new CustomKey.CustomKeyComparer());

        private static Dictionary<CustomKey, Confirmation> sCreatureConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());
        
        private static Dictionary<CustomKey, Confirmation> sGameObjectConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, Confirmation> sItemConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, Confirmation> sPageTextConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, Confirmation> sQuestConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());
        
        private static Dictionary<CustomKey, Confirmation> sCreatureLocationConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, Confirmation> sCreatureLootConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, Confirmation> sCreatureMerchandiseConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, Confirmation> sCreatureLimitedMerchandiseConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, Confirmation> sCreatureMerchandiseCostsConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, Confirmation> sCreatureTrainConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, Confirmation> sGameObjectLocationConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, Confirmation> sGameObjectLootConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, Confirmation> sItemLootConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, Confirmation> sQuestExperienceConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, Confirmation> sQuestFactionConfirmations =
            new Dictionary<CustomKey, Confirmation>(new CustomKey.CustomKeyComparer());

        private static DataTable sCreatureSchema = null;
        private static DataTable sGameObjectSchema = null;
        private static DataTable sItemSchema = null;
        private static DataTable sPageTextSchema = null;
        private static DataTable sQuestSchema = null;

        /**
         * Internal class for managing ID and Name pairs
         */
        private class IdAndName
        {
            public Int32 Id;
            public String Name;

            /**
             * Initialization constructor
             */
            public IdAndName(Int32 pId,
                             String pName)
            {
                Id = pId;
                Name = pName;
            }
        }
        /**
         * Internal class for managing confirmations
         */
        private class Confirmation
        {
            public Int32 UserId;
            public Int64 Count;

            /**
             * Initialization constructor
             */
            public Confirmation(Int32 pUserId)
            {
                UserId = pUserId;
                Count = 1;
            }
        }
        private static void AppendConfirmations(String pLabel, 
                                                ref StringBuilder pCurrentList,
                                                Dictionary<CustomKey,Confirmation> pConfirmations)
        {
            pCurrentList.AppendFormat("\n\n{0}\n", pLabel);
            lock (pConfirmations)
            {
                foreach (KeyValuePair<CustomKey, Confirmation> kvp in pConfirmations)
                {
                    pCurrentList.AppendFormat("\nUser: {1}, Count: {2}",
                                                    kvp.Key.ToString(),
                                                    kvp.Value.UserId,
                                                    kvp.Value.Count);

                }
            }
        }

        private static void CountConfirmations(String pLabel,
                                        ref StringBuilder pCurrentList,
                                        Dictionary<CustomKey, Confirmation> pConfirmations)
        {
            
            int count = 0;
            lock (pConfirmations)
            {
                foreach (KeyValuePair<CustomKey, Confirmation> kvp in pConfirmations)
                {
                    if (kvp.Value.Count == 1)
                    {
                        count++;
                    }
                }
            }
            pCurrentList.AppendFormat("\n{0}: {1}", pLabel, count);
        }

        public static string GetConfirmations()
        {

            StringBuilder ConfirmationList = new StringBuilder();            
            AppendConfirmations("Game Objects", ref ConfirmationList, sGameObjectConfirmations);
            AppendConfirmations("Game Object Locations", ref ConfirmationList, sGameObjectLocationConfirmations);
            AppendConfirmations("Game Object Loot", ref ConfirmationList, sGameObjectLootConfirmations);
            AppendConfirmations("Items", ref ConfirmationList, sItemConfirmations);
            AppendConfirmations("Item Loot", ref  ConfirmationList, sItemLootConfirmations);            
            AppendConfirmations("Quests", ref ConfirmationList, sQuestConfirmations);
            AppendConfirmations("Quest Experience", ref ConfirmationList, sQuestExperienceConfirmations);
            AppendConfirmations("Quest Faction", ref  ConfirmationList, sQuestFactionConfirmations);
            AppendConfirmations("Quest Page Text", ref ConfirmationList, sPageTextConfirmations);
            AppendConfirmations("Creatures", ref ConfirmationList, sCreatureConfirmations);
            AppendConfirmations("Creature Locations", ref ConfirmationList, sCreatureLocationConfirmations);
            AppendConfirmations("Creature Loot", ref ConfirmationList, sCreatureLootConfirmations);
            AppendConfirmations("Creature Merchandise", ref ConfirmationList, sCreatureMerchandiseConfirmations);
            AppendConfirmations("Creature Limited Merchandise", ref  ConfirmationList, sCreatureLimitedMerchandiseConfirmations);
            AppendConfirmations("Creature Merchandise Costs", ref  ConfirmationList, sCreatureMerchandiseCostsConfirmations);
            AppendConfirmations("Creature Training", ref  ConfirmationList, sCreatureTrainConfirmations);

            
            
            return ConfirmationList.ToString();            
        }

        public static string GetConfirmationCounts()
        {
            StringBuilder ConfirmationList = new StringBuilder();
            CountConfirmations("Game Objects", ref ConfirmationList, sGameObjectConfirmations);
            CountConfirmations("Game Object Locations", ref ConfirmationList, sGameObjectLocationConfirmations);
            CountConfirmations("Game Object Loot", ref ConfirmationList, sGameObjectLootConfirmations);
            CountConfirmations("Items", ref ConfirmationList, sItemConfirmations);
            CountConfirmations("Item Loot", ref  ConfirmationList, sItemLootConfirmations);
            CountConfirmations("Quests", ref ConfirmationList, sQuestConfirmations);
            CountConfirmations("Quest Experience", ref ConfirmationList, sQuestExperienceConfirmations);
            CountConfirmations("Quest Faction", ref  ConfirmationList, sQuestFactionConfirmations);
            CountConfirmations("Quest Page Text", ref ConfirmationList, sPageTextConfirmations);
            CountConfirmations("Creatures", ref ConfirmationList, sCreatureConfirmations);
            CountConfirmations("Creature Locations", ref ConfirmationList, sCreatureLocationConfirmations);
            CountConfirmations("Creature Loot", ref ConfirmationList, sCreatureLootConfirmations);
            CountConfirmations("Creature Merchandise", ref ConfirmationList, sCreatureMerchandiseConfirmations);
            CountConfirmations("Creature Limited Merchandise", ref  ConfirmationList, sCreatureLimitedMerchandiseConfirmations);
            CountConfirmations("Creature Merchandise Costs", ref ConfirmationList, sCreatureMerchandiseCostsConfirmations);
            CountConfirmations("Creature Training", ref  ConfirmationList, sCreatureTrainConfirmations);
            return ConfirmationList.ToString(); 
            
        }
    }
}
