﻿using Curse;
using System;
using System.Collections.Generic;
using System.IO;

namespace WoWDataCenter.DBC
{
    public static class SpellDBC
    {
        private static HashSet<CustomKey> sSpellHash =
            new HashSet<CustomKey>(new CustomKey.CustomKeyComparer());
        private static Dictionary<CustomKey, Int32> sSpellLookup =
            new Dictionary<CustomKey, Int32>(new CustomKey.CustomKeyComparer());

        public static Boolean Load()
        {
            CPUMonitor monitorTotal = new CPUMonitor("SpellDBC Loading");

            sSpellHash.Clear();
            if (!Array.TrueForAll<Config.SpellDBCSchema>(Config.Instance.SpellDBCSchemas,
                                                         s => Load(s)))
            {
                return false;
            }

            Logger.Log(ELogLevel.Debug,
                       null,
                       monitorTotal.ToString());
            return true;
        }
        private static Boolean Load(Config.SpellDBCSchema pSchema)
        {
           foreach(ELocale locale in pSchema.Locales)
           {
               String dbcPath = String.Format("{0}{1}{2}{3}{4}Spell.dbc",
                                           Config.Instance.DBCPath,
                                           pSchema.Version,
                                           Path.DirectorySeparatorChar,
                                           locale,
                                           Path.DirectorySeparatorChar);

                DBCReader dbcReader = DBCReader.Open(dbcPath);
                if (dbcReader == null)
                {
                    Logger.Log(ELogLevel.Error,
                               "Unable to open {0}",
                               dbcPath);
                    return false;
                }

                Int32 spellId;
                Int32 tempSpellId;
                Int32 lineId;
                UInt16 tempLineId;
                Int32 taughtId;
                Int32 nameOffset;
                Int32 rankOffset;
                String name;
                String rank;
                CustomKey key;
                
                while (dbcReader.NextRow())
                {
                    if (dbcReader.ReadField(pSchema.Id, out spellId) &&
                        dbcReader.ReadField(pSchema.LineId, out lineId) &&
                        dbcReader.ReadField(pSchema.TaughtId, out taughtId) &&
                        dbcReader.ReadField(pSchema.Name, out nameOffset) &&
                        dbcReader.ReadField(pSchema.Rank, out rankOffset) &&
                        dbcReader.ReadString(nameOffset, out name) &&
                        dbcReader.ReadString(rankOffset, out rank))
                    {
                        if (spellId > Int32.MaxValue)
                        {
                            continue;
                        }
                        key = new CustomKey(pSchema.Version,
                                            locale,
                                            spellId);
                        sSpellHash.Add(key);                        

                        if (taughtId != 0 && SkillLineAbilityDBC.LookupId(pSchema.Version,
                                                         locale,
                                                         taughtId,
                                                         out tempLineId))
                        {
                            spellId = taughtId;
                            lineId = tempLineId;
                        }
                        else if (SkillLineAbilityDBC.LookupId(pSchema.Version,
                                                        locale,
                                                        spellId,
                                                        out tempLineId))
                        {
                            lineId = tempLineId;
                        }
                        if (lineId == 0)
                        {
                            continue;
                        }

                        key = new CustomKey(pSchema.Version,
                                            locale,
                                            name,
                                            rank,
                                            lineId);

                        if (!sSpellLookup.TryGetValue(key, out tempSpellId) || spellId < tempSpellId)
                        {
                            sSpellLookup[key] = spellId;
                        }
                    }
                }
                dbcReader.Close();
            }
            return true;
        }

        public static Boolean Exists(UInt16 pVersion,
                                     ELocale pLocale,
                                     Int32 pSpellId)
        {
            return sSpellHash.Contains(new CustomKey(pVersion,
                                                  pLocale,
                                                  pSpellId));
        }
        public static Boolean LookupId(UInt16 pVersion,
                                       ELocale pLocale,
                                       String pName,
                                       String pRank,
                                       UInt16 pLineId,
                                       out Int32 pSpellId)
        {
            return sSpellLookup.TryGetValue(new CustomKey(pVersion,
                                                          pLocale,
                                                          pName,
                                                          pRank,
                                                          pLineId),
                                            out pSpellId);
        }
    }
}
