using System;
using System.IO;
using System.Text;

namespace WoWDataCenter
{
    /**
     * Generic WoW DBC Reader
     *
     * @author Shane Bryldt
     */
    public sealed class DBCReader
        : IDisposable
    {
        /**
         * Opens a DBC file and reads the header to validate it
         * 
         * @param  pPath  the path to the DBC file
         * @return        the newly created DBCReader, or null if it failed
         */
        public static DBCReader Open(String pPath)
        {
            if (!File.Exists(pPath))
            {
                return null;
            }

            DBCReader dbc = null;
            try
            {
                dbc = new DBCReader(pPath);
            }
            catch (Exception)
            {
                dbc = null;
            }
            return dbc;
        }

        /**
         * Private initialization constructor
         * 
         * @param  pPath  the path to the DBC fileder, or null if it failed
         */
        private DBCReader(String pPath)
        {
            try
            {
                FileStream stream = new FileStream(pPath, FileMode.Open, FileAccess.Read);
                mBinaryReader = new BinaryReader(stream);

                mSignature = Encoding.ASCII.GetString(mBinaryReader.ReadBytes(4));
                mRows = mBinaryReader.ReadInt32();
                mColumns = mBinaryReader.ReadInt32();
                mRowSize = mBinaryReader.ReadInt32();
                mStringsSize = mBinaryReader.ReadInt32();

                mRowStart = stream.Position;
                mStringsStart = mRowStart + (mRows * mRowSize);
                if (mStringsStart + mStringsSize > stream.Length)
                {
                    throw new OverflowException();
                }
            }
            catch (Exception exc)
            {
                Close();
                throw exc;
            }
        }

        /**
         * Read-only Signature property
         * 
         * @return  the DBC signature identifying what kind of records it contains
         */
        public String Signature
        {
            get
            {
                return mSignature;
            }
        }
        /**
         * Read-only Rows property
         * 
         * @return  the number of rows the DBC contains
         */
        public Int32 Rows
        {
            get
            {
                return mRows;
            }
        }
        /**
         * Read-only Columns property
         * 
         * @return  the number of columns for each row the DBC contains
         */
        public Int32 Columns
        {
            get
            {
                return mColumns;
            }
        }

        /**
         * Closes and cleans up the underlying binary reader
         */
        public void Close()
        {
            if (mBinaryReader != null)
            {
                mBinaryReader.Close();
                mBinaryReader = null;
            }
        }
        /**
         * Support for IDisposable
         */
        public void Dispose()
        {
            Close();
        }

        /**
         * Moves the internal cursor to the first field of the next row
         * 
         * @return  true if there is a row to read, false otherwise
         */
        public Boolean NextRow()
        {
            ++mRowIndex;
            if (mRowIndex >= mRows)
            {
                return false;
            }

            Int64 rowStart = mRowStart + (mRowIndex * mRowSize);
            if (mBinaryReader.BaseStream.Position != rowStart)
            {
                rowStart -= mBinaryReader.BaseStream.Position;
                rowStart = mBinaryReader.BaseStream.Seek(rowStart, SeekOrigin.Current);
            }
            return true;
        }

        /**
         * Moves the internal cursor to the indicated field by column index and reads it
         * 
         * @param  pColumn  the index of the field to read
         * @param  pValue   the out value to be assigned what the field contains
         * @return          true if the field could be read, false otherwise
         */
        public Boolean ReadField(Int32 pColumn,
                                 out Int32 pValue)
        {
            pValue = 0;
            if (mRowIndex >= mRows || pColumn >= mColumns)
            {
                return false;
            }

            Int64 rowStart = mRowStart + (mRowIndex * mRowSize);
            Int64 columnStart = rowStart + (pColumn * 4);
            if (mBinaryReader.BaseStream.Position != columnStart)
            {
                columnStart -= mBinaryReader.BaseStream.Position;
                columnStart = mBinaryReader.BaseStream.Seek(columnStart, SeekOrigin.Current);
            }
            pValue = mBinaryReader.ReadInt32();
            return true;
        }

        /**
         * Moves to the string table offset and reads a string, then restores original cursor location
         * 
         * @param  pOffset  the index into the string block to begin reading
         * @param  pValue   the out value to be assigned the string that is read
         * @return          true if the string could be read, false otherwise
         */
        public Boolean ReadString(Int32 pOffset,
                                  out String pValue)
        {
            pValue = null;
            if (pOffset >= mStringsSize ||
                mStringsStart + pOffset >= mBinaryReader.BaseStream.Length)
            {
                return false;
            }

            Int64 currentPosition = mBinaryReader.BaseStream.Position;
            Int64 stringStart = mStringsStart + pOffset;
            stringStart = mBinaryReader.BaseStream.Seek(stringStart, SeekOrigin.Begin);

            Char c = mBinaryReader.ReadChar();
            Boolean done = false;
            if (c == '\0')
            {
                pValue = "";
                done = true;
            }
            else
            {
                StringBuilder buf = new StringBuilder(128);
                buf.Append(c);

                while (!done &&
                       mBinaryReader.BaseStream.Position < mBinaryReader.BaseStream.Length)
                {
                    c = mBinaryReader.ReadChar();
                    if (c == '\0')
                    {
                        pValue = buf.ToString();
                        done = true;
                    }
                    else
                    {
                        buf.Append(c);
                    }
                }
            }
            mBinaryReader.BaseStream.Seek(currentPosition, SeekOrigin.Begin);
            return done;
        }

        /**
         * Member variables
         */
        private BinaryReader mBinaryReader = null;
        private String mSignature = null;
        private Int32 mRows = 0;
        private Int32 mColumns = 0;
        private Int32 mRowSize = 0;
        private Int32 mStringsSize = 0;

        private Int64 mRowStart = 0;
        private Int64 mStringsStart = 0;
        private Int32 mRowIndex = -1;
    }
}
