using Curse;
using Curse.Auth;
using Curse.WoW;
using Curse.WoW.WDB;
using WoWDataCenter.DBC;

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Configuration;
using System.IO;
using System.Security.Cryptography;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Net;
using System.Web;
using System.Web.Services;
using System.Web.Services.Protocols;
using System.Xml;

namespace WoWDataCenter
{
    /**
     * Hardware Front-End Web Service Class
     * Inherits AuthenticatableWebService
     *
     * @author Shane Bryldt
     */
    [WebService(Namespace = "http://wowservice.curse.com/")]
    [WebServiceBinding(ConformsTo = WsiProfiles.BasicProfile1_1)]
    [ToolboxItem(false)]
    public sealed class WoWDataService
        : AuthenticatableWebService
    {
        /**
         * Static constructor
         * Does one-time initialization of the wow data service
         */
        static WoWDataService()
        {
            sLocales.Add("enUS", new KeyValuePair<ELocale, Byte>(ELocale.en, 1));
            sLocales.Add("enGB", new KeyValuePair<ELocale, Byte>(ELocale.en, 2));
            sLocales.Add("koKR", new KeyValuePair<ELocale, Byte>(ELocale.kr, 3));
            sLocales.Add("frFR", new KeyValuePair<ELocale, Byte>(ELocale.fr, 2));
            sLocales.Add("deDE", new KeyValuePair<ELocale, Byte>(ELocale.de, 2));
            sLocales.Add("zhCN", new KeyValuePair<ELocale, Byte>(ELocale.cn, 4));
            sLocales.Add("zhTW", new KeyValuePair<ELocale, Byte>(ELocale.tw, 5));
            sLocales.Add("esES", new KeyValuePair<ELocale, Byte>(ELocale.es, 2));                  
            Reload();
        }

        /**
         * Support method for computing a SHA1 hash and returning a string of the ascii hex
         * 
         * @param  pData  the byte array of data to compute the hash for
         * @return        the hex string of the hash
         */
        private static String ComputeHash(Byte[] pData)
        {
            SHA1Managed sha1 = new SHA1Managed();
            Byte[] hash = sha1.ComputeHash(pData);
            return BitConverter.ToString(hash).Replace("-", "").ToLower();
        }

        /**
         * Support method for reloading the state of the service
         * Reloads config file, log file, DLL, SQL and DBC records
         * Restarts the update thread and server update thread
         */
        private static void Reload()
        {
            lock (sRebootLock)
            {
                if (sUpdateThread != null)
                {
                    sUpdateThread.Abort();
                    sUpdateThread.Join();
                    sUpdateThread = null;
                }
                if (sServerUpdateThread != null)
                {
                    sServerUpdateThread.Abort();
                    sServerUpdateThread.Join();
                    sServerUpdateThread = null;
                }
                
                sUpdateQueue.Clear();

                CPUMonitor monitorTotal = new CPUMonitor("Reload");

                if (!Config.Load())
                {
                    return;
                }

                
                sGameServerUpdateInterval = Config.Instance.GameServerUpdateInterval;
                Logger.SetLogLevel = Config.Instance.LogLevel;
                Logger.SetLogPath = Config.Instance.LogPath;
                Logger.SetAutoGrowthMegabytes = 10;
                sDLL = File.ReadAllBytes(Config.Instance.DLLFile);
                sDLLHash = ComputeHash(sDLL);

                sClientCipher = new StringCipher(Config.Instance.ClientKey);
                sAuthentication = new Authentication(Config.Instance.AuthenticationService,
                                                     Config.Instance.AuthenticationId,
                                                     Config.Instance.AuthenticationKey,
                                                     Config.Instance.SessionExpiration);
                
                if (!DB.Load())
                {
                    return;
                }

                if (!AreaTableDBC.Load())
                {
                    return;
                }
                if (!ChrClassesDBC.Load())
                {
                    return;
                }
                if (!ChrRacesDBC.Load())
                {
                    return;
                }
                if (!FactionDBC.Load())
                {
                    return;
                }
                if (!ItemDBC.Load())
                {
                    return;
                }
                if (!SkillLineDBC.Load())
                {
                    return;
                }
                if (!SkillLineAbilityDBC.Load())
                {
                    return;
                }
                if (!SkillRaceClassInfoDBC.Load())
                {
                    return;
                }
                if (!SpellDBC.Load())
                {
                    return;
                }
                DownloadServers();

                sUpdateThread = new Thread(ProcessUpdateThread);
                sUpdateThread.Priority = ThreadPriority.Highest;
                sUpdateThread.Start();

                sServerUpdateThread = new Thread(ProcessServerUpdateThread);
                sServerUpdateThread.Start();

                Logger.Log(ELogLevel.Debug,
                           null,
                           monitorTotal.ToString());
                Logger.Log(ELogLevel.Info,
                           null,
                           "Started");
            }
        }
/*
        private static Boolean GetUSServersOld(List<KeyValuePair<String, Byte>> pServers)
        {
            Byte region = sLocales["enUS"].Value;                    
            using (XmlReader reader = XmlReader.Create("http://www.worldofwarcraft.com/realmstatus/status.xml"))
            {

                if (!reader.ReadToDescendant("rs"))
                {
                    Logger.Log(ELogLevel.Debug,
                               null,
                               "Unable to find expected tree rs for US servers");
                    return false;
                }
                if (!reader.ReadToDescendant("r"))
                {
                    Logger.Log(ELogLevel.Debug,
                               null,
                               "Unable to find expected node r for US servers");
                    return false;
                }

                do
                {
                    if (!reader.MoveToAttribute("n"))
                    {
                        Logger.Log(ELogLevel.Debug,
                                   null,
                                   "Unable to find expected attribute n for US servers");
                        return false;
                    }
                    String name = reader.GetAttribute("n");
                    if (name.Length > 0 && pServers.FindIndex(kv => kv.Key == name && kv.Value == region) < 0)
                    {
                        pServers.Add(new KeyValuePair<String, Byte>(name, region));
                    }
                } while (reader.ReadToNextSibling("r"));
            }
            return true;
        }
 */

        private static String GetHTMLPage(String pURL)
        {
            String PageHtml = String.Empty;
            HttpWebRequest MyHttpWebRequest = (HttpWebRequest)WebRequest.Create(pURL);
            //Set the timeout to 5 seconds.
            MyHttpWebRequest.Timeout = 5000;
            MyHttpWebRequest.Accept = "text/xml,application/xml,application/xhtml+xml,text/html;q=0.9,text/plain;q=0.8,image/png,*/*;q=0.5";
            MyHttpWebRequest.UserAgent = "Mozilla/5.0 (Windows; U; Windows NT 5.0; en-GB; rv:1.8.1.4) Gecko/20070515 Firefox/2.0.0.4";
            try
            {
                WebResponse MyWebResponse = MyHttpWebRequest.GetResponse();
                StreamReader MyStreamReader = new StreamReader(MyWebResponse.GetResponseStream());
                PageHtml = MyStreamReader.ReadToEnd();
                MyWebResponse.Close();
                return PageHtml;
            }
            catch(Exception x)
            {
            }
            return "";
            
             
        }
        private static Boolean GetUSServers(List<KeyValuePair<String, Byte>> pServers)
        {
            Byte region = sLocales["enUS"].Value;
            string ServerStatusXml = GetHTMLPage("http://www.worldofwarcraft.com/realmstatus/status.xml");
            if (ServerStatusXml == String.Empty)
            {
                return false;
            }
            ServerStatusXml = ServerStatusXml.Replace("&nbsp;", " ");


            XmlDocument doc = new XmlDocument();
            doc.LoadXml(ServerStatusXml);
            XmlNodeList items = doc.GetElementsByTagName("r");
            foreach (XmlNode item in items)
            {                
                    String name = HttpContext.Current.Server.UrlDecode(item.Attributes["n"].Value);
                    Int32 index = pServers.FindIndex(kv => kv.Key == name && kv.Value == region);
                    if (name.Length > 0 && index < 0)
                    {
                        pServers.Add(new KeyValuePair<String, Byte>(name, region));
                    }                                            
            }
            return true;

        }
        private static Boolean GetEUServers(List<KeyValuePair<String, Byte>> pServers)
        {


            Byte region = sLocales["enGB"].Value;
            string ServerStatusXml = GetHTMLPage("http://www.wow-europe.com/en/serverstatus/index.xml");
            if (ServerStatusXml == String.Empty)
            {
                return false;
            }
            ServerStatusXml = ServerStatusXml.Replace("&nbsp;", " ");


            XmlDocument doc = new XmlDocument();            
            doc.LoadXml(ServerStatusXml);
            XmlNodeList items = doc.GetElementsByTagName("item");
            foreach (XmlNode item in items)
            {
                if(item.InnerXml.ToLower().Contains("domain=\"status\""))
                {

                    foreach(XmlNode child in item.ChildNodes )
                    {
                        if(child.Name == "title")
                        {
                            String name = HttpContext.Current.Server.UrlDecode(child.InnerText);
                            Int32 index = pServers.FindIndex(kv => kv.Key == name && kv.Value == region);
                            if (name.Length > 0 && index < 0)
                            {
                                pServers.Add(new KeyValuePair<String, Byte>(name, region));
                            }
                            continue;
                        }
                    }
                                        
                }                
            }          
            return true;            
            
        }
        private static Boolean GetChineseServers(List<KeyValuePair<String, Byte>> pServers)
        {
            Byte region = sLocales["zhCN"].Value;
            Regex chineseServer = new Regex("<td width=\"30%\" valign=\"top\"><p>(?<Name>.*?)</p></td>", RegexOptions.Compiled);
            string pageMask = "http://status.wowchina.com/server/realmlist{0}.htm";
            int numPages = 8;
            string serverStatusHtml ="";
            for (int i = 1; i <= numPages; i++)
            {
                serverStatusHtml += GetHTMLPage(String.Format(pageMask,i.ToString()));
            }
             
            if (serverStatusHtml == String.Empty)
            {
                return false;
            }

            MatchCollection serverMatches = chineseServer.Matches(serverStatusHtml);
            foreach (Match serverMatch in serverMatches)
            {
                string name = serverMatch.Groups["Name"].Value;
                pServers.Add(new KeyValuePair<String, Byte>(name, region));
            }

            return true;

        }

        /**
         * Support method for downloading the latest server lists
         * 
         * @param  pData  the byte array of data to compute the hash for
         * @return        the hex string of the hash
         */
        private static Boolean DownloadServers()
        {
            sLastServersUpdate = DateTime.UtcNow;
            CPUMonitor monitor = new CPUMonitor("DownloadServers");
            List<KeyValuePair<String, Byte>> servers = new List<KeyValuePair<String, Byte>>();

            Logger.Log(ELogLevel.Info,
                       null,
                       "DownloadServers initiated");

            /*
            if (!GetChineseServers(servers))
            {
                return false;
            }

            if (!GetEUServers(servers))
            {
                return false;
            }
            */

            if (!GetUSServers(servers))
            {
                return false;
            }
         
            
            DB.AddServers(servers);
                      

            Logger.Log(ELogLevel.Debug,
                       null,
                       monitor.ToString());
            return true;
        }
          /**
         * Publically exposed web method for rebooting the service
         * Returns a human readable string for status in the response stream
         * 
         * @param pAdminKey  the secret administration key
         */
        [WebMethod]
        public void GetStats(String pAdminKey)
        {
            Byte[] buf;
            if (pAdminKey.ToLower() != Config.Instance.AdminKey.ToLower())
            {
                Logger.Log(ELogLevel.Error,
                           Context.Request.UserHostAddress,
                           "GetStats attempted with invalid admin key");
                buf = Encoding.ASCII.GetBytes("Invalid admin key, attempt has been logged");
                Context.Response.OutputStream.Write(buf, 0, buf.Length);
                return;
            }
            int queueCount = 0;

            lock (sUpdateQueue)
            {
                queueCount = sUpdateQueue.Count;
            }

            int sessionCount = sAuthentication.GetSessionCount();            
            long memoryUsed = System.Diagnostics.Process.GetCurrentProcess().WorkingSet64 / (1024 * 1024);
            string confirmationInfo = DB.GetConfirmationCounts();
            Context.Response.Write(String.Format("Queue Count: {0}\n\nMemory Usage: {1}\n\nSession Count: {2}\n\nConfirmations: {3}", 
                queueCount.ToString(),
                memoryUsed.ToString(),
                sessionCount.ToString(),
                confirmationInfo));


        }

        [WebMethod]
        public void ReloadDLL(String pAdminKey)
        {
            Byte[] buf;
            try
            {
                if (pAdminKey.ToLower() != Config.Instance.AdminKey.ToLower())
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "Reload DLL attempted with invalid admin key");
                    buf = Encoding.ASCII.GetBytes("Invalid admin key, attempt has been logged");
                }
                else
                {
                    Logger.Log(ELogLevel.Info,
                               Context.Request.UserHostAddress,
                               "Reload DLL initiated");

                    lock (sDLL)
                    {
                        lock (sDLLHash)
                        {
                            sDLL = File.ReadAllBytes(Config.Instance.DLLFile);
                            sDLLHash = ComputeHash(sDLL);
                        }
                    }
                    buf = Encoding.ASCII.GetBytes("DLL Reloaded");
                }
            }
            catch (Exception exc)
            {
                Logger.Log(ELogLevel.Error,
                           Context.Request.UserHostAddress,
                           "Reload DLL Exception: {0}",
                           exc.Message);
                buf = Encoding.ASCII.GetBytes(exc.Message);
            }

            Context.Response.OutputStream.Write(buf, 0, buf.Length);
            return;
        }

         
        /**
         * Publically exposed web method for rebooting the service
         * Returns a human readable string for status in the response stream
         * 
         * @param pAdminKey  the secret administration key
         */
        [WebMethod]
        public void Reboot(String pAdminKey)
        {
            Byte[] buf;
            try
            {
                if (pAdminKey.ToLower() != Config.Instance.AdminKey.ToLower())
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "Reboot attempted with invalid admin key");
                    buf = Encoding.ASCII.GetBytes("Invalid admin key, attempt has been logged");
                }
                else
                {
                    Logger.Log(ELogLevel.Info,
                               Context.Request.UserHostAddress,
                               "Reboot initiated");
                    CPUMonitor monitor = new CPUMonitor("Reboot");
                    Reload();
                    Logger.Log(ELogLevel.Debug,
                               null,
                               monitor.ToString());
                    buf = Encoding.ASCII.GetBytes(monitor.ToString());
                }
            }
            catch (Exception exc)
            {
                Logger.Log(ELogLevel.Error,
                           Context.Request.UserHostAddress,
                           "Reboot Exception: {0}",
                           exc.Message);
                buf = Encoding.ASCII.GetBytes(exc.Message);
            }

            Context.Response.OutputStream.Write(buf, 0, buf.Length);
            return;
        }

        /**
         * Publically exposed web method for pulling and updating the latest server lists manually
         * Returns a human readable string for status in the response stream
         * 
         * @param pAdminKey  the secret administration key
         */
        [WebMethod]
        public void ServersUpdate(String pAdminKey)
        {
            Byte[] buf;
            try
            {
                if (pAdminKey.ToLower() != Config.Instance.AdminKey.ToLower())
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "ServersUpdate attempted with invalid admin key");
                    buf = Encoding.ASCII.GetBytes("Invalid admin key, attempt has been logged");
                }
                else
                {
                    CPUMonitor monitor = new CPUMonitor("ServersUpdate");
                    if (!DownloadServers())
                    {
                        buf = Encoding.ASCII.GetBytes("An error occured while attempting to download the servers lists.");
                    }
                    else
                    {
                        buf = Encoding.ASCII.GetBytes(monitor.ToString());
                    }
                    Logger.Log(ELogLevel.Debug,
                               null,
                               monitor.ToString());
                }
            }
            catch (Exception exc)
            {
                Logger.Log(ELogLevel.Error,
                           Context.Request.UserHostAddress,
                           "ServersUpdate Exception: {0}",
                           exc.Message);
                buf = Encoding.ASCII.GetBytes(exc.Message);
            }

            Context.Response.OutputStream.Write(buf, 0, buf.Length);
            return;
        }

        /**
         * Publically exposed web method for obtaining a hash of the latest DLL
         * Returns a StatusCode in the response stream
         * Returns the a SHA1 hash made into a hex string, then converted to a byte array in the response stream
         * 
         * @param pSession  the session of the user
         */
        [WebMethod]
        public void GetDLLHash(String pSession)
        {
            try
            {
                Int32 userId;
                StatusCode status = sAuthentication.ValidateSessionKeyToUserId(pSession, out userId);
#if ALPHA
                status = StatusCode.Ok;
                userId = 1;
#endif
                if (status != StatusCode.Ok)
                {
                    Logger.Log(ELogLevel.Access,
                               Context.Request.UserHostAddress,
                               "ValidateSessionKeyToUserId failed with status {0}",
                               status);
                    Context.Response.OutputStream.WriteByte((Byte)status);
                    return;
                }

                Byte[] hashBytes = Encoding.ASCII.GetBytes(sDLLHash);

                Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);
                Context.Response.OutputStream.Write(hashBytes, 0, hashBytes.Length);
                return;
            }
            catch (Exception exc)
            {
                Logger.Log(ELogLevel.Error,
                           Context.Request.UserHostAddress,
                           "GetDLLHash Exception: {0}",
                           exc.Message);
            }

            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Unknown);
            return;
        }

        /**
         * Publically exposed web method for downloading the latest DLL
         * Returns a StatusCode in the response stream
         * Returns the DLL bytes in the response stream
         * 
         * @param pSession  the session of the user
         */
        [WebMethod]
        public void DownloadDLL(String pSession)
        {
            try
            {
                Int32 userId;
                StatusCode status = sAuthentication.ValidateSessionKeyToUserId(pSession, out userId);
#if ALPHA
                status = StatusCode.Ok;
                userId = 1;
#endif
                if (status != StatusCode.Ok)
                {
                    Logger.Log(ELogLevel.Access,
                               Context.Request.UserHostAddress,
                               "ValidateSessionKeyToUserId failed with status {0}",
                               status);
                    Context.Response.OutputStream.WriteByte((Byte)status);
                    return;
                }

                Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);
                Context.Response.OutputStream.Write(sDLL, 0, sDLL.Length);
                return;
            }
            catch (Exception exc)
            {
                Logger.Log(ELogLevel.Error,
                           Context.Request.UserHostAddress,
                           "DownloadDLL Exception: {0}",
                           exc.Message);
            }

            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Unknown);
            return;
        }

        [WebMethod]
        public void ListConfirmations(string pAdminKey)
        {
            if (pAdminKey.ToLower() != Config.Instance.AdminKey.ToLower())
            {
                Logger.Log(ELogLevel.Error,
                           Context.Request.UserHostAddress,
                           "List confirmation attempted with invalid admin key");                

                Context.Response.Write("Invalid admin key, attempt has been logged");
                return;
            }
            Context.Response.Write(DB.GetConfirmations());           

        }

        /**
         * Publically exposed web method for obtaining a hash of the latest DLL
         * Parameters are taken through form fields, expects a form posted file
         * Returns a StatusCode in the response stream
         */
        [WebMethod]
        public void UploadUpdate()
        {            
            try
            {

#if ALPHA
                //session = Context.Request.QueryString["pSession"];
                String session = "1";
#else
                String session = Context.Request.Form["pSession"];
#endif

                if (Context.Request.Files.Count == 0)
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "No file to read from the stream");
                    Context.Response.OutputStream.WriteByte((Byte)StatusCode.InvalidStream);
                    return;
                }

                Int32 userId;
                StatusCode status = sAuthentication.ValidateSessionKeyToUserId(session, out userId);
#if ALPHA
                status = StatusCode.Ok;
                userId = 1;
#endif
                if (status != StatusCode.Ok)
                {
                    Logger.Log(ELogLevel.Access,
                               Context.Request.UserHostAddress,
                               "ValidateSessionKeyToUserId failed with status {0}",
                               status);
                    Context.Response.OutputStream.WriteByte((Byte)status);
                    return;
                }

                Logger.Log(ELogLevel.Debug,
                           Context.Request.UserHostAddress,
                           "Received {0} bytes for update file stream",
                           Context.Request.Files[0].InputStream.Length);
                Package buf = new Package(Encoding.UTF8, Context.Request.Files[0].InputStream);

#if ALPHA
                buf.DumpBinaryToFile(Config.Instance.DumpPath + userId.ToString() + "_" + DateTime.Now.Ticks + ".txt");
#endif
                
                  //buf.DumpToFile(Config.Instance.DumpPath + userId.ToString() + ".txt",
                  //             DateTime.Now.ToString("yyyy-MM-dd-HHmmss") + "] Compressed is " + buf.Length + " bytes");
                

                if (!buf.Decompress())
                {
                    Logger.Log(ELogLevel.Error,
                               Context.Request.UserHostAddress,
                               "Invalid compressed update file stream");
                    Context.Response.OutputStream.WriteByte((Byte)StatusCode.InvalidStream);
                    return;
                }


                //buf.DumpToFile(Config.Instance.DumpPath + userId.ToString() + ".txt",
                    //DateTime.Now.ToString("yyyy-MM-dd-HHmmss") + "] Decompressed is " + buf.Length + " bytes");


                Update update = new Update(Config.Instance.MinProfilerVersion);                
                
                if (!update.Read(buf))
                {

                    switch (update.UpdateStatus)
                    {
                        case StatusCode.InvalidStream:
                            Logger.Log(ELogLevel.Error,
                                  Context.Request.UserHostAddress,
                                  "Invalid update file stream");
                            break;
                        case StatusCode.InvalidGameClientVersion:
                            Logger.Log(ELogLevel.Error,
                                Context.Request.UserHostAddress,
                                "Unsupported game client version: " + update.ClientVersion);
                            break;
                        case StatusCode.InvalidProfilerVersion:
                            Logger.Log(ELogLevel.Error,
                                Context.Request.UserHostAddress,
                                "Out of date profiler version: " + update.ProfilerVersion);
                            break;

                    }
                    Context.Response.OutputStream.WriteByte((Byte)update.UpdateStatus);
                    return;
                }               
                              
                Logger.Log(ELogLevel.Debug,
                           Context.Request.UserHostAddress,
                           "Processing {0} update with profiler version {1} and cache version {2}",
                           update.Locale.ToUpper(),
                           update.ProfilerVersion,
                           update.CacheVersion);

                
                lock (sUpdateQueue)
                {
                     // Do not clear queue for trusted users
                    if (sUpdateQueue.Count >= Config.Instance.MaxQueuedUpdates && 
                        !Array.Exists<int>(Config.Instance.Trusted, i => i == userId))
                    {
                        update.CreatureCache.Clear();
                        update.GameObjectCache.Clear();
                        update.ItemCache.Clear();
                        update.PageTextCache.Clear();
                        update.QuestCache.Clear();

                        update.Areas.Clear();
                        update.Factions.Clear();
                        update.NPCs.Clear();
                        update.QuestGameObjects.Clear();
                        update.LootableGameObjects.Clear();
                        update.LootableItems.Clear();
                        update.Quests.Clear();

                        Logger.IgnoredAction(userId,
                                             Context.Request.UserHostAddress,
                                             "Update",
                                             "Update queue has exceeded maximum capacity, dropped all data except player profiles");
                    }
                    sUpdateQueue.Enqueue(new UpdateThreadParam(userId,
                                                               Context.Request.UserHostAddress,
                                                               update));
                }               

                Context.Response.OutputStream.WriteByte((Byte)StatusCode.Ok);
                return;
            }
            catch (Exception exc)
            {
                Logger.Log(ELogLevel.Error,
                           Context.Request.UserHostAddress,
                           "UploadUpdate Exception: {0}",
                           exc.Message);
                Logger.Log(ELogLevel.Error,
                           Context.Request.UserHostAddress,
                           exc.StackTrace);
            }

            Context.Response.OutputStream.WriteByte((Byte)StatusCode.Unknown);
            return;
        }

        /**
         * Private update processing thread
         */
        private static void ProcessUpdateThread()
        {
            Boolean aborted = false;
            UpdateThreadParam param;
            while (!aborted)
            {
                try
                {
                    param = null;
                    lock (sUpdateQueue)
                    {
                        if (sUpdateQueue.Count > 0)
                        {
                            param = sUpdateQueue.Dequeue();
                        }
                    }

                    if (param == null)
                    {
                        Thread.Sleep(1000);
                        continue;
                    }

                    lock (sRebootLock)
                    {
                        Int32 userId = ((UpdateThreadParam)param).ID;
                        String userHost = ((UpdateThreadParam)param).Host;
                        Update update = ((UpdateThreadParam)param).Update;
                        CPUMonitor monitorTotal = new CPUMonitor("ProcessUpdateThread for User {0}", userId);

                        if (update.CacheVersion == 0)
                        {
                            Logger.IgnoredAction(userId, userHost, "Update", "Cache version is 0");
                            continue;
                        }

                        DateTime now = DateTime.UtcNow;
                        DateTime lastupdate;
                        if (!sUpdateTimes.TryGetValue(userId, out lastupdate) ||
                            now.Subtract(lastupdate).TotalSeconds >
                            Config.Instance.RateLimit ||
                            Array.Exists<int>(Config.Instance.Trusted, i => i == userId))
                        {
                            sUpdateTimes[userId] = DateTime.UtcNow;
                        }
                        else
                        {
                            Logger.IgnoredAction(userId, userHost, "Update", "Exceeded update rate limit");
#if !ALPHA
                            continue;
#endif
                        }

                        KeyValuePair<ELocale, Byte> kv;
                        if (!sLocales.TryGetValue(update.Locale, out kv))
                        {
                            Logger.IgnoredAction(userId, userHost, "Update", "Unknown locale {0}", update.Locale);
                            continue;
                        }
                        ELocale locale = kv.Key;
                        Byte region = kv.Value;


                        //MGC: Do not validate data from trusted users.
                        if (!IsTrustedUser(userId))
                        {

                            Boolean skipUpdate = false;
                            switch (locale)
                            {
                                case ELocale.en:
                                case ELocale.fr:
                                case ELocale.de:
                                case ELocale.es:
                                    foreach (Player player in update.Players)
                                    {
                                        if (!DB.ValidateServer(ref player.Server, region))
                                        {
                                            Logger.IgnoredAction(userId,
                                                                 userHost,
                                                                 "Update", "Unknown server {0} for region {1} in update",
                                                                 player.Server,
                                                                 region);
                                            skipUpdate = true;
                                            break;
                                        }
                                    }
                                    if (skipUpdate)
                                    {
                                        continue;
                                    }
                                    break;
                                default:
                                    break;
                            }

                            if (!Array.Exists<String>(Config.Instance.Whitelist, s => s.ToLower() == update.Realmlist.ToLower()))
                            {
                                Logger.IgnoredAction(userId, userHost, "Update", "Unknown realmlist {0}", update.Realmlist);
                                continue;
                            }

                            //Validate the creature cache, for max ID's
                            Int32 maxCreatureId = 0;
                            update.CreatureCache.ForEach(c => maxCreatureId = Math.Max(maxCreatureId, c.Id));
                            if (maxCreatureId > Config.Instance.MaxCreatureId)
                            {
                                Logger.IgnoredAction(userId,
                                                     userHost,
                                                     "Update",
                                                     "Invalid creature {0}",
                                                     maxCreatureId);
                                continue;
                            }

                            //Validate the item cache, for max ID's

                            // Do not do this for trusted users, until we figure out the implications of the Item.dbc changes
                            if (!Array.Exists<int>(Config.Instance.Trusted, i => i == userId))
                            {
                                foreach (WIDBItem item in update.ItemCache)
                                {
                                    if (!ItemDBC.Exists(update.CacheVersion,
                                                        locale,
                                                        item.Id,
                                                        item.InventorySlot))
                                    {
                                        skipUpdate = true;
                                        Logger.IgnoredAction(userId,
                                                             userHost,
                                                             "Update",
                                                             "Invalid item {0}, slot: {1}",
                                                             item.Id, item.InventorySlot);
                                        break;
                                    }
                                }                                
                            }
                            if (skipUpdate)
                            {
                                continue;
                            }
                        }

                        DB.UpdateCachedCreatures(userId,
                                                 userHost,
                                                 update.CacheVersion,
                                                 locale,
                                                 update.CreatureCache);
                        DB.UpdateCachedGameObjects(userId,
                                                   userHost,
                                                   update.CacheVersion,
                                                   locale,
                                                   update.GameObjectCache);
                        DB.UpdateCachedItems(userId,
                                             userHost,
                                             update.CacheVersion,
                                             locale,
                                             update.ItemCache);
                        DB.UpdateCachedPageTexts(userId,
                                                 userHost,
                                                 update.CacheVersion,
                                                 locale,
                                                 update.PageTextCache);
                        DB.UpdateCachedQuests(userId,
                                              userHost,
                                              update.CacheVersion,
                                              locale,
                                              update.QuestCache);

                        DB.UpdatePlayers(userId,
                                         userHost,
                                         update.CacheVersion,
                                         locale,
                                         update.Players,
                                         update.Factions,
                                         region);

                        DB.UpdateNPCs(userId,
                                      userHost,
                                      update.CacheVersion,
                                      locale,
                                      update.NPCs,
                                      update.Areas,
                                      update.Factions);

                        DB.UpdateGameObjects(userId,
                                             userHost,
                                             update.CacheVersion,
                                             locale,
                                             update.QuestGameObjects,
                                             update.LootableGameObjects,
                                             update.GameObjectCache,
                                             update.Areas);

                        DB.UpdateItems(userId,
                                       userHost,
                                       update.CacheVersion,
                                       locale,
                                       update.LootableItems);

                        DB.UpdateQuests(userId,
                                        userHost,
                                        update.CacheVersion,
                                        locale,
                                        update.Quests,
                                        update.Factions);

                        Logger.Log(ELogLevel.Debug,
                                   userHost,
                                   monitorTotal.ToString());
                    }
                }
                catch (ThreadAbortException tae)
                {
                    aborted = true;
                    sServerUpdateThread.Join(100);
                    Logger.Log(ELogLevel.Debug,
                               null,
                               "ProcessUpdateThread Aborted: {0}", tae.Message + "\n" + tae.StackTrace);
                }
                catch (Exception exc)
                {
                    Logger.Log(ELogLevel.Debug,
                               null,
                               "ProcessUpdateThread Exception: {0}",
                               exc.Message);
                    Logger.Log(ELogLevel.Debug,
                               null,
                               exc.StackTrace);
                }
            }
        }


        private static bool IsTrustedUser(Int32 pUserId)
        {
            return Array.Exists<int>(Config.Instance.Trusted, i => i == pUserId);
        }

        /**
         * Private server update processing thread
         */
        private static void ProcessServerUpdateThread()
        {
            Boolean aborted = false;
            TimeSpan delay = TimeSpan.FromSeconds(Convert.ToDouble(Config.Instance.ServerUpdateDelay));
            while (!aborted)
            {
                try
                {
                    if (DateTime.UtcNow.Subtract(sLastServersUpdate) >= delay)
                    {
                        DownloadServers();
                    }
                    Thread.Sleep(sGameServerUpdateInterval);
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    Logger.Log(ELogLevel.Debug,
                               null,
                               "ProcessServerUpdateThread Aborted");
                }
                catch (Exception exc)
                {
                    Logger.Log(ELogLevel.Debug,
                               null,
                               "ProcessServerUpdateThread Exception: {0}",
                               exc.Message);
                    Logger.Log(ELogLevel.Debug,
                               null,
                               exc.StackTrace);
                }
            }
        }

        /**
         * Private static member data
         */
        private static Dictionary<String, KeyValuePair<ELocale, Byte>> sLocales = new Dictionary<String, KeyValuePair<ELocale, Byte>>();        
        private static Object sRebootLock = new Object();
        private static Thread sUpdateThread = null;
        private static Byte[] sDLL = null;
        private static String sDLLHash = null;
        private static Queue<UpdateThreadParam> sUpdateQueue = new Queue<UpdateThreadParam>();
        private static Dictionary<Int32, DateTime> sUpdateTimes = new Dictionary<Int32, DateTime>();
        private static Thread sServerUpdateThread = null;
        private static DateTime sLastServersUpdate = DateTime.MinValue;
        private static Int32 sGameServerUpdateInterval = 60000;        

        /**
         * Internal class for passing update thread parameters
         */
        private class UpdateThreadParam
        {
            public Int32 ID;
            public String Host;
            public Update Update;

            /**
             * Initialization constructor
             */
            public UpdateThreadParam(Int32 pId,
                                     String pHost,
                                     Update pUpdate)
            {
                ID = pId;
                Host = pHost;
                Update = pUpdate;
            }
        }
    }
}
