﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Reflection;
using System.Text;
using Curse.TypeScriptSharp.Model;

namespace Curse.TypeScriptSharp
{
    public class CSharpHelper : CodeGenerator, ICodeGenerator
    {

        private readonly HashSet<string> _overrideImports;
        private readonly HashSet<string> _sharedContractNamespaces;
        private readonly string _modelNamespace;

        public CSharpHelper(string modelNamespace, HashSet<string> overrideImports, HashSet<string> sharedContractNamespaces)
        {
            _overrideImports = overrideImports;
            _modelNamespace = modelNamespace;
            _sharedContractNamespaces = sharedContractNamespaces;
        }


        public HashSet<string> OverrideImports
        {
            get { return _overrideImports; }
        }

        public string ModelNamespace
        {
            get { return _modelNamespace; }
        }

        public string FileExtension
        {
            get { return ".cs"; }
        }

        public bool ExportAllKnownTypes { get { return true; } }

        private static readonly Dictionary<string, string> SystemTypeMap = new Dictionary<string, string>()
        {
            {"Int32", "int"},
            {"Int64", "long"},
            {"String", "string"},
            {"Guid", "Guid"},
            {"DateTime", "DateTime"},
            {"Boolean", "bool"},
            {"int", "int"},
            {"long", "long"},
            {"string", "string"},                        
            {"bool", "bool"},
            {"CipherAlgorithmType", "int"}
        };

        public string GetUrlToken(string paramName, int ordinal)
        {
            return "{" + ordinal + "}";
        }

        public string GetRpcName(string methodName)
        {
            return methodName;
        }

        public void WriteEnum(StringBuilder sb, Type type)
        {
            
            // To Dictionary
            // Start Enum Def
            sb.AppendFormat("public enum {0} {{", type.Name);
            sb.AppendLine();

            foreach (var fi in type.GetFields(BindingFlags.Public | BindingFlags.Static))
            {
                sb.Append("\t");
                sb.AppendFormat("{0} = {1},", fi.Name, Convert.ToInt64(fi.GetRawConstantValue()));
                sb.AppendLine();
            }

            // Close Enum Def
            sb.Append("}");
            sb.AppendLine();
        }

        public void WriteImport(StringBuilder sb, string importNamespace)
        {
            sb.AppendFormat("using {0};", importNamespace);
            sb.AppendLine();
        }

        public void WriteModuleHeader(StringBuilder sb, string moduleNamespace, string moduleName)
        {            
            sb.AppendFormat("namespace {0}", _modelNamespace).AppendLine();
            sb.Append("{").AppendLine();            
        }

        public void WriteModuleFooter(StringBuilder sb)
        {                        
            sb.Append("}").AppendLine();
        }
        
        private static readonly HashSet<string>  _processedTypes = new HashSet<string>();

        public override void WriteInterface(StringBuilder sb, InterfaceDefinition definition)
        {
            if (_processedTypes.Contains(definition.Name) && !_sharedContractNamespaces.Contains(definition.Namespace))
            {
                Console.WriteLine("Skipping type: " + definition.Name);
                return;
            }

            _processedTypes.Add(definition.Name);
            // Start Class Def
            sb.AppendFormat("public class {0}", definition.Name);
            sb.Append("{");
            sb.AppendLine();                        
            
            // Properties
            foreach (var prop in definition.Properties)
            {
                sb.Append("\t");
                sb.AppendFormat("public {0} {1} {{ get; set; }}", prop.TypeName, prop.Name);
                sb.AppendLine();
            }

            // Close Class Def
            sb.Append("}");
            sb.AppendLine();
        }

        public void WriteImports(HashSet<Type> knownTypes, StringBuilder sb, string excludeNamespace = null, string path = "models/")
        {
            if (_overrideImports == null)
            {
                return;                
            }

            var imports = _overrideImports.Select(p => new ImportDefinition {Name = p, Path = p}).ToArray();
            WriteImports(sb, imports, path);
                        
        }

        public void WriteImports(StringBuilder sb, IEnumerable<ImportDefinition> imports, string path = "models/")
        {
            foreach (var import in imports)
            {
                sb.AppendFormat("using {0};", import.Name).AppendLine();
            }
        }

        public void WriteOperation(StringBuilder methodBuilder, OperationDefinition operationDefinition)
        {
            throw new NotImplementedException();
        }

        public override string GetTypeName(Type outerType, HashSet<Type> knownTypes, Type propertyType)
        {
            if (SystemTypeMap.ContainsKey(propertyType.Name))
            {
                return SystemTypeMap[propertyType.Name];
            }

            if (propertyType.IsArray) // Arrays
            {

                var innerType = propertyType.GetElementType();                
                var inner = GetTypeName(outerType, knownTypes, innerType);                
                return inner + "[]";
                
            }
            
            if (propertyType.IsGenericType && propertyType.GetGenericTypeDefinition().Name.StartsWith("Dictionary`")) // Dictionaries
            {
                var keyType = propertyType.GetGenericArguments()[0];
                var elementType = propertyType.GetGenericArguments()[1];

                return string.Format("Dictionary<{0}, {1}> ", GetTypeName(outerType, knownTypes, keyType), GetTypeName(outerType, knownTypes, elementType)); // TODO: Make this strongly typed
            }
            
            if (propertyType.IsGenericType && (propertyType.GetGenericTypeDefinition().Name.StartsWith("HashSet`")
                || propertyType.GetGenericTypeDefinition().Name.StartsWith("ICollection`")
                || propertyType.GetGenericTypeDefinition().Name.StartsWith("List`")
                || propertyType.GetGenericTypeDefinition().Name.StartsWith("IEnumerable`")
                || propertyType.GetGenericTypeDefinition().Name.StartsWith("IReadOnlyCollection`")
                )) // Lists
            {
                var listType = propertyType.GetGenericArguments()[0];
                var typeScriptType = GetTypeName(outerType, knownTypes, listType);
                return typeScriptType + "[]"; // TODO: Make this strongly typed
            }

            if (propertyType.IsGenericType && propertyType.GetGenericTypeDefinition().Name.StartsWith("Nullable`"))
            {
                var nullableType = propertyType.GetGenericArguments()[0];
                var typeScriptType = GetTypeName(outerType, knownTypes, nullableType);
                return typeScriptType + "?";                
            }

            if (propertyType.IsGenericType)
            {
                var typeArgs = propertyType.GetGenericArguments();
                var resultTypeName = propertyType.Name.Substring(0, propertyType.Name.IndexOf("`"));

                foreach (var args in typeArgs)
                {
                    resultTypeName += "Of" + args.Name;
                }

                return resultTypeName;
            }

            return ServiceContractHelper.GetContractTypeName(propertyType);
         }

        public void WriteWebApiMethod(StringBuilder sb, HashSet<Type> knownTypes, WebApiMethodInfo methodInfo)
        {
            if (methodInfo.IsRegional) // Skip for now. Needs to be implemented.
            {
                return;
            }

            sb.Append("\t\t");
            sb.Append("public ");

            var innerType = "object";
            var returnType = "object";
            if (methodInfo.ResponseType != null)
            {
                if (methodInfo.ResponseType == typeof(void))
                {
                    innerType = "void";
                }
                else
                {
                    innerType = GetTypeName(null, knownTypes, methodInfo.ResponseType);
                }                
            }
            returnType = innerType == "void" ? "ServiceResponse" : "ServiceResponse<" + innerType + ">";
            sb.AppendFormat("{0} {1}", returnType, methodInfo.RpcName);
            sb.Append("(");

            var paramsAdded = 0;

            foreach (var parameter in methodInfo.AllParameters)
            {
                if (++paramsAdded > 1)
                {
                    sb.Append(", ");
                }

                sb.AppendFormat("{0} {1}", GetTypeName(null, knownTypes, parameter.Type), parameter.Name);
            }


            sb.Append(")");

           
            sb.Append(" {");
            sb.AppendLine();

            //return this.get<any>(`${fileID}/{1}`.format(fileID,filename));

            var ci = CultureInfo.CurrentCulture;
            
            if (methodInfo.ResponseType != null)
            {                
                var methodName = ci.TextInfo.ToTitleCase(methodInfo.HttpVerb.ToString());
                var methodTypeParameter = innerType != "void" ? "<" + innerType + ">" : "";

                var urlPattern = methodInfo.UrlParameters.Any() ? "string.Format(\"" + methodInfo.UrlPattern + "\", " + string.Join(", ", methodInfo.UrlParameters.Select(p => p.Name)) + ")"
                                        : "\"" + methodInfo.UrlPattern + "\"";

                sb.Append("\t\t\t");
                sb.Append("return ");
                

                if (methodInfo.RequestParameters.Any() && (methodInfo.HttpVerb == WebApiHttpVerb.Post || methodInfo.HttpVerb == WebApiHttpVerb.Patch))
                {
                    sb.AppendFormat("{0}{1}({2}, {3});", methodName, methodTypeParameter, urlPattern, string.Join(", ", methodInfo.RequestParameters.Select(p => p.Name).ToArray()));
                }
                else if (methodInfo.RequestParameters.Any())
                {
                    var counter = -1;


                    var requestParams = string.Join("&", methodInfo.RequestParameters.Select(p => p.Name + "={" + (++counter) + "}").ToArray());
                    var requestVals = string.Join(",", methodInfo.RequestParameters.Select(p => p.Type == typeof (string) ? "Uri.EscapeDataString(" + p.Name + ")" : p.Name).ToArray());

                    sb.AppendFormat("{0}{1}({2} + string.Format(\"?{3}\",{4}));", methodName, methodTypeParameter, urlPattern, requestParams, requestVals);
                }
                else
                {
                    sb.AppendFormat("{0}{1}({2});", methodName, methodTypeParameter, urlPattern);
                }

                //switch (methodInfo.HttpVerb)
                //{
                //    case WebApiHttpVerb.Get:
                //    case WebApiHttpVerb.Delete:
                //    case WebApiHttpVerb.Options:
                //        if (methodInfo.RequestParameters.Any())
                //        {
                //            sb.AppendFormat("{0}{1}(\"{2}?{3}\");", methodName, methodTypeParameter, methodInfo.UrlPattern, string.Join("&", methodInfo.RequestParameters.Select(p => p.Name + "=${" + p.Name + "}")));
                //        }
                //        else
                //        {
                //            sb.AppendFormat("{0}{1}(\"{2}\");", methodName, methodTypeParameter, methodInfo.UrlPattern);
                //        }

                //        break;
                //    case WebApiHttpVerb.Post:
                //    case WebApiHttpVerb.Patch:

                        
                //        if (methodInfo.RequestParameters.Any())
                //        {
                //            sb.AppendFormat("{0}{1}({2}, {3});", methodName, methodTypeParameter, urlPattern, string.Join(", ", methodInfo.RequestParameters.Select(p => p.Name).ToArray()));
                //        }
                //        else
                //        {
                //            sb.AppendFormat("{0}{1}({2});", methodName, methodTypeParameter, urlPattern);
                //        }
                //        break;

                //}

                sb.AppendLine();
            }

            sb.Append("\t\t");
            sb.Append("}");
            sb.AppendLine();
            sb.AppendLine();
        }
    }
}
