﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;

namespace Curse.TypeScriptSharp
{
    class Program
    {

        static void GenerateVoiceClient(string outputFolder, string templateFolder)
        {
            var voiceServiceGenerator = new ProtoServiceGenerator(new TypeScriptHelper(), @"C:\Projects\VoiceServices\Repository\Source\Curse.Voice.Contracts\bin\Debug",
              new[] {                                 
                    @"Curse.Voice.Contracts.dll"
                },
              outputFolder,
              templateFolder,
              "VoiceService");

            voiceServiceGenerator.Generate();          
        }


        static readonly string[] Services = { "accounts", "bugs", "calls", "contacts", "conversations", "giveaways", "groups", "polls", "reporting", "sessions", "sync", "logins", "stats", "avatars", "images", "files"};

        static void GenerateTypeScriptMicroServices(string outputFolder, string templateFolder, string[] excludeServices = null)
        {            
            var services = Services;

            if (excludeServices != null)
            {
                services = services.Where(p => !excludeServices.Contains(p)).ToArray();
            }


            foreach (var service in services)
            {
                Console.WriteLine("Generating '" + service + "' ...");
                var contactServiceGenerator = new WebApiServiceGenerator
               (
                    new HashSet<string> { "Curse.Friends.NotificationContracts", "Curse.Friends.Enums" },
                    new TypeScriptHelper(),
                   string.Format(@"C:\Projects\CloudServices\Repository\Source\Curse.Friends.{0}WebService\bin\Curse.Friends.{0}WebService.dll", service),
                   "Curse.", null,
                   outputFolder,
                   templateFolder,
                   "{0}Service"
               );

                contactServiceGenerator.Generate();
            }
        }

        static void GenerateCSharpMicroServices(string sourceFolder, string outputFolder, string[] whiteList = null, string[] blackList = null)
        {            
            var serviceNames = Services;
            if (whiteList != null)
            {
                serviceNames = serviceNames.Where(p => whiteList.Contains(p)).ToArray();
            }

            if (blackList != null)
            {
                serviceNames = serviceNames.Where(p => !blackList.Contains(p)).ToArray();
            }

            var allControllers = new List<WebApiControllerInfo>();

            var outputNamespace = "Curse.ServiceClients";
            var contractNamespace = "Curse.ServiceClients.Contracts";

            foreach (var service in serviceNames)
            {
                Console.WriteLine("Generating '" + service + "' ...");
                var generator = new WebApiServiceGenerator
               (
                    new HashSet<string> { "Curse.Friends.NotificationContracts", "Curse.Friends.Enums" },
                    new CSharpHelper(contractNamespace,
                            new HashSet<string> { contractNamespace, outputNamespace, "System", "System.Collections.Generic" },
                            new HashSet<string> { "Curse.Friends.NotificationContracts", "Curse.Friends.Enums", "Curse.Friends.MicroService.Contracts" }), 
                   string.Format(@"C:\Projects\CloudServices\Repository\Source\Curse.Friends.{0}WebService\bin\Curse.Friends.{0}WebService.dll", service),
                   "Curse.", null,
                   Path.Combine(outputFolder, "Generated"),
                   Path.Combine(sourceFolder, @"Templates\WebServiceClientTemplate.cs"), 
                   "{0}Service"
               );


                allControllers.AddRange(generator.Generate());
            }

            var urls = new StringBuilder();
            var clients = new StringBuilder();
            var constructors = new StringBuilder();

            // Generate the service configs
            foreach (var controller in allControllers)
            {
                urls.Append("\t\t\t");
                urls.AppendFormat("public abstract string {0} {{ get; }}", controller.ControllerName).AppendLine();

                clients.Append("\t\t");
                clients.AppendFormat("public readonly {0}Service {0};", controller.ControllerName).AppendLine();

                constructors.Append("\t\t\t");
                constructors.Append("try").AppendLine();
                constructors.Append("\t\t\t");
                constructors.Append("{").AppendLine();
                constructors.Append("\t\t\t\t");
                constructors.AppendFormat("{0} = new {0}Service(Urls.{0});", controller.ControllerName).AppendLine();
                constructors.Append("\t\t\t");
                constructors.Append("}").AppendLine();
                constructors.Append("\t\t\t");
                constructors.Append("catch (Exception ex)").AppendLine();
                constructors.Append("\t\t\t");
                constructors.Append("{").AppendLine();
                constructors.Append("\t\t\t\t");
                constructors.AppendFormat("ServicesConfiguration.LogError(ex, \"Failed to create service client {0}\");", controller.ControllerName).AppendLine();
                constructors.Append("\t\t\t");
                constructors.Append("}").AppendLine().AppendLine();

            }

            var servicesTemplate = File.ReadAllText(Path.Combine(sourceFolder, @"Templates\ServicesTemplate.cs"));
            var servicesClass = servicesTemplate.Replace("// URLS", urls.ToString())
                                                .Replace("// CLIENTS", clients.ToString())
                                                .Replace("// CONSTRUCTORS", constructors.ToString())
                                                .Replace("BaseServicesTemplate", "BaseServices")
                                                .Replace("ServiceUrlsTemplate", "ServiceUrls");


            File.WriteAllText(Path.Combine(outputFolder, "Generated", "BaseServices.cs"), servicesClass);

            if (!sourceFolder.Equals(outputFolder))
            {
                var di = new DirectoryInfo(sourceFolder);
                var sourceFiles = di.GetFiles("*.cs", SearchOption.TopDirectoryOnly);
                foreach (var fi in sourceFiles)
                {
                    fi.CopyTo(Path.Combine(outputFolder, "Generated", fi.Name), true);
                }
               
            }
           
        }

        static void GenerateNotificationClient(string outputFolder, string templateFolder)
        {
            var notificationServiceGenerator = new ProtoServiceGenerator(new TypeScriptHelper(), 
                @"C:\Projects\CloudServices\Repository\Source\Curse.Friends.NotificationContracts\bin\Debug\",
              new[] {                                 
                    @"Curse.Friends.NotificationContracts.dll",
                    @"Curse.SocketMessages.dll",
                    @"Curse.Friends.Enums.dll"},
              outputFolder,
              templateFolder,
              "NotificationsService");

            notificationServiceGenerator.Generate();
        }

        static void Main(string[] args)
        {
            while (true)
            {
                Console.Clear();
                Console.WriteLine("Welcome to the Curse services generator!");                
                Console.WriteLine("1. Generate CSharp micro services library, for clients.");
                Console.WriteLine("2. Generate CSharp micro services library, for Cloud services.");
                Console.WriteLine("3. Generate CSharp micro services library, for Voice services.");
                Console.WriteLine("4. Generate TypeScript voice client library.");
                Console.WriteLine("5. Generate TypeScript notification client library.");
                Console.WriteLine("6. Generate TypeScript micro services library, for clients.");
                Console.WriteLine("7. Generate TypeScript library for curse-client-typescript");

                var selection = Console.ReadKey(true).Key;
                Console.Clear();

                switch (selection)
                {
                    case ConsoleKey.Escape:
                        return;
                    case ConsoleKey.D0:
                        
                        GenerateTypeScriptMicroServices(@"C:\Projects\Laguna\Repository\source\laguna\services\generated", @"C:\Projects\FriendsWebClient\Repository\Source\generated\Curse.Friends.WebServiceClientTemplate.ts", new[] { "stats" });
                        GenerateVoiceClient(@"C:\Projects\Laguna\Repository\source\laguna\services\generated", @"C:\Projects\VoiceServices\Repository\Source\Curse.Voice.TypeScriptClient\Templates\Curse.Voice.BaseWebSocketClientTemplate.ts");
                        GenerateNotificationClient(@"C:\Projects\Laguna\Repository\source\laguna\services\generated", @"C:\Projects\VoiceServices\Repository\Source\Curse.Voice.TypeScriptClient\Templates\Curse.Voice.BaseWebSocketClientTemplate.ts");
                        break;
                    case ConsoleKey.D1:
                        GenerateCSharpMicroServices(@"C:\Projects\Curse.ServiceClients\Repository\Source\CSharp", @"C:\Projects\Curse.ServiceClients\Repository\Source\CSharp");
                        
                        break;

                    case ConsoleKey.D2:
                        GenerateCSharpMicroServices(@"C:\Projects\Curse.ServiceClients\Repository\Source\CSharp", @"C:\Projects\CloudServices\Repository\Source\Curse.Friends.ServiceClients");
                        break;
                    case ConsoleKey.D3:
                        GenerateCSharpMicroServices(@"C:\Projects\Curse.ServiceClients\Repository\Source\CSharp", @"C:\Projects\VoiceServices\Repository\Source\Curse.Voice.ServiceClients");
                        break;
                    case ConsoleKey.D4:
                        GenerateVoiceClient(@"C:\Projects\Laguna\Repository\source\laguna\services\generated", @"C:\Projects\VoiceServices\Repository\Source\Curse.Voice.TypeScriptClient\Templates\Curse.Voice.BaseWebSocketClientTemplate.ts");
                        break;

                    case ConsoleKey.D5:
                        GenerateNotificationClient(@"C:\Projects\Laguna\Repository\source\laguna\services\generated", @"C:\Projects\VoiceServices\Repository\Source\Curse.Voice.TypeScriptClient\Templates\Curse.Voice.BaseWebSocketClientTemplate.ts");
                        break;

                    case ConsoleKey.D6:
                        GenerateTypeScriptMicroServices(@"C:\Projects\Laguna\Repository\source\laguna\services\generated", @"C:\Projects\FriendsWebClient\Repository\Source\generated\Curse.Friends.WebServiceClientTemplate.ts", new[] { "stats" });
                        break;
                    case ConsoleKey.D7:
                        
                        GenerateTypeScriptMicroServices(@"C:\Projects\curse-client-typescript\Repository",
                                                        @"C:\Projects\FriendsWebClient\Repository\Source\client-templates\Curse.Friends.WebServiceClientTemplate.ts", new[] { "stats" });

                        GenerateVoiceClient(@"C:\Projects\curse-client-typescript\Repository", 
                                            @"C:\Projects\FriendsWebClient\Repository\Source\client-templates\Curse.Friends.BaseWebSocketClientTemplate.ts");
                        
                        GenerateNotificationClient(@"C:\Projects\curse-client-typescript\Repository", 
                                                   @"C:\Projects\FriendsWebClient\Repository\Source\client-templates\Curse.Friends.BaseWebSocketClientTemplate.ts");
                        break;
                        
                    default:
                        Console.WriteLine("Invalidation selection! Press any key to continue...");
                        continue;
                }

                Console.WriteLine("Complete!");
                Console.ReadKey(true);
            }
        }
    }
}
