﻿using Curse.MurmurHash;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;

namespace Curse.TypeScriptSharp
{
    public class ProtoServiceGenerator
    {
        private readonly ICodeGenerator _codeGenerator;
        private readonly string[] _assemblySources;
        private readonly string _outputDirectory;
        private readonly string _proxyClientTemplate;
        private readonly string _proxyClientClassName;        
        private readonly string _baseAssemblyPath;

        private readonly string _contractTypeEnumName;
        private readonly string _eventsInterfaceName;

        public ProtoServiceGenerator(ICodeGenerator codeGenerator, string baseAssemblyPath, string[] assemblySources, string outputDirectory, string proxyClientTemplate, string proxyClientClassName)
        {
            _codeGenerator = codeGenerator;
            _baseAssemblyPath = baseAssemblyPath;
            _assemblySources = assemblySources;
            _outputDirectory = outputDirectory;
            _proxyClientTemplate = proxyClientTemplate;
            _proxyClientClassName = proxyClientClassName;            
            _baseAssemblyPath = baseAssemblyPath;

            _contractTypeEnumName = proxyClientClassName + "ContractType";
            _eventsInterfaceName = proxyClientClassName + "Events";
        }


        static ProtoServiceGenerator()
        {
            AppDomain.CurrentDomain.ReflectionOnlyAssemblyResolve += CurrentDomainOnReflectionOnlyAssemblyResolve;
        }

        private static Assembly CurrentDomainOnReflectionOnlyAssemblyResolve(object sender, ResolveEventArgs args)
        {
            
            if (args.Name.StartsWith("System"))
            {
                return Assembly.ReflectionOnlyLoad(args.Name);
            }

            var dllName = args.Name.Substring(0, args.Name.IndexOf(",")) + ".dll";
            var parentAssemblyPath = Path.GetDirectoryName(args.RequestingAssembly.Location);
            var path = Path.Combine(parentAssemblyPath, dllName);
            if (!File.Exists(path))
            {
                Console.WriteLine("Unable to load assembly: " + path);
                Console.ReadLine();
            }
            var assembly = Assembly.ReflectionOnlyLoadFrom(path);
            return assembly;
        }

        private Assembly GetAssembly(string assemblyLocation)
        {
            var assemblies = AppDomain.CurrentDomain.ReflectionOnlyGetAssemblies();
            var assembly = assemblies.FirstOrDefault(p => Path.GetFileName(p.Location).Equals(assemblyLocation));
            if (assembly == null)
            {
                assembly = Assembly.ReflectionOnlyLoadFrom(Path.Combine(_baseAssemblyPath, assemblyLocation));    
            }

            return assembly;
        }

        public void Generate()
        {            
            var knownTypes = new HashSet<Type>();
            var requestTypes = new HashSet<Type>();
            var responseTypes = new HashSet<Type>();

            // First build a list of known types:
            foreach (var assemblyLocation in _assemblySources)
            {
                var assembly = GetAssembly(assemblyLocation);                
                var types = assembly.GetTypes();

                foreach(var type in types)
                {
                    if (type.IsEnum || type.GetCustomAttributesData().Any(p => p.AttributeType.Name.Equals("ServiceContractAttribute")) || type.GetCustomAttributesData().Any(p => p.AttributeType.Name.Equals("ProtoContractAttribute")))
                    {
                        if(type.GetCustomAttributesData().Any(p => p.AttributeType.Name.Equals("ObsoleteAttribute")))
                        {
                            Console.WriteLine("Skipping obsolete type: " + type.Name);
                            continue;
                            
                        }
                        knownTypes.Add(type);
                    }                    
                }                
            }

            // For each assmbly, generate a module
            foreach (var assemblyLocation in _assemblySources)
            {
                var assembly = GetAssembly(assemblyLocation);
                var types = assembly.GetTypes();
                var sb = new StringBuilder();

                var ns = types.FirstOrDefault().Namespace;

                _codeGenerator.WriteImports(knownTypes, sb, ns, "");
                               
                sb.AppendLine();

                foreach (var type in types.OrderBy(p => p.Name))
                {                   
                    var attributes = type.GetCustomAttributesData();
                    if (attributes.Any(p => p.AttributeType.Name == "ObsoleteAttribute"))
                    {
                        continue;
                    }

                    var serviceContractAttribute = attributes.FirstOrDefault(p => p.AttributeType.Name.Equals("ServiceContractAttribute"));

                    if (!type.IsEnum && serviceContractAttribute == null && !attributes.Any(p => p.AttributeType.Name.Equals("ProtoContractAttribute")))
                    {
                        Console.WriteLine("Skipping type: " + type.Name);
                        continue;
                    }

                    Console.WriteLine("Importing type: " + type.Name);

                    if(type.IsClass)
                    {
                   
                        _codeGenerator.WriteInterface(knownTypes, sb, type);
                        var contractDirection = ServiceContractHelper.GetContractDirection(type);

                        if (contractDirection.HasFlag(ServiceContractHelper.ServiceContractDirection.Request))
                        {
                            requestTypes.Add(type);
                        }

                        if (contractDirection.HasFlag(ServiceContractHelper.ServiceContractDirection.Notification) || contractDirection.HasFlag(ServiceContractHelper.ServiceContractDirection.Response))
                        {
                            responseTypes.Add(type);
                        }
                      
                    }
                    else if (type.IsEnum)
                    {
                        _codeGenerator.WriteEnum(sb, type);
                    }
                    sb.AppendLine();
                }

                sb.AppendLine();
                
                var moduleOutput = sb.ToString();
                var outputFolder = Path.Combine(_outputDirectory, "models");
                if (!Directory.Exists(outputFolder))
                {
                    Directory.CreateDirectory(outputFolder);
                }

                var outputPath = Path.Combine(outputFolder, ns + ".ts");

                File.WriteAllText(outputPath, moduleOutput);
            }

            Console.WriteLine("Writing proxy client...");
            WriteProxyClient(knownTypes, requestTypes, responseTypes);
        }
        
        private void WriteProxyClient(HashSet<Type> knownTypes, HashSet<Type> requestTypes, HashSet<Type> responseTypes)
        {
            // Generate a stub method for each request type
            var template = File.ReadAllText(_proxyClientTemplate);
            
            if (!template.Contains("/// METHODS"))
            {
                throw new Exception("Template file must have a placeholder for request types!");
            }

            var methodsSB = new StringBuilder();
            var eventsSB = new StringBuilder();
            var enumSB = new StringBuilder();
            var contractReg = new StringBuilder();

            var allTypes = new HashSet<Type>(requestTypes.Concat(responseTypes));

            foreach (var type in allTypes)
            {
                // Contract Enum
                var typeName = Encoding.ASCII.GetBytes(type.FullName);
                var typeID = (Int32)MurmurHash2.GetHashCode(ref typeName, typeName.Length, 1);
                enumSB.AppendFormat("\t{0} = {1},", type.Name, typeID).AppendLine();    
            }

            

            // Iterate over each request type and generate a stub method
            foreach(var type in requestTypes)
            {
                // Contract Enum
                var typeName = Encoding.ASCII.GetBytes(type.FullName);
                
                // Send method
                methodsSB.Append("\t");
                methodsSB.AppendFormat("send{0}(contract:{1}) {{", type.Name, type.GetModuleTypeName(null));
                methodsSB.AppendLine();
                methodsSB.Append("\t\t");
                methodsSB.AppendFormat("this.sendContract({0}.{1}, contract);", _contractTypeEnumName, type.Name);
                methodsSB.AppendLine();
                methodsSB.Append("\t");
                methodsSB.Append("}");
                methodsSB.AppendLine();
                methodsSB.AppendLine();
            }

            // Iterate over each response type, and create stub methods
            foreach (var type in responseTypes)
            {

                var actionName = type.Name.Substring(0, 1).ToLower() + type.Name.Substring(1);
                var eventName = type.Name.ToCamelCase();

                // Events                
                eventsSB.AppendFormat("\t{0}(data: {1}): void;", actionName, type.GetModuleTypeName(null)).AppendLine();                                

                // Callbacks
                methodsSB.Append("\t");
                methodsSB.AppendFormat("on{0}(contract:{1}) {{", type.Name, type.GetModuleTypeName(null));
                methodsSB.AppendLine();
                methodsSB.Append("\t\t ");
                methodsSB.AppendFormat("this.events.{0}(contract);", eventName);
                methodsSB.AppendLine();
                methodsSB.Append("\t");
                methodsSB.Append("}");
                methodsSB.AppendLine();
                methodsSB.AppendLine();

                contractReg.AppendFormat("\t\t\t[{0}.{1}]: (contract: any) => this.on{1}(contract),", _contractTypeEnumName, type.Name).AppendLine();                
            }


            var generatedMethods = methodsSB.ToString();

            var sb = new StringBuilder();
            _codeGenerator.WriteImports(knownTypes, sb);

            var clientJS = template
                .Replace("/// IMPORTS", sb.ToString())
                .Replace("/// CONTRACTMAPS", enumSB.ToString())
                .Replace("/// SERVICEVENTS", eventsSB.ToString())
                .Replace("/// CONTRACTREGISTRATION", contractReg.ToString())
                .Replace("/// METHODS", generatedMethods)
                .Replace("WebSocketClientTemplate", _proxyClientClassName)
                .Replace("ServiceContractTypeTemplate", _contractTypeEnumName)
                .Replace("ServiceEventsTemplate", _eventsInterfaceName);

            var proxyClientPath = Path.Combine(_outputDirectory, _proxyClientClassName + ".ts");
            File.WriteAllText(proxyClientPath, clientJS);

        }

       
    }
}
