﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.ServiceModel.Web;
using System.Text;

namespace Curse.TypeScriptSharp
{
    public static class RestServiceGenerator
    {
        public static void Generate(string assemblySource, string namespaceFilter, string[] ignoreNamespaces,
            string webServiceType, string outputDirectory, string proxyClientTemplate, string proxyClientOutput)
        {
            var assembly = Assembly.LoadFrom(assemblySource);
            var knownTypes = new HashSet<Type>();
            var exploredTypes = new HashSet<Type>();

            // Get all operation contracts, and reflect over their return and input types
            var serviceType = assembly.GetType(webServiceType);

            var methods = serviceType.GetMethods(BindingFlags.Instance | BindingFlags.Public);
            var validMethods = new HashSet<MethodInfo>();

            foreach (var method in methods)
            {
                var attributes = method.GetCustomAttributes();
                if (attributes.Any(p => p.GetType().Name == "ObsoleteAttribute"))
                {
                    Console.WriteLine("Skipping custom attribute: " + method.Name);
                    continue;
                }

                if (!attributes.Any(p => p.GetType().Name == "OperationContractAttribute"))
                {
                    continue;
                }

                validMethods.Add(method);
                TypeExplorer.Explore(namespaceFilter, knownTypes, method.ReturnType, exploredTypes);
                if (method.ReturnParameter != null)
                {
                    var parameters = method.GetParameters();
                    foreach (var p in parameters)
                    {
                        TypeExplorer.Explore(namespaceFilter, knownTypes, p.ParameterType, exploredTypes);
                    }
                }
            }

            // Generate the interface definitions for each known type
            var typesByNamespace =
                knownTypes.Where(p => !ignoreNamespaces.Contains(p.Namespace)).GroupBy(p => p.Namespace);
            var namespaces = typesByNamespace.Select(p => p.Key).ToArray();
            StringBuilder sb;

            foreach (var ns in typesByNamespace)
            {
                sb = new StringBuilder();

                foreach (var otherNamespace in namespaces.Where(p => p != ns.Key))
                {
                    var sanitizedNamespace = otherNamespace.Replace(".", "_");

                    sb.AppendFormat("import * as {0} from './{1}';", sanitizedNamespace, otherNamespace);
                    sb.AppendLine();
                }

                foreach (var type in ns)
                {
                    if (type.IsEnum)
                    {
                        TypeScriptHelper.WriteTypeScriptEnum(sb, type);
                    }
                    else
                    {
                        TypeScriptHelper.WriteTypeScriptInterface(knownTypes, sb, type);
                    }
                }

                sb.AppendLine();

                var moduleOutput = sb.ToString();

                File.WriteAllText(Path.Combine(outputDirectory, ns.Key + ".ts"), moduleOutput);
            }

            sb = new StringBuilder();
            foreach (var method in validMethods)
            {
                var isGet = method.GetCustomAttribute<WebGetAttribute>() != null;
                sb.Append(method.Name);
                sb.Append("(");
                var parameters = method.GetParameters();
                for (var i = 0; i < parameters.Length; i++)
                {
                    var arg = parameters[i];
                    if (i > 0)
                    {
                        sb.Append(", ");
                    }
                    sb.AppendFormat("{0}:{1}", arg.Name,
                        TypeScriptHelper.GetTypeScriptType(null, knownTypes, arg.ParameterType));
                }
                sb.Append(")");

                var returnType = "any";
                if (method.ReturnParameter != null)
                {
                    returnType = TypeScriptHelper.GetTypeScriptType(null, knownTypes, method.ReturnType);
                    sb.AppendFormat(":Promise<{0}>", returnType);
                }
                sb.Append(" {");
                sb.AppendLine();
                if (method.ReturnParameter != null)
                {
                    if (isGet)
                    {
                        sb.AppendFormat("return this.get<{0}>(\"{1}\");", returnType, method.Name);
                    }
                    else
                    {
                        sb.AppendFormat("return this.post<{0}>(\"{1}\", {2});", returnType, method.Name,
                            parameters.FirstOrDefault().Name);
                    }
                    sb.AppendLine();
                }
                sb.Append("}");
                sb.AppendLine();
                sb.AppendLine();
            }

            // Generate a stub method for each request type
            var template = File.ReadAllText(proxyClientTemplate);

            if (!template.Contains("/// METHODS"))
            {
                throw new Exception("Template file must have a placeholder for request types!");
            }


            var generatedMethods = sb.ToString();

            sb = new StringBuilder();
            TypeScriptHelper.WriteImports(knownTypes, sb);

            var generatedImports = sb.ToString();

            var clientJS = template.Replace("/// METHODS", generatedMethods).Replace("/// IMPORTS", generatedImports);

            clientJS = clientJS.Replace("BaseWebServiceClientTemplate", "BaseWebServiceClient");
            File.WriteAllText(proxyClientOutput, clientJS);
        }

        
    }
}