﻿using System;
using System.Diagnostics;
using System.Linq;

namespace Curse.TypeScriptSharp
{
    public static class ServiceContractHelper
    {
        public static string GetContractTypeName(Type type)
        {
            var sanitizedTypeName = type.Name;

            var serviceContractAttribute = type.GetCustomAttributesData().FirstOrDefault(p => p.AttributeType.Name.Equals("ServiceContractAttribute"));
            if (serviceContractAttribute != null)
            {
                var directionArg = serviceContractAttribute.ConstructorArguments.FirstOrDefault(p => p.ArgumentType == typeof(string));
                if (directionArg.Value != null)
                {
                    sanitizedTypeName = (string)directionArg.Value;
                }
            }
            else if (type.IsGenericType)
            {
                sanitizedTypeName = type.Name.Substring(0, type.Name.IndexOf("`")) + "Of" + type.GetGenericArguments()[0].Name;
            }

            return sanitizedTypeName;
        }

        [Flags]
        public enum ServiceContractDirection
        {
            /// <summary>
            /// Direction no specified (should be determined by class name)
            /// </summary>
            Unknown = 0,

            /// <summary>
            /// Contract is sent from client to server
            /// </summary>
            Request = 1,

            /// <summary>
            /// Contract is sent from server to client
            /// </summary>
            Response = 2,

            /// <summary>
            /// Contract is sent from server to client, but not as a reply to a request.
            /// </summary>
            Notification = 4,

            /// <summary>
            /// Indicates that this contract type is embedded as part of another contract
            /// </summary>
            Embedded = 65536

        }

        internal static ServiceContractDirection GetContractDirection(Type type)
        {           
            var sanitizedTypeName = GetContractTypeName(type);
            
            var serviceContractAttribute = type.GetCustomAttributesData().FirstOrDefault(p => p.AttributeType.Name.Equals("ServiceContractAttribute"));
            if (serviceContractAttribute != null)
            {
                var directionArg = serviceContractAttribute.ConstructorArguments.FirstOrDefault(p => p.ArgumentType.IsEnum);
                if (directionArg.Value != null)
                {
                    var directionVal = (int)directionArg.Value;
                    return (ServiceContractDirection)directionVal;
                }
            }

            var direction = ServiceContractDirection.Unknown;

            if (sanitizedTypeName.EndsWith("Request") || sanitizedTypeName.Equals("Handshake"))
            {
                direction = direction | ServiceContractDirection.Request;

            }

            if (sanitizedTypeName.EndsWith("Response"))
            {
                direction = direction | ServiceContractDirection.Response;
            }

            if (sanitizedTypeName.EndsWith("Notification") || sanitizedTypeName.Equals("Handshake"))
            {
                direction = direction | ServiceContractDirection.Notification;
            }

            return direction;
        }
    }
}
