﻿using System;
using System.Collections.Generic;
using System.Reflection;
using System.Text;
using System.Linq;
using Curse.TypeScriptSharp.Model;

namespace Curse.TypeScriptSharp
{
    public class TypeScriptHelper : CodeGenerator, ICodeGenerator
    {
        public void WriteEnum(StringBuilder sb, Type type)
        {
            // To Dictionary
            // Start Enum Def
            sb.AppendFormat("export enum {0} {{", type.Name);
            sb.AppendLine();

            foreach (FieldInfo fi in type.GetFields(BindingFlags.Public | BindingFlags.Static))
            {
                sb.Append("\t");
                sb.AppendFormat("{0} = {1},", fi.Name, Convert.ToInt64(fi.GetRawConstantValue()));
                sb.AppendLine();

            }

            // Close Enum Def
            sb.Append("}");
            sb.AppendLine();
        }

        public string FileExtension
        {
            get { return ".ts"; }
        }

        public HashSet<string> OverrideImports
        {
            get { return null; }
        }

        public string GetUrlToken(string paramName, int ordinal)
        {
            return "${" + paramName + "}";
        }

        public string GetRpcName(string methodName)
        {
            return methodName[0].ToString().ToLower() + methodName.Substring(1);            
        }

        public bool ExportAllKnownTypes { get { return true; } }

        public override void WriteInterface(StringBuilder sb, InterfaceDefinition definition)
        {
            // Start Class Def
            sb.AppendFormat("export interface {0} {{", definition.Name);
            sb.AppendLine();  

            // Properties
            foreach (var prop in definition.Properties)
            {
                sb.Append("\t");
                sb.AppendFormat("{0}{1}: {2};", prop.Name, prop.IsOptional ? "?" : "", prop.TypeName);
                sb.AppendLine();
            }

            // Close Class Def
            sb.Append("}");
            sb.AppendLine();
        }

        public void WriteImports(HashSet<Type> knownTypes, StringBuilder sb, string excludeNamespace = null, string path = "models/")
        {
            var imports = new List<ImportDefinition>();
            var allNamespaces = knownTypes.GroupBy(p => p.Namespace);
            foreach (var ns in allNamespaces.Where(ns => ns.Key != excludeNamespace))
            {
                if (ns.Key == null)
                {
                    continue;
                }

                imports.Add(new ImportDefinition
                {
                    Name = ns.Key.Replace(".", "_"),
                    Path = ns.Key
                });
            }

            WriteImports(sb, imports, path);
        }

        public void WriteImports(StringBuilder sb, IEnumerable<ImportDefinition> imports, string path = "models/")
        {
            foreach (var import in imports)
            {
                sb.AppendFormat("import * as {0} from './{1}{2}';", import.Name, path, import.Path).AppendLine();
            }
        }

        private static readonly Dictionary<string, string> SystemTypeMap = new Dictionary<string, string>()
        {
            {"int", "number"},
            {"Int32", "number"},
            {"Int64", "number"},
            {"uint", "number"},
            {"UInt32", "number"},
            {"UInt64", "number"},
            {"String", "string"},
            {"string", "string"},
            {"number", "number"},            
            {"Guid", "string"},
            {"DateTime", "Date"},
            {"bool", "boolean"},
            {"Boolean", "boolean"},
        };

        public override string GetTypeName(Type outerType, HashSet<Type> knownTypes, Type propertyType)
        {
            if (SystemTypeMap.ContainsKey(propertyType.Name))
            {
                return SystemTypeMap[propertyType.Name];
            }

            if (propertyType.IsArray) // Arrays
            {

                var innerType = propertyType.GetElementType();
                if (innerType == typeof(Byte))
                {
                    return "any";
                    return "ArrayBuffer";
                }

                var inner = GetTypeName(outerType, knownTypes, innerType);

                if (knownTypes.Contains(innerType) || SystemTypeMap.ContainsKey(inner))
                {
                    return inner + "[]";
                }
            }
            else if (propertyType.IsGenericType && propertyType.GetGenericTypeDefinition().Name.StartsWith("Dictionary`")) // Dictionaries
            {
                var keyType = propertyType.GetGenericArguments()[0];
                if (keyType.IsEnum)
                {
                    keyType = typeof(Int32);
                }
                var elementType = propertyType.GetGenericArguments()[1];

                return string.Format("{{ [key: {0}]: {1} }}", GetTypeName(outerType, knownTypes, keyType), GetTypeName(outerType, knownTypes, elementType)); // TODO: Make this strongly typed
            }
            else if (propertyType.IsGenericType && (propertyType.GetGenericTypeDefinition().Name.StartsWith("HashSet`")
                || propertyType.GetGenericTypeDefinition().Name.StartsWith("ICollection`")
                || propertyType.GetGenericTypeDefinition().Name.StartsWith("List`")
                || propertyType.GetGenericTypeDefinition().Name.StartsWith("IEnumerable`")
                || propertyType.GetGenericTypeDefinition().Name.StartsWith("IReadOnlyCollection`")
                )) // Lists
            {
                var listType = propertyType.GetGenericArguments()[0];
                var typeScriptType = GetTypeName(outerType, knownTypes, listType);
                return typeScriptType + "[]"; // TODO: Make this strongly typed
            }

            if (propertyType.IsGenericType && propertyType.GetGenericTypeDefinition().Name.StartsWith("Nullable`"))
            {
                var nullableType = propertyType.GetGenericArguments()[0];
                var typeScriptType = GetTypeName(outerType, knownTypes, nullableType);
                return typeScriptType;
            }

            if (propertyType.IsGenericType)
            {
                var typeArgs = propertyType.GetGenericArguments();
                var resultTypeName = propertyType.GetModuleTypeName(outerType).Substring(0, propertyType.GetModuleTypeName(outerType).IndexOf("`"));
                foreach (var args in typeArgs)
                {
                    resultTypeName += "Of" + args.Name;
                }

                return resultTypeName;
            }

            if (knownTypes.Contains(propertyType))
            {
                if (propertyType.IsGenericType)
                {
                    return propertyType.GetModuleTypeName(outerType);
                }

                return propertyType.GetModuleTypeName(outerType);
            }

            return "any";
        }

        public void WriteOperation(StringBuilder methodBuilder, OperationDefinition operationDefinition)
        {
            // Signature
            methodBuilder.AppendFormat("{0}(", operationDefinition.Name);
            var allParams = operationDefinition.PathParameters
                .Concat(operationDefinition.BodyParameter == null ? new ParameterDefinition[0] : new[] { operationDefinition.BodyParameter })
                .Concat(operationDefinition.QueryParameters.Where(p => p.IsRequired))
                .Concat(operationDefinition.QueryParameters.Where(p => !p.IsRequired))
                .Select(p => string.Format("{0}{1}:{2}", p.Name, p.IsRequired ? "" : "?", p.TypeName));
            methodBuilder.Append(string.Join(", ", allParams)).Append(')');
            methodBuilder.AppendFormat(":Promise<{0}>", operationDefinition.ResponseTypeName).Append("{").AppendLine();

            // Body

            // Do replacements in the path
            methodBuilder.Append('\t').AppendFormat("let relativeUrl = \"{0}\"", operationDefinition.Path);
            foreach (var pathParams in operationDefinition.PathParameters)
            {
                methodBuilder.AppendFormat(".replace(\"{{{0}}}\",{0}.toString())", pathParams.Name);
            }
            methodBuilder.Append(';').AppendLine();

            // Add queries
            var queries = operationDefinition.QueryParameters;
            if (queries.Any())
            {
                methodBuilder.Append("\tvar queryParams:{[param:string]:string} = {}").AppendLine();
                foreach (var queryParam in queries)
                {
                    if (queryParam.IsRequired)
                    {
                        methodBuilder.AppendFormat("\tqueryParams[\"{0}\"] = {0}.toString();", queryParam.Name).AppendLine();
                    }
                    else
                    {
                        methodBuilder.AppendFormat("\tif({0} !== undefined){{", queryParam.Name).AppendLine();
                        methodBuilder.AppendFormat("\tqueryParams[\"{0}\"] = {0}.toString();", queryParam.Name).AppendLine();
                        methodBuilder.Append("\t}").AppendLine();
                    }
                }
            }

            // this.action
            methodBuilder.AppendFormat("\treturn this.{0}<{1}>(relativeUrl", operationDefinition.Action.ToLower(), operationDefinition.ResponseTypeName);
            if (operationDefinition.Action.ToLower() != "get")
            {
                methodBuilder.AppendFormat(",{0}", operationDefinition.BodyParameter != null ? operationDefinition.BodyParameter.Name : "undefined");
            }
            if (queries.Any())
            {
                methodBuilder.Append(",queryParams");
            }
            methodBuilder.Append(");").AppendLine();

            methodBuilder.Append("}").AppendLine();
        }

        public void WriteImport(StringBuilder sb, string importNamespace)
        {
            var sanitizedNamespace = importNamespace.Replace(".", "_");

            sb.AppendFormat("import * as {0} from './{1}';", sanitizedNamespace, importNamespace);
            sb.AppendLine();
        }

        public void WriteModuleHeader(StringBuilder sb, string moduleNamespace, string moduleName)
        {
            
        }

        public void WriteModuleFooter(StringBuilder sb)
        {
            
        }

        public void WriteWebApiMethod(StringBuilder sb, HashSet<Type> knownTypes, WebApiMethodInfo methodInfo)
        {
            sb.Append('\t');
            sb.Append(methodInfo.RpcName);
            sb.Append("(");

            var paramsAdded = 0;

            foreach (var parameter in methodInfo.AllParameters)
            {
                if (++paramsAdded > 1)
                {
                    sb.Append(", ");
                }

                sb.AppendFormat("{0}:{1}", parameter.Name, GetTypeName(null, knownTypes, parameter.Type));
            }


            sb.Append(")");

            var returnType = "any";
            if (methodInfo.ResponseType != null)
            {
                if (methodInfo.ResponseType == typeof(void))
                {
                    returnType = "void";
                }
                else
                {
                    returnType = GetTypeName(null, knownTypes, methodInfo.ResponseType);
                }

                sb.AppendFormat(":Promise<{0}>", returnType);
            }
            sb.Append(" {");
            sb.AppendLine();

            //return this.get<any>(`${fileID}/{1}`.format(fileID,filename));

            var requestMethodName = methodInfo.HttpVerb.ToString().ToLower();
            if (methodInfo.IsRegional)
            {
                requestMethodName = requestMethodName + "ForRegion";
            }
          
            if (methodInfo.ResponseType != null)
            {
                sb.Append("\t\t");

                if (methodInfo.IsRegional)
                {
                    switch (methodInfo.HttpVerb)
                    {
                        case WebApiHttpVerb.Get:
                        case WebApiHttpVerb.Options:
                            if (methodInfo.RequestParameters.Any())
                            {
                                sb.AppendFormat("return this._{0}<{1}>(regionKey,`{2}?{3}`);", requestMethodName, returnType, methodInfo.UrlPattern,
                                    string.Join("&", methodInfo.RequestParameters.Select(p => p.Type == typeof (string) ? p.Name + "=${encodeURIComponent(" + p.Name + ")}" : p.Name + "=${" + p.Name + "}")));
                            }
                            else
                            {
                                sb.AppendFormat("return this._{0}<{1}>(regionKey,`{2}`);", requestMethodName, returnType, methodInfo.UrlPattern);
                            }

                            break;
                        case WebApiHttpVerb.Delete:
                        case WebApiHttpVerb.Post:
                        case WebApiHttpVerb.Patch:


                            if (methodInfo.RequestParameters.Any())
                            {
                                sb.AppendFormat("return this._{0}<{1}>(regionKey,`{2}`, {3});", requestMethodName, returnType, methodInfo.UrlPattern, string.Join(", ", methodInfo.RequestParameters.Select(p => p.Name)));
                            }
                            else
                            {
                                sb.AppendFormat("return this._{0}<{1}>(regionKey,`{2}`);", requestMethodName, returnType, methodInfo.UrlPattern);
                            }
                            break;

                    }

                }
                else
                {
                    switch (methodInfo.HttpVerb)
                    {
                        case WebApiHttpVerb.Get:
                        case WebApiHttpVerb.Options:
                            if (methodInfo.RequestParameters.Any())
                            {
                                sb.AppendFormat("return this._{0}<{1}>(`{2}?{3}`);", requestMethodName, returnType, methodInfo.UrlPattern,
                                    string.Join("&", methodInfo.RequestParameters.Select(p => p.Type == typeof (string) ? p.Name + "=${encodeURIComponent(" + p.Name + ")}" : p.Name + "=${" + p.Name + "}")));
                            }
                            else
                            {
                                sb.AppendFormat("return this._{0}<{1}>(`{2}`);", requestMethodName, returnType, methodInfo.UrlPattern);
                            }

                            break;
                        case WebApiHttpVerb.Delete:
                        case WebApiHttpVerb.Post:
                        case WebApiHttpVerb.Patch:


                            if (methodInfo.RequestParameters.Any())
                            {
                                sb.AppendFormat("return this._{0}<{1}>(`{2}`, {3});", requestMethodName, returnType, methodInfo.UrlPattern, string.Join(", ", methodInfo.RequestParameters.Select(p => p.Name)));
                            }
                            else
                            {
                                sb.AppendFormat("return this._{0}<{1}>(`{2}`);", requestMethodName, returnType, methodInfo.UrlPattern);
                            }
                            break;

                    }

                }

                
                sb.AppendLine();
            }

            sb.Append('\t');
            sb.Append("}");
            sb.AppendLine();
            sb.AppendLine();
        }
    }
}
