﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Web.Http;
using System.Web.Http.Description;

namespace Curse.TypeScriptSharp
{

    public enum WebApiHttpVerb
    {
        Get,
        Post,
        Delete,
        Options,
        Patch
    }

    public class WebApiUrlParameter
    {
        public string Name { get; set; }
        public Type Type { get; set; }
        public bool IsOptional { get; set; }

        public WebApiUrlParameter(string name, Type type, bool isOptional)
        {
            Name = name;
            Type = type;
            IsOptional = isOptional;
        }
    }

    public class WebApiMethodInfo
    {       
        public string UrlPattern { get; set; }
        
        private readonly List<WebApiUrlParameter> _urlParameters = new List<WebApiUrlParameter>();

        public IReadOnlyCollection<WebApiUrlParameter> UrlParameters
        {
            get { return _urlParameters; }

        }

        private readonly List<WebApiUrlParameter> _allParameters = new List<WebApiUrlParameter>();

        public IReadOnlyCollection<WebApiUrlParameter> AllParameters
        {
            get { return _allParameters; }
            
        }

        private readonly List<WebApiUrlParameter> _requestParameters = new List<WebApiUrlParameter>();

        public IReadOnlyCollection<WebApiUrlParameter> RequestParameters
        {
            get { return _requestParameters; }

        }

        private readonly MethodInfo _methodInfo;        
        private readonly WebApiControllerInfo _parent;
        
        public Type ResponseType { get; set; }
        public string RpcName { get; set; }

        public HashSet<Type> AllTypes
        {
            get
            {
                var allTypes = new HashSet<Type>();
                if (ResponseType != null)
                {
                    allTypes.Add(ResponseType);
                }

                foreach (var param in _allParameters)
                {
                    allTypes.Add(param.Type);
                }

                return allTypes;
            }
            
        }

        /// <summary>
        /// POST, GET, DELETE, ETC
        /// </summary>
        public WebApiHttpVerb HttpVerb { get; set; }

        private readonly ICodeGenerator _codeGenerator;

        private readonly bool _isRegional;

        public bool IsRegional
        {
            get { return _isRegional; }
        }

        public WebApiMethodInfo(ICodeGenerator codeGenerator, WebApiControllerInfo controllerInfo, MethodInfo methodInfo, bool isRegional)
        {
            _codeGenerator = codeGenerator;
            _methodInfo = methodInfo;
            _parent = controllerInfo;
            _isRegional = isRegional;
            Explore();
        }


        private WebApiHttpVerb GetHttpVerb()
        {
            if (_methodInfo.GetCustomAttribute<HttpGetAttribute>() != null)
            {
                return WebApiHttpVerb.Get;
            }

            if (_methodInfo.GetCustomAttribute<HttpPostAttribute>() != null)
            {
                return WebApiHttpVerb.Post;
            }

            if (_methodInfo.GetCustomAttribute<HttpOptionsAttribute>() != null)
            {
                return WebApiHttpVerb.Options;
            }

            if (_methodInfo.GetCustomAttribute<HttpDeleteAttribute>() != null)
            {
                return WebApiHttpVerb.Delete;
            }

            if (_methodInfo.GetCustomAttribute<HttpPatchAttribute>() != null)
            {
                return WebApiHttpVerb.Patch;
            }

            throw new InvalidOperationException("Unknown http method for web api method: " + _methodInfo.Name);
        }

        private Type GetResponseType()
        {            
            var attribute = _methodInfo.GetCustomAttribute<ResponseTypeAttribute>();
            
            if (attribute == null)
            {
                var methodResponseType = _methodInfo.ReturnType;

                if (methodResponseType != typeof(IHttpActionResult))
                {
                    return methodResponseType;
                }

                if (HttpVerb == WebApiHttpVerb.Get)
                {
                    throw new InvalidOperationException("Unknown response type for web api method: " + _methodInfo.Name);    
                }

                return typeof (void);

            }

            return attribute.ResponseType;
        }

        public bool CanBeGenerated
        {
            get { return ResponseType != typeof (Stream); }
        }


        private void Explore()
        {
            HttpVerb = GetHttpVerb();
            ResponseType = GetResponseType();

            UrlPattern = _parent.RoutePrefix + "/" + _methodInfo.GetCustomAttribute<RouteAttribute>().Template;
            UrlPattern = UrlPattern.Trim('/');
           
            // Get the parameters
            var methodParams = _methodInfo.GetParameters();
            var paramCount = -1;

            if (_isRegional)
            {
                _allParameters.Add(new WebApiUrlParameter("regionKey", typeof(string), false));      
            }
            
            foreach (var parameter in methodParams)
            {

                var fromuri = parameter.GetCustomAttributes<FromUriAttribute>();
                var urlToken = "{" + parameter.Name + "}";
                if (UrlPattern.Contains(urlToken))
                {
                    ++paramCount;                    
                    _urlParameters.Add(new WebApiUrlParameter(parameter.Name, parameter.ParameterType, parameter.IsOptional));
                    UrlPattern = UrlPattern.Replace(urlToken, _codeGenerator.GetUrlToken(parameter.Name, paramCount));
                }
                else if (fromuri != null && fromuri.Any())
                {
                    UrlPattern = UrlPattern + "?" + parameter.Name + "=" + _codeGenerator.GetUrlToken(parameter.Name, paramCount);
                }
                else
                {
                    _requestParameters.Add(new WebApiUrlParameter(parameter.Name, parameter.ParameterType, parameter.IsOptional));
                }

                _allParameters.Add(new WebApiUrlParameter(parameter.Name, parameter.ParameterType, parameter.IsOptional));       
            }


            RpcName = _codeGenerator.GetRpcName(_methodInfo.Name);

            if (_isRegional)
            {
                RpcName += "ForRegion";
            }
        }

        public void Generate(StringBuilder sb, HashSet<Type> knownTypes)
        {
            _codeGenerator.WriteWebApiMethod(sb, knownTypes, this);
        }
    
    }
}
