﻿
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net;
using System.Reflection;
using System.Text;
using System.Text.RegularExpressions;
using Curse.TypeScriptSharp.Model;
using Newtonsoft.Json.Linq;

namespace Curse.TypeScriptSharp
{
    public static class WebApiRestServiceGenerator
    {
        public static void Generate(string swaggerSource, string[] enumAssemblyPaths, string outputDirectory, string proxyClientTemplate, string proxyClientOutput)
        {
            // Read Swagger Definition
            JToken swaggerRoot;
            if (swaggerSource.StartsWith("http"))
            {
                using (var client = new WebClient())
                {
                    swaggerRoot = JToken.Parse(client.DownloadString(swaggerSource));
                }
            }
            else
            {
                swaggerRoot = JToken.Parse(File.ReadAllText(swaggerSource));
            }

            // Ensure placeholders for imports and methods
            var template = File.ReadAllText(proxyClientTemplate);
            if (!template.Contains("/// METHODS") || !template.Contains("/// IMPORTS"))
            {
                throw new Exception("Template file must have a placeholder for methods and for imports!");
            }

            // Load assemblies with supporting enums to be able to map name to value
            var enumTypes = new HashSet<Type>();
            foreach (var enumAssemblyPath in enumAssemblyPaths)
            {
                foreach (var type in Assembly.LoadFrom(enumAssemblyPath).GetTypes().Where(t=>t.IsEnum))
                {
                    enumTypes.Add(type);
                }
            }

            // Methods
            var methodBuilder = new StringBuilder();
            foreach (JProperty path in swaggerRoot["paths"].Children())
            {
                foreach (JProperty action in path.Value.Children())
                {
                    TypeScriptHelper.WriteOperation(methodBuilder, ParseAction(path, action));
                }
            }

            // Imports (very simple and hardcoded with swagger)
            var importsBuilder = new StringBuilder();
            TypeScriptHelper.WriteImports(importsBuilder, new[] {new ImportDefinition {Name = "Definitions", Path = "Definitions"}});

            // Write WebService file
            var clientJs = template.Replace("/// METHODS", methodBuilder.ToString()).Replace("/// IMPORTS", importsBuilder.ToString());
            clientJs = clientJs.Replace("BaseWebServiceClientTemplate", "BaseWebServiceClient");
            File.WriteAllText(proxyClientOutput, clientJs);

            // Definitions of request types and dependencies
            var definitionsBuilder = new StringBuilder();
            var usedEnums = new HashSet<string>();
            foreach (JProperty definition in swaggerRoot["definitions"].Children())
            {
                TypeScriptHelper.WriteTypeScriptInterface(definitionsBuilder, ParseDefinition(definition, ref usedEnums));
            }

            // Enums have to be handled separately because they only appear as integers with known values in Swagger
            foreach (var enumType in usedEnums)
            {
                var type = enumTypes.FirstOrDefault(t => t.Name == enumType);
                if (type == null)
                {
                    Debugger.Break();
                    continue;
                }

                TypeScriptHelper.WriteTypeScriptEnum(definitionsBuilder, type);
            }

            // Write Definitions file
            File.WriteAllText(Path.Combine(outputDirectory, "Definitions.ts"), definitionsBuilder.ToString());
        }

        private static OperationDefinition ParseAction(JProperty path, JProperty action)
        {
            var operationDefinition = new OperationDefinition
            {
                Name = action.Value["operationId"].Value<string>(),
                Action = action.Name,
                Path = path.Name
            };

            // Parameters
            var parameters = (JArray)action.Value["parameters"];
            if (parameters != null)
            {

                foreach (var parameter in parameters)
                {
                    var parameterDefinition = new ParameterDefinition
                    {
                        Name = parameter["name"].Value<string>(),
                        IsRequired = parameter["required"].Value<bool>()
                    };
                    var @in = parameter["in"].Value<string>();

                    parameterDefinition.TypeName = GetTypeName(parameter);

                    switch (@in)
                    {
                        case "path":
                            operationDefinition.PathParameters.Add(parameterDefinition);
                            break;
                        case "body":
                            operationDefinition.BodyParameter = parameterDefinition;
                            break;
                        case "query":
                            operationDefinition.QueryParameters.Add(parameterDefinition);
                            break;
                    }
                }
            }

            // Return Type
            var responsesRoot = action.Value["responses"];
            operationDefinition.ResponseTypeName = responsesRoot != null ? GetTypeName(responsesRoot.First.First) : "any";

            return operationDefinition;
        }

        private static InterfaceDefinition ParseDefinition(JProperty definition, ref HashSet<string> usedEnums)
        {
            var requiredPropertyNames = new HashSet<string>();
            var required = (JArray)definition.Value["required"];
            if (required != null)
            {
                foreach (var requiredProperty in required)
                {
                    requiredPropertyNames.Add(requiredProperty.Value<string>());
                }
            }
            var properties = new List<PropertyDefinition>();
            foreach (JProperty property in definition.Value["properties"].Children())
            {
                var propertyDefinition = new PropertyDefinition
                {
                    Name = property.Name,
                    IsOptional = !requiredPropertyNames.Contains(property.Name)
                };

                var enumProperty = property.Value["enum"];
                if (enumProperty != null)
                {
                    // this is an enum, not just an integer!
                    var match = Regex.Match(property.Value["description"].Value<string>(), @"\(Enum:(?<Name>.*?)\)");
                    propertyDefinition.TypeName = match.Groups["Name"].Value;
                    usedEnums.Add(propertyDefinition.TypeName);
                }
                else
                {
                    propertyDefinition.TypeName = GetTypeName(property.Value, false);
                }

                properties.Add(propertyDefinition);
            }

            return new InterfaceDefinition
            {
                Name = definition.Name,
                Properties = properties
            };
        }

        private static string GetTypeScriptTypeFromSwaggerType(string swaggerType)
        {
            if (swaggerType == "string")
            {
                return "string";
            }
            if (swaggerType == "integer")
            {
                return "number";
            }
            if (swaggerType == "boolean")
            {
                return "boolean";
            }

            return "any";
        }

        private static string GetTypeScriptTypeFromSwaggerReference(string swaggerReference, bool includeNamespace=true)
        {
            if (swaggerReference == "#/definitions/Object")
            {
                return "any";
            }
            return (includeNamespace ? "Definitions." : "") + swaggerReference.Substring(swaggerReference.LastIndexOf("/") + 1);
        }

        private static string GetTypeName(JToken root, bool includeNamespace=true)
        {
            var type = root["type"];
            if (type != null)
            {
                // Built-in Swagger type
                return GetTypeScriptTypeFromSwaggerType(type.Value<string>());
            }

            var rootRef = root["$ref"];
            if (rootRef != null)
            {
                // Reference Type
                return GetTypeScriptTypeFromSwaggerReference(rootRef.Value<string>(), includeNamespace);
            }

            var schema = root["schema"];
            var schemaRef = schema["$ref"];
            if (schemaRef != null)
            {
                // Reference Type
                return GetTypeScriptTypeFromSwaggerReference(schemaRef.Value<string>(), includeNamespace);
            }

            var schemaType = schema["type"].Value<string>();
            if (schemaType == "array")
            {
                var items = schema["items"];
                var itemRef = items["$ref"];
                return string.Format("[{0}]", itemRef != null ? GetTypeScriptTypeFromSwaggerReference(itemRef.Value<string>(), includeNamespace) : GetTypeScriptTypeFromSwaggerType(items["type"].Value<string>()));
            }

            return GetTypeScriptTypeFromSwaggerType(schemaType);
        }
    }
}
