﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Web.Http;

namespace Curse.TypeScriptSharp
{
    public class WebApiServiceGenerator
    {
        private readonly string _webAssemblyPath;
        private readonly string _namespaceFilter;
        private readonly string[] _ignoreNamespaces;
        private readonly string _outputDirectory;
        private readonly string _proxyClientTemplate;        
        private readonly string _proxyClientClassName;
        private readonly ICodeGenerator _codeGenerator;
        
        private readonly HashSet<Type> _knownTypes = new HashSet<Type>();
        private readonly HashSet<Type> _exploredTypes = new HashSet<Type>();
        private readonly List<WebApiControllerInfo> _controllers = new List<WebApiControllerInfo>();
        private readonly HashSet<string> _fullNamespaceExports;
        

        public WebApiServiceGenerator(HashSet<string> fullNamespaceExports, ICodeGenerator codeGenerator, string webAssemblyPath, string namespaceFilter, string[] ignoreNamespaces, string outputDirectory, string proxyClientTemplate, string proxyClientClassName)
        {
            if (!File.Exists(webAssemblyPath))
            {
                throw new Exception("Invalid assembly location: " + webAssemblyPath);
            }
            _fullNamespaceExports = fullNamespaceExports;
            _codeGenerator = codeGenerator;
            _webAssemblyPath = webAssemblyPath;
            _namespaceFilter = namespaceFilter;
            _ignoreNamespaces = ignoreNamespaces ?? new string[0];
            _outputDirectory = outputDirectory;
            _proxyClientTemplate = proxyClientTemplate;
            _proxyClientClassName = proxyClientClassName;
        }

        public IReadOnlyCollection<WebApiControllerInfo> Generate()
        {
            var assembly = Assembly.LoadFrom(_webAssemblyPath);
            var controllers = new List<Type>();            
            var publicTypes = assembly.GetExportedTypes();
            
            Console.WriteLine("Filtering types");
            foreach (var type in publicTypes)
            {
                if (IsMicroServiceController(type))
                {
                    Console.WriteLine("Discovered new controller: " + type.Name);
                    controllers.Add(type);
                }
            }

            Console.WriteLine("Inspecting controllers...");
            foreach (var type in controllers)
            {
                Console.WriteLine("Processing controller: " + type);
                try
                {
                    var controller = new WebApiControllerInfo(_codeGenerator, type);
                    _controllers.Add(controller);
                }
                catch (InvalidOperationException)
                {
                    Console.WriteLine("Skipping controller {0} because it has no methods to generate", type.Name);
                }
            }

            if (_controllers.Count == 0)
            {
                Console.WriteLine("Skipping assembly {0} because it has no controllers to generate", assembly.FullName);
            }
            else
            {
                Console.WriteLine("Exploring types...");
                ExploreTypes();

                Console.WriteLine("Creating TypeScript interface files...");
                WriteInterfaceExports();

                Console.WriteLine("Creating proxy client...");
                WriteProxyClient();
            }

            return _controllers.ToArray();
        }

        private bool IsMicroServiceController(Type type)
        {
            var baseType = type.BaseType;
            while (baseType != null)
            {
                if (baseType.Name == "MicroServiceController" || baseType == typeof (ApiController))
                {
                    return true;
                }
                baseType = baseType.BaseType;
            }

            return false;
        }

        private void WriteProxyClient()
        {

            foreach (var controller in _controllers)
            {
                var sb = new StringBuilder();
                controller.Generate(sb, _knownTypes);

                // Generate a stub method for each request type
                var template = File.ReadAllText(_proxyClientTemplate);

                if (!template.Contains("/// METHODS") && !template.Contains("// METHODS"))
                {
                    throw new Exception("Template file must have a placeholder for request types!");
                }


                var generatedMethods = sb.ToString();

                sb = new StringBuilder();
                _codeGenerator.WriteImports(_knownTypes, sb);

                var generatedImports = sb.ToString();

                var clientJS = template.Replace("/// METHODS", generatedMethods).Replace("// METHODS", generatedMethods)
                    .Replace("/// IMPORTS", generatedImports).Replace("// IMPORTS", generatedImports);

                var proxyClientClassName = string.Format(_proxyClientClassName, controller.ControllerName);
                clientJS = clientJS.Replace("WebServiceClientTemplate", proxyClientClassName);

                var proxyClientOutput = Path.Combine(_outputDirectory, proxyClientClassName + _codeGenerator.FileExtension);
                File.WriteAllText(proxyClientOutput, clientJS);
            }
        }
        
        static Type[] GetClasses(string nameSpace, Assembly asm)
        {            

            List<Type> namespacelist = new List<Type>();
            List<Type> classlist = new List<Type>();

            foreach (Type type in asm.GetTypes())
            {
                if (type.Namespace == nameSpace)
                    namespacelist.Add(type);
            }

            foreach (var classname in namespacelist)
            {
                classlist.Add(classname);
            }
                
            return classlist.ToArray();
        }

        private void WriteInterfaceExports()
        {

            // Generate the interface definitions for each known type
            var typesByNamespace = _knownTypes.Where(p => !_ignoreNamespaces.Contains(p.Namespace)).GroupBy(p => p.Namespace);
            var namespaces = typesByNamespace.Select(p => p.Key).ToArray();
            var outputFolder = Path.Combine(_outputDirectory, "models");
            if (outputFolder != null)
            {
                Directory.CreateDirectory(outputFolder);
            }
            
            foreach (var ns in typesByNamespace)
            {

                if (_fullNamespaceExports.Contains(ns.Key) && !_codeGenerator.ExportAllKnownTypes)
                {
                    Console.WriteLine("Skipping known type...");
                    continue;                    
                }

                Console.WriteLine("Writing model export for " + ns.Key + " ...");
                var sb = new StringBuilder();

                if (_codeGenerator.OverrideImports != null)
                {
                    foreach (var otherNamespace in _codeGenerator.OverrideImports)
                    {
                        _codeGenerator.WriteImport(sb, otherNamespace);
                    }
                }
                else
                {
                    foreach (var otherNamespace in namespaces.Where(p => p != ns.Key))
                    {
                        _codeGenerator.WriteImport(sb, otherNamespace);
                    }    
                }
                

                var types = _fullNamespaceExports.Contains(ns.Key) ? GetClasses(ns.Key, ns.FirstOrDefault().Assembly) : ns.ToArray();
                
                foreach (var type in types)
                {
                    _knownTypes.Add(type);
                }

                _codeGenerator.WriteModuleHeader(sb, ns.Key, ns.Key.Replace(".", "_"));

                foreach (var type in types)
                {                    
                    if (type.GetCustomAttributesData().Any(p => p.AttributeType.Name.Equals("ObsoleteAttribute")))
                    {
                        Console.WriteLine("Skipping obsolete type: " + type.Name);
                        continue;
                    }

                    if (type.IsEnum)
                    {
                        _codeGenerator.WriteEnum(sb, type);
                    }
                    else
                    {
                       
                        _codeGenerator.WriteInterface(_knownTypes, sb, type);
                    }
                }                

                sb.AppendLine();

                _codeGenerator.WriteModuleFooter(sb);

                var moduleOutput = sb.ToString();

                File.WriteAllText(Path.Combine(outputFolder, ns.Key + _codeGenerator.FileExtension), moduleOutput);
            }
        }

        private void ExploreTypes()
        {           
            foreach (var controller in _controllers)
            {
                foreach (var method in controller.Methods)
                {
                    foreach (var t in method.AllTypes)
                    {
                        TypeExplorer.Explore(_namespaceFilter, _knownTypes, t, _exploredTypes);
                    }
                }                               
            }            
        }        
    }
}