﻿using System;
using Curse.SocketInterface;
using Curse.SocketMessages;
using System.Security.Cryptography;

namespace Curse.Voice.Contracts
{
    
    public class TransmitRequest : Contract<TransmitRequest>
    {
        public const int FixedSize = 9;

        public bool LoopbackEnabled { get; set; }        
        public UInt32 SequenceNumber { get; set; }
        public int VoiceDataLength { get; set; }        
        public byte[] VoiceData { get; set; }

        public override Message ToMessage(int sessionID = 0, SymmetricAlgorithm encryptionAlgorithm = null)
        {
            VoiceDataLength = VoiceData.Length;
            var body = new byte[FixedSize + VoiceData.Length];
            body.SetBytesLE(LoopbackEnabled, 0);
            body.SetBytesLE(SequenceNumber, 1);
            body.SetBytesLE(VoiceDataLength, 5);
            Buffer.BlockCopy(VoiceData, 0, body, FixedSize, VoiceData.Length);
            return Message.FromOutgoing(sessionID, body, MessageType, false);
        }

        public override void ParseMessage(Message message)
        {
            var body = message.GetBody();

            if (body.Length < FixedSize)
            {
                throw new ContractParserException("Invalid message body (Too small). Total Bytes: " + body.Length);
            }

            LoopbackEnabled = body.ToBooleanLE(0);
            SequenceNumber = body.ToUInt32LE(1);
            VoiceDataLength = body.ToInt32LE(5);

            if (VoiceDataLength > 1024)
            {
                throw new ContractParserException("Invalid voic e packet size (Too large). Total Bytes: " + VoiceDataLength);
            }

            if (body.Length != FixedSize + VoiceDataLength)
            {
                throw new ContractParserException("Invalid voice packet size (Does not match header). Total Bytes: " + VoiceDataLength);
            }

            VoiceData = new byte[VoiceDataLength];
            Buffer.BlockCopy(body, FixedSize, VoiceData, 0, VoiceDataLength);
        }

        public override bool Validate()
        {
            return VoiceDataLength > 0 && VoiceData != null;
        }
    }
}
