﻿using System.Net;
using System.ServiceModel.Web;
using Curse.Friends.Statistics.Models;
using Curse.Logging;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.Serialization;
using System.ServiceModel;
using System.Text;
using Curse.Voice.Service.Extensions;
using System.Collections.Concurrent;
using System.Data.SqlClient;
using Newtonsoft.Json;
using System.Runtime.Caching;

namespace Curse.Voice.Service
{
    [ServiceBehavior(AddressFilterMode = AddressFilterMode.Any, ConcurrencyMode = ConcurrencyMode.Multiple, UseSynchronizationContext = false, InstanceContextMode = InstanceContextMode.PerCall)]
    public class CurseFriendsMonitor : ICurseFriendsMonitor
    {

        private static readonly ObjectCache DefaultCache = MemoryCache.Default;
        private static ConcurrentDictionary<string, int> _machineLookup = new ConcurrentDictionary<string, int>();

        public void UploadRealtimeStatsJson(string apiKey, string statsJson, int regionID, string machineName, FriendsHostType hostType)
        {
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " + OperationContext.Current.GetClientIPAddress());
                return;
            }


            try
            {
                // Get the host ID
                var hostID = GetHostID(machineName, regionID, hostType);
                
                // Upsert the data
                using (var conn = new SqlConnection(CoreServiceConfiguration.Instance.ReportingConnectionString))
                {
                    conn.Open();

                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "UpdateFriendsRealtimeStats";
                        cmd.CommandType = System.Data.CommandType.StoredProcedure;
                        cmd.Parameters.AddWithValue("@HostID", hostID);
                        cmd.Parameters.AddWithValue("@Stats", statsJson);
                        cmd.ExecuteNonQuery();
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process realtime stats!", CoreServiceConfiguration.Instance.ReportingConnectionString);
            }    
        }

        public void UploadRealtimeStats(string apiKey, FriendsHostStats stats, int regionID, string machineName, FriendsHostType hostType)
        {
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " + OperationContext.Current.GetClientIPAddress());
                return;
            }


            try
            {
                // Get the host ID
                var hostID = GetHostID(machineName, regionID, hostType);

                // Serialize the stats
                var serialized = JsonConvert.SerializeObject(stats);

                // Upsert the data
                using (var conn = new SqlConnection(CoreServiceConfiguration.Instance.ReportingConnectionString))
                {
                    conn.Open();

                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "UpdateFriendsRealtimeStats";
                        cmd.CommandType = System.Data.CommandType.StoredProcedure;
                        cmd.Parameters.AddWithValue("@HostID", hostID);
                        cmd.Parameters.AddWithValue("@Stats", serialized);
                        cmd.ExecuteNonQuery();
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process realtime stats!", CoreServiceConfiguration.Instance.ReportingConnectionString);
            }                        
        }

        public void UploadPeriodicStats(string apiKey, FriendsHostStats stats, int regionID, string machineName, FriendsHostType hostType)
        {
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " + OperationContext.Current.GetClientIPAddress());
                return;
            }


            try
            {
                
                if (stats.StartDate.Second != 0 || (stats.StartDate.Minute != 0 && stats.StartDate.Minute != 15 && stats.StartDate.Minute != 30 && stats.StartDate.Minute != 45))
                {
                    Logger.Error("Failed to process uploaded periodic stats. The StartDate property is invalid!", stats.StartDate);
                    return;
                }

                // Get the host ID
                var hostID = GetHostID(machineName, regionID, hostType);
                
                // Serialize the stats
                var serialized = JsonConvert.SerializeObject(stats); 

                
                // Upsert the data
                using (var conn = new SqlConnection(CoreServiceConfiguration.Instance.ReportingConnectionString))
                {
                    conn.Open();

                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "UpdateFriendsPeriodicStats";
                        cmd.CommandType = System.Data.CommandType.StoredProcedure;
                        cmd.Parameters.AddWithValue("@HostID", hostID);
                        cmd.Parameters.AddWithValue("@StartDate", stats.StartDate);
                        cmd.Parameters.AddWithValue("@Stats", serialized);
                        cmd.ExecuteNonQuery();
                    }
                }
               
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process realtime stats!");
            }        
        }

        public void UploadPeriodicStatsJson(string apiKey, string statsJson, int regionID, string machineName, FriendsHostType hostType)
        {
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " + OperationContext.Current.GetClientIPAddress());
                return;                
            }


            try
            {

                var stats = JsonConvert.DeserializeObject<FriendsHostStats>(statsJson);
                if (stats.StartDate.Second != 0 || (stats.StartDate.Minute != 0 && stats.StartDate.Minute != 15 && stats.StartDate.Minute != 30 && stats.StartDate.Minute != 45))
                {
                    Logger.Error("Failed to process uploaded periodic stats. The StartDate property is invalid!", stats.StartDate);
                    return;
                }

                // Get the host ID
                var hostID = GetHostID(machineName, regionID, hostType);                

                // Upsert the data
                using (var conn = new SqlConnection(CoreServiceConfiguration.Instance.ReportingConnectionString))
                {
                    conn.Open();

                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "UpdateFriendsPeriodicStats";
                        cmd.CommandType = System.Data.CommandType.StoredProcedure;
                        cmd.Parameters.AddWithValue("@HostID", hostID);
                        cmd.Parameters.AddWithValue("@StartDate", stats.StartDate);
                        cmd.Parameters.AddWithValue("@Stats", statsJson);
                        cmd.ExecuteNonQuery();
                    }
                }

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process realtime stats!");
            }
        }

        public void UploadRegionStats(string apiKey, FriendsRegionStats stats, int regionID)
        {
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " + OperationContext.Current.GetClientIPAddress());
                return;
            }


            try
            {
                if (stats.StartDate.Second != 0 || (stats.StartDate.Minute != 0 && stats.StartDate.Minute != 15 && stats.StartDate.Minute != 30 && stats.StartDate.Minute != 45))
                {
                    Logger.Error("Failed to process uploaded region stats. The StartDate property is invalid!", stats.StartDate);
                    return;
                }

                // Serialize the stats
                var serialized = JsonConvert.SerializeObject(stats);


                // Upsert the data
                using (var conn = new SqlConnection(CoreServiceConfiguration.Instance.ReportingConnectionString))
                {
                    conn.Open();

                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "UpdateFriendsRegionStats";
                        cmd.CommandType = System.Data.CommandType.StoredProcedure;
                        cmd.Parameters.AddWithValue("@RegionID", regionID);
                        cmd.Parameters.AddWithValue("@StartDate", stats.StartDate);
                        cmd.Parameters.AddWithValue("@Stats", serialized);
                        cmd.ExecuteNonQuery();
                    }
                }

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process region stats!");
            }   
        }

        public void UploadRegionStatsJson(string apiKey, string statsJson, int regionID)
        {
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " + OperationContext.Current.GetClientIPAddress());
                return;
            }


            try
            {
                var stats = JsonConvert.DeserializeObject<FriendsRegionStats>(statsJson);
                if (stats.StartDate.Second != 0 || (stats.StartDate.Minute != 0 && stats.StartDate.Minute != 15 && stats.StartDate.Minute != 30 && stats.StartDate.Minute != 45))
                {
                    Logger.Error("Failed to process uploaded region stats. The StartDate property is invalid!", stats.StartDate);
                    return;
                }                

                // Upsert the data
                using (var conn = new SqlConnection(CoreServiceConfiguration.Instance.ReportingConnectionString))
                {
                    conn.Open();

                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "UpdateFriendsRegionStats";
                        cmd.CommandType = System.Data.CommandType.StoredProcedure;
                        cmd.Parameters.AddWithValue("@RegionID", regionID);
                        cmd.Parameters.AddWithValue("@StartDate", stats.StartDate);
                        cmd.Parameters.AddWithValue("@Stats", statsJson);
                        cmd.ExecuteNonQuery();
                    }
                }

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process region stats!");
            }
        }

        private static int GetHostID(string machineName, int regionID, FriendsHostType hostType)
        {
            var cacheKey = "GetHostID-MachineName:" + machineName + ";HostType: " + hostType;
            var hostID = DefaultCache.Get(cacheKey);

            if (hostID == null)
            {


                using (var conn = new SqlConnection(CoreServiceConfiguration.Instance.ReportingConnectionString))
                {
                    conn.Open();

                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "UpdateFriendsHost";
                        cmd.CommandType = System.Data.CommandType.StoredProcedure;
                        cmd.Parameters.AddWithValue("@Name", machineName);
                        cmd.Parameters.AddWithValue("@RegionID", regionID);
                        cmd.Parameters.AddWithValue("@Type", (int)hostType);
                        hostID = cmd.ExecuteScalar();
                    }
                }

                DefaultCache.Add(cacheKey, hostID, new CacheItemPolicy { Priority = CacheItemPriority.NotRemovable });
            }

            return (int)hostID;
        }
    }
}
