﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using Curse.Logging;
using Curse.Voice.Service.Extensions;
using Curse.Voice.Service.Models;
using Curse.Voice.Service.Responses;
using Curse.Voice.Contracts;

namespace Curse.Voice.Service
{
    [ServiceBehavior(AddressFilterMode = AddressFilterMode.Any, ConcurrencyMode = ConcurrencyMode.Multiple, UseSynchronizationContext = false, InstanceContextMode = InstanceContextMode.PerCall)]
    public class CurseVoiceMonitor : ICurseVoiceMonitor
    {
        
        public VoiceHostConfiguration[] GetHostConfigurations(string apiKey)
        {
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " + OperationContext.Current.GetClientIPAddress());
                return null;
            }

            return VoiceHostConfiguration.GetAll();
        }

        public CurrentVoiceHostsResponse GetCurrentVoiceHosts(string apiKey)
        {
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " + OperationContext.Current.GetClientIPAddress());
                return new CurrentVoiceHostsResponse { Status = VoiceHostsStatus.Failed, Message = "The Api Key provided did not match what was expected." };
            }

            var response = new CurrentVoiceHostsResponse();
            try
            {
                response.VoiceHostsToRegion = new Dictionary<VoiceRegion, HashSet<VoiceHost>>();
                foreach (var host in VoiceHostManager.Instance.Hosts)
                {
                    if (!response.VoiceHostsToRegion.ContainsKey(host.Region) || response.VoiceHostsToRegion[host.Region] == null)
                    {
                        response.VoiceHostsToRegion[host.Region] = new HashSet<VoiceHost>();
                    }
                    response.VoiceHostsToRegion[host.Region].Add(host);
                }
                response.Status = response.VoiceHostsToRegion.Any() ? VoiceHostsStatus.Successful : VoiceHostsStatus.NoHosts;
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
                response.Status = VoiceHostsStatus.Failed;
                response.VoiceHostsToRegion = new Dictionary<VoiceRegion, HashSet<VoiceHost>>();
            }

            return response;
        }

        public VoiceHostDetailsResponse GetHostDetails(string apiKey, int id)
        {
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " + OperationContext.Current.GetClientIPAddress());
                return new VoiceHostDetailsResponse { Status = VoiceHostDetailsStatus.Failed, Message = "The Api Key provided did not match what was expected." }; ;
            }

            VoiceHostDetailsResponse response = new VoiceHostDetailsResponse();
            try
            {
                var host = VoiceHostManager.Instance.Hosts.FirstOrDefault(p => p.ID == id);

                if (host == null)
                {
                    response.Status = VoiceHostDetailsStatus.HostNotFound;
                }
                else
                {
                    response.Status = VoiceHostDetailsStatus.Successful;
                    response.Host = host;
                    try
                    {
                        var statisticsResponse = host.GetInstanceStatistics();
                        if (statisticsResponse != null)
                        {
                            response.InstanceStatistics = statisticsResponse.Statistics;

                        }
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Unable to retreieve host statistics!");
                    }

                }

            }
            catch (Exception ex)
            {
                Logger.Error(ex);
                response.Status = VoiceHostDetailsStatus.Failed;
                response.Message = "Error: " + ex.Message;
            }

            return response;

        }

        public HostConnectionDiagnosticsResponse GetHostConnectionDiagnostics(string apiKey, int id)
        {
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " + OperationContext.Current.GetClientIPAddress());
                return new HostConnectionDiagnosticsResponse { Status = HostConnectionDiagnosticsStatus.Failed, Message = "The Api Key provided did not match what was expected." };
            }

            var response = new HostConnectionDiagnosticsResponse();
            try
            {
                var host = VoiceHostManager.Instance.Hosts.FirstOrDefault(p => p.ID == id);

                if (host == null)
                {
                    response.Status = HostConnectionDiagnosticsStatus.HostNotFound;
                }
                else
                {
                    response.Status = HostConnectionDiagnosticsStatus.Successful;
                    response.Host = host;
                    try
                    {
                        var hostResponse = host.GetConnectionDiagnostics();
                        if (hostResponse != null)
                        {
                            response.Diagnostics = hostResponse.Diagnostics;

                        }
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Unable to retreieve host statistics!");
                    }

                }

            }
            catch (Exception ex)
            {
                Logger.Error(ex);
                response.Status = HostConnectionDiagnosticsStatus.Failed;
                response.Message = "Error: " + ex.Message;
            }

            return response;

        }


        public VoiceInstanceType[] GetVoiceInstanceTypes(string apiKey)
        {
            return (VoiceInstanceType[])Enum.GetValues(typeof (VoiceInstanceType));
        }
    }
}
