﻿using System.Collections.Generic;
using Curse.Logging;
using Curse.ServiceAuthentication;
using Curse.Voice.Helpers;
using Curse.Voice.HostManagement.Models;
using Curse.Voice.HostManagement.Responses;
using Curse.Voice.Service.Extensions;
using Curse.Voice.Service.GeoCoding;
using Curse.Voice.Service.Helpers;
using Curse.Voice.Service.Models;
using Curse.Voice.Service.Responses;
using Curse.Voice.Service.ServiceModels;
using System;
using System.Collections.Concurrent;
using System.Data;
using System.Diagnostics;
using System.Linq;
using System.Net;
using System.ServiceModel;
using System.Threading;
using Curse.CloudFlare;
using Curse.Voice.Contracts;
using Curse.Voice.Service.Requests;
using System.Threading.Tasks;

namespace Curse.Voice.Service
{
    [ServiceBehavior(
    AddressFilterMode = AddressFilterMode.Any,
    ConcurrencyMode = ConcurrencyMode.Multiple,
    UseSynchronizationContext = false,
    InstanceContextMode = InstanceContextMode.PerCall)]
    public class CurseVoiceCoreService : ICurseVoiceService
    {

        static CurseVoiceCoreService()
        {
            new Thread(HostUpdateDeploymentThread) { Name = "HostUpdateThread", IsBackground = true }.Start();
        }

        #region Client Methods


        public AutoMatchHandshakeResponse AutoMatchHandshake(int attempt, Int64 autoMatchKey)
        {
            var response = new AutoMatchHandshakeResponse { Status = AutoMatchHandshakeStatus.Error };

            try
            {
                if (attempt == 1) // Just create the handshake. Do not return anything
                {
                    VoiceHandshake.CreateLegacy(autoMatchKey);
                    response.HasMatches = false;
                }
                else
                {
                    var handshake = VoiceHandshake.CheckLegacy(autoMatchKey);
                    response.HasMatches = handshake.HasMatches;
                    response.PlayerCount = handshake.PlayerCount;
                }

                response.Status = AutoMatchHandshakeStatus.Successful;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "AutoMatchCheckin failed");
                response.StatusMessage = "Error: " + ex.Message;
            }

            return response;
        }


        public DeclineAutoMatchResponse DeclineAutoMatch(Int64 autoMatchKey)
        {
            return DoDeclineAutoMatch(AuthenticationProvider.CurrentSession.UserID, autoMatchKey);
        }

        public DeclineAutoMatchResponse DeclineAutoMatchV2(string apiKey, int userID, long autoMatchKey)
        {
            if(!TestApiKey(apiKey, IPAddress.None))
            {
                return new DeclineAutoMatchResponse { Status = DeclineAutoMatchStatus.Error };
            }

            return DoDeclineAutoMatch(userID, autoMatchKey);
        }

        private DeclineAutoMatchResponse DoDeclineAutoMatch(int userID, long autoMatchKey)
        {
            try
            {
                VoiceHandshake.Decline(userID, autoMatchKey);
                return new DeclineAutoMatchResponse { Status = DeclineAutoMatchStatus.Successful };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to decline auto match.");
                return new DeclineAutoMatchResponse { Status = DeclineAutoMatchStatus.Error };
            }
        }

        public AutoMatchHandshakeResponse AutoMatchHandshakeByDisplayName(int attempt, Int64 autoMatchKey, string displayName, int? gameID)
        {
            var response = new AutoMatchHandshakeResponse { Status = AutoMatchHandshakeStatus.Error };
            var session = AuthenticationProvider.CurrentSession;

            return DoAutoMatchHandshakeByDisplayName(session.UserID, attempt, autoMatchKey, displayName, gameID);
        }

        public AutoMatchHandshakeResponse AutoMatchHandshakeByDisplayNameV2(string apiKey, int userID, int attempt, long autoMatchKey, string displayName, int? gameID)
        {
            if (!TestApiKey(apiKey, IPAddress.None))
            {
                return new AutoMatchHandshakeResponse { Status = AutoMatchHandshakeStatus.Error };
            }

            return DoAutoMatchHandshakeByDisplayName(userID, attempt, autoMatchKey, displayName, gameID);
        }

        private AutoMatchHandshakeResponse DoAutoMatchHandshakeByDisplayName(int userID, int attempt, long autoMatchKey, string displayName, int? gameID)
        {
            var response = new AutoMatchHandshakeResponse { Status = AutoMatchHandshakeStatus.Error };

            try
            {
                if (string.IsNullOrEmpty(displayName))
                {
                    Logger.Warn("Auto match handshake called with a null or empty display name.", new { attempt, autoMatchKey, DisplayName = displayName ?? "NULL", GameID = gameID.HasValue ? gameID.Value.ToString() : "NULL" });
                    return new AutoMatchHandshakeResponse { Status = AutoMatchHandshakeStatus.Error };
                }

                if (attempt == 1) // Just create the handshake. Do not return anything
                {
                    VoiceHandshake.Create(autoMatchKey, displayName, userID);
                    response.HasMatches = false;
                }
                else if (attempt == 2)
                {
                    var handshake = VoiceHandshake.Check(autoMatchKey);
                    if (handshake.HasMatches)
                    {
                        // Calculate the match's fingerprint
                        var fingerprint = VoiceHandshake.CalculateFingerprint(gameID, handshake.Users);

                        // Update the current handshake's fingerprint
                        VoiceHandshake.UpdateFingerprint(autoMatchKey, fingerprint);
                    }

                    response.HasMatches = handshake.HasMatches;
                    response.PlayerCount = handshake.PlayerCount;
                    response.Users = handshake.Users;
                }
                else
                {
                    var handshake = VoiceHandshake.Check(autoMatchKey);
                    if (handshake.HasMatches)
                    {
                        var fingerprint = VoiceHandshake.CalculateFingerprint(gameID, handshake.Users);

                        var existingInstance = VoiceInstance.GetActiveByAutoMatchKey(autoMatchKey);

                        // We found an existing instance, so simply use it
                        if (existingInstance != null)
                        {
                            response.ExistingInviteCode = existingInstance.InviteCode;
                            response.ExistingInstanceCode = existingInstance.Code.ToString();

                        }
                        else
                        {
                            // Look for an earlier instance, with this same fingerprint. If one is found, simply update the related voice instance.
                            var priorInstance = VoiceInstance.GetByAutoMatchFingerprint(fingerprint);

                            if (priorInstance != null) // If one is found, update its auto match key
                            {
                                VoiceInstance.MigrateVoiceInstanceKey(autoMatchKey, priorInstance.ID);
                                response.ExistingInviteCode = priorInstance.InviteCode;
                                response.ExistingInstanceCode = priorInstance.Code.ToString();
                            }
                        }
                        // Update the current handshake's fingerprint
                        VoiceHandshake.UpdateFingerprint(autoMatchKey, fingerprint);
                    }

                    response.HasMatches = handshake.HasMatches;
                    response.PlayerCount = handshake.PlayerCount;
                    response.Users = handshake.Users;
                }

                response.Status = AutoMatchHandshakeStatus.Successful;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "AutoMatchCheckin failed");
                response.StatusMessage = "Error: " + ex.Message;
            }

            return response;
        }

        public AutoMatchVoiceSessionResponse AutoMatchVoiceSession(string userDisplayName, int? gameID, Version clientVersion, Int64 autoMatchKey)
        {
            var response = new AutoMatchVoiceSessionResponse { Status = AutoMatchVoiceSessionStatus.Error };
            var session = AuthenticationProvider.CurrentSession;

            // Prevent invalid auto match keys
            if (autoMatchKey < 0)
            {
                return response;
            }

            try
            {
                // Try to find this instance (only if it is active)
                var instance = VoiceInstance.GetActiveByAutoMatchKey(autoMatchKey);

                if (instance == null)
                {
                    var status = DoCreateVoiceSession(false, userDisplayName, clientVersion, VoiceInstanceType.AutoMatch, VoiceInstanceMode.Audio, gameID, session.UserID, autoMatchKey, null, null, null, out instance);

                    // Incompatible Client
                    if (status == CreateVoiceSessionStatus.IncompatibleClient)
                    {
                        response.Status = AutoMatchVoiceSessionStatus.IncompatibleClient;
                        return response;
                    }

                    // Auto-match already exists (likely created by the user that just checked in)
                    if (status == CreateVoiceSessionStatus.AlreadyExists)
                    {
                        instance = VoiceInstance.GetActiveByAutoMatchKey(autoMatchKey);
                    }
                }

                if (instance != null)
                {
                    response = new AutoMatchVoiceSessionResponse
                    {
                        InstanceCode = instance.Code.ToString(),
                        GameID = instance.GameID,
                        InviteUrl = instance.InviteUrl,
                        HostName = instance.VoiceHost.HostName,
                        IPAddress = instance.VoiceHost.IPAddress,
                        Port = CoreServiceConfiguration.Instance.VoicePortNumber,
                        Status = AutoMatchVoiceSessionStatus.Successful,
                        HostID = instance.VoiceHostID,
                        RegionName = instance.VoiceHost.Region.DisplayName
                    };
                }

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "AutoMatchVoiceSession failed");
            }
            
            return response;
        }

        public CreateVoiceSessionResponse CreateVoiceSessionOnHost(string apiKey, string userDisplayName, int userID, int hostID)
        {
            var ipAddress = OperationContext.Current.GetClientIPAddress();

            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " + ipAddress);
                return new CreateVoiceSessionResponse { Status = CreateVoiceSessionStatus.Error };
            }

            try
            {
                var host = VoiceHost.GetByID(hostID);

                if (host == null)
                {
                    Logger.Warn("Unable to create voice session. Host could not be found:" + hostID);
                    return new CreateVoiceSessionResponse { Status = CreateVoiceSessionStatus.Error, StatusMessage = "Unknown voice host: " + hostID };
                }

                VoiceInstance instance;

                var status = DoCreateVoiceSession(true, userDisplayName, null, VoiceInstanceType.AdHoc, VoiceInstanceMode.Audio, null, 0, null, null, host, null, out instance);

                if (status != CreateVoiceSessionStatus.Successful)
                {
                    return new CreateVoiceSessionResponse
                    {
                        Status = status,
                    };
                }

                return new CreateVoiceSessionResponse
                {
                    InstanceCode = instance.Code.ToString(),
                    GameID = null,
                    InviteUrl = instance.InviteUrl,
                    HostName = instance.VoiceHost.HostName,
                    IPAddress = instance.VoiceHost.IPAddress,
                    Port = CoreServiceConfiguration.Instance.VoicePortNumber,
                    Status = CreateVoiceSessionStatus.Successful,
                    HostID = instance.VoiceHostID,
                    RegionName = instance.VoiceHost.Region.DisplayName,
                };

            }
            catch (Exception ex)
            {
                return new CreateVoiceSessionResponse
                {
                    Status = CreateVoiceSessionStatus.Error,
                    StatusMessage = ex.Message + ", Stack: " + ex.StackTrace
                };
            }

        }

        private static bool IsUserSpamming(int userID)
        {
            var lastFive = VoiceInstance.GetCountByUserID(userID, DateTime.UtcNow.AddMinutes(-5));
            if (lastFive > 10)
            {
                return true;
            }

            var lastHour = VoiceInstance.GetCountByUserID(userID, DateTime.UtcNow.AddHours(-1));
            if (lastHour > 200)
            {
                return true;
            }

            return false;
        }

        public CreateVoiceSessionResponse CreateVoiceSession(string userDisplayName, int? gameID, int? userID, bool destroyExistingSession, Version clientVersion)
        {

            try
            {
                var session = AuthenticationProvider.CurrentSession;

                if (VoiceBanManager.IsBanned(session.UserID))
                {
                    return new CreateVoiceSessionResponse
                    {
                        Status = CreateVoiceSessionStatus.Throttled
                    };
                }

                if (IsUserSpamming(session.UserID))
                {
                    VoiceBanManager.BanForSpamming(session.UserID);

                    return new CreateVoiceSessionResponse
                    {
                        Status = CreateVoiceSessionStatus.Throttled
                    };
                }

                VoiceInstance instance;
                var status = DoCreateVoiceSession(false, userDisplayName, clientVersion, VoiceInstanceType.AdHoc, VoiceInstanceMode.Audio, gameID, session.UserID, null, null, null, null, out instance);

                if (status == CreateVoiceSessionStatus.Successful)
                {
                    return new CreateVoiceSessionResponse
                    {
                        InstanceCode = instance.Code.ToString(),
                        GameID = gameID,
                        InviteUrl = instance.InviteUrl,
                        HostName = instance.VoiceHost.HostName,
                        IPAddress = instance.VoiceHost.IPAddress,
                        Port = CoreServiceConfiguration.Instance.VoicePortNumber,
                        Status = CreateVoiceSessionStatus.Successful,
                        HostID = instance.VoiceHostID,
                        RegionName = instance.VoiceHost.Region.DisplayName,
                    };
                }

                return new CreateVoiceSessionResponse
                {
                    Status = status
                };

            }
            catch (Exception ex)
            {
                return new CreateVoiceSessionResponse
                    {
                        Status = CreateVoiceSessionStatus.Error,
                        StatusMessage = ex.Message + ", Stack: " + ex.StackTrace
                    };
            }

        }

        private CreateVoiceSessionStatus DoCreateVoiceSession(bool isWebCall, string userDisplayName, Version clientVersion, VoiceInstanceType type, VoiceInstanceMode mode, int? gameID, int userID, Int64? autoMatchKey, string externalID, VoiceHost voiceHost, IPAddress clientIPAddress, out VoiceInstance instance, int preferredRegion = 0)
        {
            if (clientVersion == null && !isWebCall)
            {
                var ipAddress = OperationContext.Current.GetClientIPAddress();
                Logger.Warn("Attempt to create a session with a missing client version!", new { UserName = userDisplayName, IPAddress = ipAddress });
                instance = null;
                return CreateVoiceSessionStatus.IncompatibleClient;
            }

            if (isWebCall)
            {
                userDisplayName = userDisplayName ?? "Curse Voice Test";
            }
            else if (!string.IsNullOrEmpty(userDisplayName) && userDisplayName.Length > 128)
            {
                Logger.Warn("Attempt to create a session with an invalid user display name: " + userDisplayName);
                instance = null;
                return CreateVoiceSessionStatus.Error;
            }

            var attemptedHostIDs = new HashSet<int>();
            var hostSpecific = voiceHost != null;
            var maxAttempts = hostSpecific ? 1 : 3;
            var provisionSuccessful = false;
            var characterLength = (type == VoiceInstanceType.Group || type == VoiceInstanceType.Friend) ? 10 : autoMatchKey.HasValue ? 6 : 4;

            var createdInstance = new VoiceInstance
            {
                Code = Guid.NewGuid(),
                DateCreated = DateTime.UtcNow,
                GameID = gameID,
                Status = VoiceInstanceStatus.Active,
                UserID = userID,
                InviteCode = UrlShortner.RandomCharacters(characterLength),
                InviteDisplayName = userDisplayName,
                AutoMatchKey = autoMatchKey,
                ExternalID = externalID,
                Type = type,
                Mode = mode
            };

            var region = preferredRegion > 0 ? VoiceRegion.GetByID(preferredRegion) : null;

            var provisionResult = new VoiceHostProvisionResult { IsSuccessful = false };

            // Try to provision the instance
            for (var attempt = 1; attempt <= maxAttempts; attempt++)
            {
                // Find the best host for this user
                if (!hostSpecific)
                {
                    // Determine the best host for this request
                    voiceHost = region == null
                        ? VoiceHostManager.Instance.FindBestHost(clientIPAddress ?? OperationContext.Current.GetClientIPAddress(), clientVersion, mode, attemptedHostIDs)
                        : VoiceHostManager.Instance.FindBestHost(region, clientVersion, mode, attemptedHostIDs);


                    // If no host is returned, return out with a general failure
                    if (voiceHost == null)
                    {
                        if (VoiceHostManager.Instance.Hosts.All(p => p.Status != VoiceHostStatus.Online))
                        {
                            Logger.Warn("Unable to provision a voice session. No hosts are online.");
                            instance = null;
                            return CreateVoiceSessionStatus.NoHostsAvailable;
                        }

                        if (VoiceHostManager.Instance.Hosts.Any(p => clientVersion >= p.Version.MinimumClientVersion))
                        {
                            Logger.Warn("Unable to provision a voice session. No hosts are available for provisioning.");
                            instance = null;
                            return CreateVoiceSessionStatus.NoHostsAvailable;
                        }

                        Logger.Warn("User tried to create a voice session from an incompatible client version.", new { ClientVersion = clientVersion == null ? "Unknown Client Version" : clientVersion.ToString() });
                        instance = null;
                        return CreateVoiceSessionStatus.IncompatibleClient;
                    }

                    attemptedHostIDs.Add(voiceHost.ID);
                }

                createdInstance.VoiceHostID = voiceHost.ID;

                // Attempt to provision
                provisionResult = voiceHost.ProvisionInstance(createdInstance);

                if (provisionResult.IsSuccessful)
                {
                    provisionSuccessful = true;
                    break;
                }

                Logger.Warn(provisionResult.Exception, "Failed to provision voice instance on attempt " + attempt + ": " + provisionResult.LogMessage, provisionResult.LogData);
            }

            if (!provisionSuccessful)
            {
                instance = null;
                Logger.Error(provisionResult.Exception, "Failed to provision voice instance after " + maxAttempts + " attempts", new { LastAttempt = provisionResult.LogData });
                return CreateVoiceSessionStatus.NoHostsAvailable;
            }

            using (var conn = DatabaseHelper.Instance.GetConnection())
            {

                var saveStatus = VoiceInstanceSaveStatus.Error;

                for (var j = 0; j < 20; j++)
                {
                    saveStatus = createdInstance.Save(conn);

                    // Save was successful
                    if (saveStatus == VoiceInstanceSaveStatus.Success)
                    {
                        break;
                    }

                    // Auto match key is a dupe, so return out
                    if (saveStatus == VoiceInstanceSaveStatus.DuplicateAutoMatchKey || saveStatus == VoiceInstanceSaveStatus.DuplicateExternalID)
                    {
                        instance = null;
                        return CreateVoiceSessionStatus.AlreadyExists;
                    }

                    // Use 4 or 5 characters, depending on the number of attempts                                                
                    if (characterLength < 5 && j >= 5)
                    {
                        characterLength = 5;
                    }
                    else if (characterLength < 6 && j >= 10)
                    {
                        characterLength = 6;
                    }

                    // Try a new code
                    createdInstance.InviteCode = UrlShortner.RandomCharacters(characterLength);
                }

                // Failed to save, after 20 attempts
                if (saveStatus != VoiceInstanceSaveStatus.Success)
                {
                    Logger.Error("Failed to save new voice instance to the database after 10 attempts.", new { Status = saveStatus });
                    instance = null;
                    return CreateVoiceSessionStatus.Error;
                }
            }

            instance = createdInstance;
            return CreateVoiceSessionStatus.Successful;
        }

        public GetVoiceSessionResponse GetVoiceSession(string inviteCode, Version clientVersion)
        {
            return VoiceSessionManager.GetVoiceSession(inviteCode, clientVersion);
        }

        public ReportGameSessionsResponse ReportGameSessions(GameSessionMetric[] metrics)
        {
            var session = AuthenticationProvider.CurrentSession;
            return DoReportGameSessions(session.UserID, metrics);
        }

        public ReportGameSessionsResponse ReportGameSessionsV2(string apiKey, int userID, GameSessionMetric[] metrics)
        {
            if(!TestApiKey(apiKey, IPAddress.None))
            {
                return new ReportGameSessionsResponse { Successful = false };
            }
            return DoReportGameSessions(userID, metrics);
        }

        private ReportGameSessionsResponse DoReportGameSessions(int userID, GameSessionMetric[] metrics)
        {
            if (metrics == null || metrics.Length == 0)
            {
                return new ReportGameSessionsResponse() { Successful = false };
            }


            if (userID == 0)
            {
                Logger.Debug("User submitted metrics from an unknown auth session.");
                return new ReportGameSessionsResponse() { Successful = false };
            }

            if (metrics.Length > 20)
            {
                metrics = metrics.Where(p => p.Successful).ToArray();

                if (metrics.Length > 20)
                {
                    Logger.Warn("User submitted too many game metrics.", new { Count = metrics.Length, Metrics = metrics.Take(10).ToArray() });
                    return new ReportGameSessionsResponse() { Successful = false };
                }
            }

            foreach (var metric in metrics)
            {
                metric.UserID = userID;
            }

            ReportingHelper.AddMetrics(metrics);
            return new ReportGameSessionsResponse() { Successful = true };
        }


        public BasicVoiceServiceResponse ReportGameState(string sessionGuid, int gameID)
        {
            return DoReportGameState(sessionGuid, gameID);
        }

        public BasicVoiceServiceResponse ReportGameStateV2(string apiKey, string sessionGuid, int gameID)
        {
            if(!TestApiKey(apiKey, IPAddress.None))
            {
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Forbidden };
            }
            return DoReportGameState(sessionGuid, gameID);
        }

        private BasicVoiceServiceResponse DoReportGameState(string sessionGuid, int gameID)
        {
            if (string.IsNullOrWhiteSpace(sessionGuid))
            {
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Failed };
            }

            Guid session;

            if (!Guid.TryParse(sessionGuid, out session))
            {
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Failed };
            }

            return VoiceSessionManager.ReportGameState(sessionGuid, gameID);
        }

        private static readonly Dictionary<int, int> VoiceRegionToFriendsRegionMap = new Dictionary<int, int>
        {
            // 1 = US-East
            // 3 = EU-West
            // 5 = AP-Southeast
            {1, 1},
            {2, 1},
            {3, 3},
            {4, 5},
            {5, 5},
            {6, 5},
            {7, 1},
            {8, 1},
            {9, 1},
            {10, 3},
        };

        #endregion

        public int GetFriendsServiceRegionForIP(string ipAddress, string apiKey)
        {
            try
            {
                if (!TestApiKey(apiKey, OperationContext.Current.GetClientIPAddress()))
                {
                    return -1;
                }

                IPAddress address;
                if (!IPAddress.TryParse(ipAddress, out address))
                {
                    return -2;
                }

                var range = IPRange.GetRange(address.ToInt64());
                if (range == null)
                {
                    return -3;
                }

                int friendsRegion;
                if (!VoiceRegionToFriendsRegionMap.TryGetValue(range.VoiceRegionID, out friendsRegion))
                {
                    return 1;
                }

                return friendsRegion;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[GetFriendsServiceRegionForIP] Unknown error.", new { ipAddress });
                return -4;
            }
        }

        #region Web Site Methods

        public UnlockVoiceSessionResponse UnlockVoiceSession(string apiKey, IPAddress ipAddress, string sessionCode, int userID)
        {
            try
            {
                if (!TestApiKey(apiKey, ipAddress))
                {
                    return new UnlockVoiceSessionResponse { Status = UnlockVoiceSessionStatus.Forbidden };
                }

                var instance = VoiceInstance.GetByCode(sessionCode);

                if (instance == null)
                {
                    return new UnlockVoiceSessionResponse { Status = UnlockVoiceSessionStatus.NotFound };
                }

                // Instance has already been unlocked
                if (instance.Type != VoiceInstanceType.Friend)
                {
                    return new UnlockVoiceSessionResponse { Status = UnlockVoiceSessionStatus.Successful, InviteCode = instance.InviteCode };
                }

                // Ensure the call is a friend call and the requestor belongs to the friend session
                if (instance.ExternalID.Split(':').Select(int.Parse).All(id => id != userID))
                {
                    Logger.Warn("User attempted to unlock a session that they have no access to.", new { instance, userID });
                    return new UnlockVoiceSessionResponse { Status = UnlockVoiceSessionStatus.Forbidden };
                }

                if (instance.VoiceHost.UnlockInstance(instance))
                {
                    instance.Unlock();

                    return new UnlockVoiceSessionResponse
                    {
                        Status = UnlockVoiceSessionStatus.Successful,
                        InviteCode = instance.InviteCode
                    };
                }

                return new UnlockVoiceSessionResponse { Status = UnlockVoiceSessionStatus.Error };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to unlock voice session.");
                return new UnlockVoiceSessionResponse { Status = UnlockVoiceSessionStatus.Error };
            }
        }

        public KickUsersFromGroupCallResponse KickUsersFromGroupCall(string apiKey, IPAddress address, Guid groupID, int kickerID, int[] usersToKick)
        {
            switch (RemoveUsersFromGroupCall(apiKey, address, groupID, kickerID, usersToKick, UserDisconnectReason.Kicked).Status)
            {
                case RemoveUsersFromGroupCallStatus.Forbidden:
                    return new KickUsersFromGroupCallResponse { Status = KickUsersFromGroupCallStatus.Forbidden };
                case RemoveUsersFromGroupCallStatus.NotFound:
                    return new KickUsersFromGroupCallResponse { Status = KickUsersFromGroupCallStatus.NotFound };
                case RemoveUsersFromGroupCallStatus.Successful:
                    return new KickUsersFromGroupCallResponse { Status = KickUsersFromGroupCallStatus.Successful };
                default:
                    return new KickUsersFromGroupCallResponse { Status = KickUsersFromGroupCallStatus.Error };
            }
        }

        public RemoveUsersFromGroupCallResponse RemoveUsersFromGroupCall(string apiKey, IPAddress address, Guid groupID, int requestorID, int[] usersToRemove, UserDisconnectReason reason)
        {
            if (!TestApiKey(apiKey, address))
            {
                return new RemoveUsersFromGroupCallResponse { Status = RemoveUsersFromGroupCallStatus.Forbidden };
            }

            try
            {
                var instance = VoiceInstance.GetActiveByExternalIDAndType(groupID.ToString(), VoiceInstanceType.Group);
                if (instance == null || instance.Status != VoiceInstanceStatus.Active)
                {
                    return new RemoveUsersFromGroupCallResponse { Status = RemoveUsersFromGroupCallStatus.NotFound };
                }

                if (instance.VoiceHost.RemoveUsers(instance, requestorID, usersToRemove, reason))
                {
                    return new RemoveUsersFromGroupCallResponse { Status = RemoveUsersFromGroupCallStatus.Successful };
                }

                return new RemoveUsersFromGroupCallResponse { Status = RemoveUsersFromGroupCallStatus.Error };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to remove users from group voice session.");
                return new RemoveUsersFromGroupCallResponse { Status = RemoveUsersFromGroupCallStatus.Error };
            }
        }

        public BasicVoiceServiceResponse AddPendingVoiceUsers(string apiKey, IPAddress ipAddress, string instanceGuid, int userID, PendingUserRequest[] pendingUsers)
        {
            if (!TestApiKey(apiKey, ipAddress))
            {
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Forbidden };
            }

            try
            {
                var instance = VoiceInstance.GetByCode(instanceGuid);

                if (instance == null)
                {
                    Logger.Warn("User attempted to add a pending user to a non-existent voice session.");
                    return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.NotFound };
                }

                if (instance.Status != VoiceInstanceStatus.Active)
                {
                    Logger.Debug("User attempted to add a pending user to an inactive voice session.");
                    return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.NotFound };
                }

                var status = instance.VoiceHost.AddPendingUsers(instance, pendingUsers);

                if (status == AddPendingUsersStatus.Success)
                {
                    return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Successful };
                }

                switch (status)
                {
                    case AddPendingUsersStatus.Forbidden:
                        return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Forbidden };
                    case AddPendingUsersStatus.NotConnected:
                        return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Failed };
                    case AddPendingUsersStatus.NotFound:
                        return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.NotFound };
                    default:
                        return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Error };
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to add pending users to voice host.");
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Error };
            }
        }

        public BasicVoiceServiceResponse RemovePendingVoiceUser(string apiKey, IPAddress ipAddress, string inviteCode, int userID)
        {
            if (!TestApiKey(apiKey, ipAddress))
            {
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Forbidden };
            }

            try
            {
                var instance = VoiceInstance.GetByInviteCode(inviteCode);
                if (instance == null)
                {
                    Logger.Warn("User attempted to remove a pending user from a non-existent voice session.");
                    return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.NotFound };
                }

                if (instance.Status != VoiceInstanceStatus.Active)
                {
                    Logger.Debug("User attempted to remove a pending user from an inactive voice session.");
                    return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.NotFound };
                }

                var status = instance.VoiceHost.RemovePendingUser(instance, userID);
                if (status == RemovePendingUserStatus.Successful)
                {
                    return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Successful };
                }

                switch (status)
                {
                    case RemovePendingUserStatus.Forbidden:
                        return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Forbidden };
                    case RemovePendingUserStatus.NotConnected:
                        return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Failed };
                    case RemovePendingUserStatus.NotFound:
                        return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.NotFound };
                    default:
                        return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Error };
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to remove pending user from voice host.");
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Error };
            }
        }

        public FindVoiceSessionResponse FindVoiceSession(string apiKey, string inviteCode)
        {

            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method, 'FindVoiceSession'");
                return new FindVoiceSessionResponse { Status = FindVoiceSessionStatus.Error };
            }

            try
            {
                var instance = VoiceInstance.GetByInviteCode(inviteCode);
                if (instance == null)
                {
                    return new FindVoiceSessionResponse
                    {
                        Status = FindVoiceSessionStatus.NotFound
                    };
                }

                return new FindVoiceSessionResponse
                {
                    Status = FindVoiceSessionStatus.Successful,
                    CreatorName = instance.InviteDisplayName,
                    GameID = instance.GameID
                };

            }
            catch (Exception ex)
            {
                return new FindVoiceSessionResponse
                {
                    Status = FindVoiceSessionStatus.Error,
                    StatusMessage = ex.Message + ", Stack: " + ex.StackTrace
                };
            }

        }

        public FindVoiceSessionResponse GetVoiceSessionDetails(string apiKey, string inviteCode)
        {

            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method, 'FindVoiceSession'");
                return new FindVoiceSessionResponse { Status = FindVoiceSessionStatus.Error };
            }

            try
            {
                var instance = VoiceInstance.GetByInviteCode(inviteCode);
                if (instance == null || instance.Status != VoiceInstanceStatus.Active)
                {
                    return new FindVoiceSessionResponse
                    {
                        Status = FindVoiceSessionStatus.NotFound
                    };
                }

                return new FindVoiceSessionResponse
                {
                    Status = FindVoiceSessionStatus.Successful,
                    CreatorName = instance.InviteDisplayName,
                    GameID = instance.GameID
                };

            }
            catch (Exception ex)
            {
                return new FindVoiceSessionResponse
                {
                    Status = FindVoiceSessionStatus.Error,
                    StatusMessage = ex.Message + ", Stack: " + ex.StackTrace
                };
            }

        }

        public ChangeVoiceSessionModeResponse ChangeVoiceSessionMode(string apiKey, IPAddress ipAddress, string sessionCode, int userID, VoiceInstanceMode mode)
        {
            try
            {
                if (!TestApiKey(apiKey, ipAddress))
                {
                    return new ChangeVoiceSessionModeResponse { Status = ChangeVoiceSessionModeStatus.Forbidden };
                }

                var instance = VoiceInstance.GetByCode(sessionCode);

                if (instance == null)
                {
                    return new ChangeVoiceSessionModeResponse { Status = ChangeVoiceSessionModeStatus.NotFound };
                }

                // Instance has already been unlocked
                if (instance.Mode == mode)
                {
                    return new ChangeVoiceSessionModeResponse { Status = ChangeVoiceSessionModeStatus.Successful };
                }


                if (instance.ChangeMode(mode))
                {
                    return new ChangeVoiceSessionModeResponse
                    {
                        Status = ChangeVoiceSessionModeStatus.Successful,
                    };
                }

                return new ChangeVoiceSessionModeResponse { Status = ChangeVoiceSessionModeStatus.Error };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to change voice session mode.");
                return new ChangeVoiceSessionModeResponse { Status = ChangeVoiceSessionModeStatus.Error };
            }
        }

        public bool PruneHandshakes(string apiKey)
        {
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method, 'PruneHandshakes'");
                return false;
            }

            try
            {
                return RunPruneOperation("PruneVoiceHandshakes");
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to prune voice handshakes.");
            }

            return true;
        }

        public bool PruneInstances(string apiKey)
        {
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method, 'PruneInstances'");
                return false;
            }

            try
            {
                return RunPruneOperation("PruneOfflineVoiceInstances");
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to prune voice instances.");
            }

            return true;
        }

        public bool UpdateInstancesWithMaxUsers(string apiKey, UpdateInstanceUserCountRequest[] updatedCounts)
        {
            if (!string.Equals(apiKey, CoreServiceConfiguration.Instance.ApiKey))
            {
                Logger.Warn("Attempt was made to access an API restricted method, 'UpdateInstanceWithMaxUsers'");
                return false;
            }

            foreach (var updatedCount in updatedCounts)
            {
                var instanceID = VoiceInstance.GetIDFromDatabaseByCode(updatedCount.InstanceCode);

                if (!instanceID.HasValue)
                {
                    Logger.Warn("Unable to find a voice instance with code " + updatedCount.InstanceCode);
                    continue;
                }

                VoiceInstance.UpdateMaxUsersInCall(updatedCount.MaxUsers, instanceID.Value);
            }
            return true;
        }

        private bool RunPruneOperation(string operationName)
        {
            using (var conn = DatabaseHelper.Instance.GetConnection())
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandTimeout = 300;
                    cmd.CommandText = operationName;
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.ExecuteNonQuery();
                }
            }
            return true;
        }


        #endregion

        #region Server Methods

        public string HealthCheck()
        {
            return "I am alive!";
        }


        public string TestDatabaseModels()
        {

#if DEBUG
#else
            throw new Exception("Not allowed outside of debug mode.");
#endif
            DateTime startTime = DateTime.UtcNow;
            for (int i = 0; i < 100; i++)
            {
                var voiceInstance = new VoiceInstance
                    {
                        Code = Guid.NewGuid(),
                        DateCreated = DateTime.UtcNow,
                        GameID = null,
                        VoiceHostID = 1
                    };
                voiceInstance.Save();


            }

            return (DateTime.UtcNow - startTime).TotalMilliseconds.ToString("###,##0.00") + "ms";

        }

        public string TestIPAddress(string rawAddress)
        {
            IPAddress ipAddress;

            if (!IPAddress.TryParse(rawAddress, out ipAddress))
            {
                return "Invalid IP Address: " + rawAddress;
            }

            var range = IPRange.GetRange(ipAddress.ToInt64());

            if (range == null)
            {
                return "Unable to geo locate IP address: " + rawAddress;
            }

            var voiceRegion = VoiceRegion.GetByID(range.VoiceRegionID);

            if (voiceRegion == null)
            {
                return "IP address was mapped, but no voice region has been set for it: " + rawAddress;
            }

            return "This IP address has been mapped to country '" + range.CountryCode + "' and region '" + (range.RegionCode ?? "N/A") + "'. This maps to voice region '" + voiceRegion.Name + "' with an effective region of '" + voiceRegion.EffectiveRegion.Name + "'";

        }

        private void RegisterDns(VoiceHost host)
        {
            CloudFlareApi.UpdateDnsHost(CoreServiceConfiguration.Instance.HostDomainName, host.PublicHostName, host.IPAddress);
        }

        private void RegisterDnsTask(VoiceHost host)
        {
            var attempt = 0;
            while (true)
            {
                try
                {
                    RegisterDns(host);
                    host.ChangeStatus(VoiceHostStatus.Online);
                    return;
                }
                catch (Exception ex)
                {
                    if (++attempt < 10)
                    {
                        Logger.Warn(ex, "Failed to register DNS with CloudFlare after " + attempt + " attempt.");
                        continue;
                    }

                    Logger.Error(ex, "Failed to register DNS with CloudFlare after " + attempt + " attempts. This host will not be routable.");
                }

                Thread.Sleep(5000);
            }
        }

        public BasicVoiceServiceResponse FailoverHost(string apiKey, int hostID)
        {
            var ipAddress = OperationContext.Current.GetClientIPAddress();
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " + ipAddress);
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Forbidden };
            }

            Logger.Info("Received an API request to failover a voice host:" + hostID);
            var host = VoiceHost.GetByID(hostID);
            if (host == null)
            {
                Logger.Warn("Unable to failover voice host. It was not found in the database.");
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.NotFound };
            }

            Logger.Warn("Starting failover for host: " + host.HostName);

            Task.Factory.StartNew(() =>
            {
                host.Failover(VoiceHostStatus.ShuttingDown);
                Logger.Warn("Changing status to offline: " + host.HostName);
                host.ChangeStatus(VoiceHostStatus.Offline);
            }, TaskCreationOptions.LongRunning);

            return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Successful };
        }

        public VoiceHostConfigurationResponse GetVoiceHostConfiguration(string apiKey, string externalID)
        {
            var ipAddress = OperationContext.Current.GetClientIPAddress();

            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " + ipAddress);
                return null;
            }

            Guid externalIDGuid;

            try
            {
                externalIDGuid = Guid.Parse(externalID);
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to parse guid from string: " + externalID);
                return null;
            }

            var voiceHost = VoiceHost.GetByExternalID(externalIDGuid);

            if (voiceHost == null)
            {
                Logger.Warn("Failed to get voice host with external ID: " + externalID);
                return null;
            }


            return new VoiceHostConfigurationResponse { Configuration = voiceHost.GetConfigurationContract() };
        }

        public RegisterVoiceHostResponse RegisterVoiceHost(string apiKey, string externalID, string hostName, Version version)
        {
            return DoRegisterVoiceHost(apiKey, externalID, hostName, version, VoiceInstanceMode.Audio);
        }

        public RegisterVoiceHostResponse RegisterVoiceHostV2(string apiKey, string externalID, string hostName, Version version, VoiceInstanceMode supportedModes)
        {
            return DoRegisterVoiceHost(apiKey, externalID, hostName, version, supportedModes);
        }

        private RegisterVoiceHostResponse DoRegisterVoiceHost(string apiKey, string externalID, string hostName, Version version, VoiceInstanceMode supportedModes)
        {
            var ipAddress = OperationContext.Current.GetClientIPAddress();

            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " + ipAddress);
                return new RegisterVoiceHostResponse { Status = RegisterVoiceHostStatus.Error };
            }

            Logger.Info("Registering voice host: " + externalID, new
            {
                hostName, 
                hostVersion = version.ToString(),
                hostIp = ipAddress.ToString(), 
                hostModes = supportedModes
            });

            try
            {
                var voiceHostVersion = VoiceHostVersion.GetByVersion(version);

#if CONFIG_DEBUG || CONFIG_STAGING

                // Get the IP address
                ipAddress = Dns.GetHostAddresses(hostName).FirstOrDefault(p => p.AddressFamily == System.Net.Sockets.AddressFamily.InterNetwork);

                if (ipAddress == null)
                {
                    throw new ArgumentException("IP address cannot be null");
                }

                if (voiceHostVersion == null)
                {
                    voiceHostVersion = new VoiceHostVersion
                    {
                        VersionString = version.ToString(),
                        MinimumClientVersionString = "1.0.0.0",
                        Environment = CoreServiceConfiguration.Instance.Environment
                    };
                    voiceHostVersion.SaveToDatabase();
                }
#endif

                var externalIDGuid = Guid.Parse(externalID);
                var voiceHost = VoiceHost.GetByIPAddressOrExternalID(ipAddress, externalIDGuid, true);
                var latestVoiceHostVersion = VoiceHostVersion.GetLatestByEnvironment(CoreServiceConfiguration.Instance.Environment);

                if (voiceHost == null)
                {
                    voiceHost = new VoiceHost();
                    var regionID = GetVoiceRegionFromIP(ipAddress);
                    voiceHost.RegionID = regionID;
                    voiceHost.SupportedModes = supportedModes;
                }
                else   // Mark offline any voice instances for this host
                {
                    voiceHost.MarkInstancesOffline();
                }

                voiceHost.ExternalID = externalIDGuid;
                voiceHost.DateOnline = DateTime.UtcNow;
                voiceHost.HostName = hostName;
                voiceHost.PublicHostName = VoiceHost.GeneratePublicHostName(hostName);
                voiceHost.IPAddress = ipAddress.ToString();
                voiceHost.Environment = CoreServiceConfiguration.Instance.Environment;

                if (voiceHostVersion == null)
                {
                    voiceHost.VersionID = latestVoiceHostVersion.ID;
                }
                else
                {
                    voiceHost.VersionID = voiceHostVersion.ID;
                }

                if (latestVoiceHostVersion == null)
                {
                    voiceHost.Status = VoiceHostStatus.Online;
                }
                else if (voiceHost.Version.Status == VoiceHostVersionStatus.Pending)
                {
                    Logger.Info("Voice host '" + hostName + "' is running pending version (" + voiceHost.VersionString + "). It will not auto update.", new { hostName, hostVersion = version.ToString(), hostIp = ipAddress.ToString() });
                    voiceHost.Status = VoiceHostStatus.Online;
                }
                else if (voiceHost.VersionID != latestVoiceHostVersion.ID)
                {
                    Logger.Info("Voice host '" + hostName + "' is running version (" + voiceHost.VersionString + "), and the latest version is (" + latestVoiceHostVersion.VersionString + ")", new { hostName, hostVersion = version.ToString(), hostIp = ipAddress.ToString() });
                    voiceHost.Status = VoiceHostStatus.Updating;
                }
                else
                {
                    voiceHost.Status = VoiceHostStatus.Online;
                }

                voiceHost.SaveToDatabase();

                if (voiceHost.ID == 0)
                {
                    voiceHost = VoiceHost.GetByIPAddressOrExternalID(ipAddress, externalIDGuid, true);

                    if (voiceHost == null)
                    {
                        Logger.Error("Failed to register new voice host. It could not be retrieved after being saved.");
                        return new RegisterVoiceHostResponse { Status = RegisterVoiceHostStatus.Error };
                    }
                }


#if true || !CONFIG_DEBUG && !CONFIG_STAGING

                // The version is out of date (and is also not the pending version, which is being deployed)
                if (voiceHost.Status == VoiceHostStatus.Updating)
                {
                    Logger.Info("Voice host is being queued for update: " + externalID, new { hostName, hostVersion = version.ToString(), hostIp = ipAddress.ToString() });
                    _hostUpdateQueue.Enqueue(voiceHost.ID);
                    return new RegisterVoiceHostResponse { Status = RegisterVoiceHostStatus.Error };
                }
#endif

#if !CONFIG_DEBUG && !CONFIG_STAGING
                // Update the host's DNS
                try
                {
                    RegisterDns(voiceHost);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to register voice host DNS on first attempt. Starting a thread...");
                    // Kick off a thread to register DNS
                    Task.Factory.StartNew(() => RegisterDnsTask(voiceHost));

                    // Return early
                    return new RegisterVoiceHostResponse { Status = RegisterVoiceHostStatus.Successful, MinimumClientVersion = voiceHostVersion.MinimumClientVersionString, DataRegionID = voiceHost.DataRegionID };
                }
#endif


                voiceHost.ChangeStatus(VoiceHostStatus.Online);

                try
                {
                    voiceHost.UpdateStatistics();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to update host stats, during startup!");
                }


                Logger.Info("Host has registered itself with the central service: " + voiceHost.HostName, new { voiceHost.ID, voiceHost.FullHostName, voiceHost.IPAddress });
                return new RegisterVoiceHostResponse { Status = RegisterVoiceHostStatus.Successful, MinimumClientVersion = voiceHostVersion.MinimumClientVersionString, DataRegionID = voiceHost.DataRegionID, Configuration = voiceHost.GetConfigurationContract() };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Voice host registration failed!");
                return new RegisterVoiceHostResponse { Status = RegisterVoiceHostStatus.Error };
            }
        }

        private static readonly ConcurrentQueue<int> _hostUpdateQueue = new ConcurrentQueue<int>();

        private static void HostUpdateDeploymentThread()
        {
            while (true)
            {
                Thread.Sleep(5000);

                try
                {
                    if (_hostUpdateQueue.Count == 0)
                    {
                        continue;
                    }

                    var latestVersion = VoiceHostVersion.GetLatestByEnvironment(CoreServiceConfiguration.Instance.Environment);
                    if (latestVersion == null)
                    {
                        Logger.Warn("Unable to determine latest version for environment '" + CoreServiceConfiguration.Instance.Environment + "'");
                        continue;
                    }

                    var payload = VoiceHostPayload.GetByVersionID(latestVersion.ID);
                    if (payload == null)
                    {
                        Logger.Warn("Unable to get payload for version '" + latestVersion.VersionString + "'");
                        continue;
                    }

                    int hostID;
                    while (_hostUpdateQueue.TryDequeue(out hostID))
                    {
                        var host = VoiceHost.GetByID(hostID);
                        if (host == null)
                        {
                            Logger.Warn("Failed process host update for unknown host #" + hostID);
                            continue;
                        }

                        Logger.Info("Pushing server update for version '" + latestVersion.VersionString + "' to voice host '" + host.IPAddress + "'");

                        try
                        {
                            if (HostUpdateService.DeployUpdate(null, host, latestVersion, payload))
                            {
                                Logger.Info("Success! Update succeeded for version '" + latestVersion.VersionString + "' to new voice host '" + host.IPAddress + "'");
                            }
                            else
                            {
                                Logger.Warn("Failure! Update failed for version '" + latestVersion.VersionString + "' to new voice host '" + host.IPAddress + "'");
                            }
                        }
                        catch (Exception ex)
                        {
                            Logger.Error(ex, "Failed to deploy queued update!");
                        }

                    }
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unhandled error in HostUpdateThread");
                }
            }
        }


        private static int GetVoiceRegionFromIP(IPAddress ipAddress)
        {

            try
            {

                // Do a reverse DNS lookup on the IP
                string reverseDnsHost = Dns.GetHostEntry(ipAddress).HostName;

                // Get everything past the first host segment
                reverseDnsHost = reverseDnsHost.Substring(reverseDnsHost.IndexOf(".") + 1);

                Logger.Info("Getting voice host provider region for IP '" + ipAddress + "' and Host '" + reverseDnsHost + "'");

                // Look for a provider record that matches
                var providerRegion = VoiceHostProviderRegion.GetByDomainName(reverseDnsHost);

                if (providerRegion != null)
                {
                    Logger.Info("Found voice host provider region for host", providerRegion);
                    return providerRegion.RegionID;
                }

                var ipSubnet = string.Join(".", ipAddress.ToString().Split('.').Take(3));

                Logger.Info("Unable to found region via reverse DNS. Looking for region via IP mask: " + ipSubnet);

                providerRegion = VoiceHostProviderRegion.GetByIPRange(ipSubnet);

                if (providerRegion != null)
                {
                    Logger.Info("Found voice host provider region for host", providerRegion);
                    return providerRegion.RegionID;
                }
            }
            catch (Exception ex)
            {
                Logger.Info(ex, "Failed to do reverse DNS lookup of IP: " + ipAddress + ". It will be geo located instead.");
            }


            // Geo locate the IP
            var range = IPRange.GetRange(ipAddress.ToInt64());

            Logger.Info("Registering a voice host from IP: " + ipAddress + ", Country: " + (range == null ? "Unknown" : range.CountryCode) + ", Region: " + (range == null ? "Unknown" : range.RegionCode));

            if (range != null)
            {
                return range.VoiceRegionID;
            }

            Logger.Info("Unable to identify voice host location, via reverse DNS or geo IP. The default region will be assigned instead!");

            return VoiceRegion.DefaultRegionID;
        }

        public ShutdownHostResponse ShutdownVoiceHost(string apiKey, string externalID)
        {
            var ipAddress = OperationContext.Current.GetClientIPAddress();

            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " + ipAddress);
                return new ShutdownHostResponse { Status = ShutdownHostStatus.Forbidden };
            }

            try
            {
                // Geo locate the IP                
                var externalIDGuid = Guid.Parse(externalID);
                var voiceHost = VoiceHost.GetByIPAddressOrExternalID(ipAddress, externalIDGuid);

                if (voiceHost == null)
                {
                    Logger.Warn("Attempt to shutdown non-existing voice host: " + externalID);
                    return new ShutdownHostResponse { Status = ShutdownHostStatus.Forbidden };
                }

                return voiceHost.Shutdown();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Voice host shutdown failed!");
                return new ShutdownHostResponse { Status = ShutdownHostStatus.Error };
            }
        }


        public bool UploadLogData(LogDataPayload payload)
        {
            if (payload.ApiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " + OperationContext.Current.GetClientIPAddress());
                return false;
            }

            try
            {


                if (payload.TypeID == 0 && (payload.Type == LogDataPayloadType.VoiceHost || payload.Type == LogDataPayloadType.Legacy))
                {
                    var externalIDGuid = Guid.Parse(payload.ExternalID);

                    // Try to identify this host
                    var voiceHost = VoiceHost.GetByExternalID(externalIDGuid);

                    if (voiceHost == null)
                    {
                        Logger.Warn("Unable to find voice host by external ID.",
                            new
                            {
                                payload.ExternalID,
                                payload.HostName,
                                IPAddress = OperationContext.Current.GetClientIPAddress()
                            });
                        return false;
                    }
                }

                if (!LoggingHelper.AddPayload(payload))
                {
                    return false;
                }

                return true;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to handle log data upload.");
                return false;
            }
        }


        /// <summary>
        /// Finds an existing (and active) voice session for this group, or provisions a new one.
        /// </summary>
        public CreateVoiceSessionResponse GroupVoiceSession(string apiKey, IPAddress ipAddress, Version clientVersion, int userID, string groupName, Guid groupID, bool forceNewSession)
        {
            Logger.Warn("Attempt to call deprecated method: GroupVoiceSession");
            return new CreateVoiceSessionResponse {Status = CreateVoiceSessionStatus.IncompatibleClient};
        }

        /// <summary>
        /// Finds an existing (and active) voice session for this friendship, or provisions a new one.
        /// </summary>
        /// <returns></returns>
        public CreateVoiceSessionResponse FriendVoiceSession(string apiKey, IPAddress ipAddress, Version clientVersion,
            int userID, string username, int friendID, bool forceNewSession)
        {
            Logger.Warn("Attempt to call deprecated method: FriendVoiceSession");
            return new CreateVoiceSessionResponse { Status = CreateVoiceSessionStatus.IncompatibleClient };
        }


        public ProvisionVoiceSessionResponse FriendVoiceSessionV2(string apiKey, IPAddress ipAddress, Version clientVersion, int userID, string username, int friendID, bool forceNewSession, PendingUserRequest pendingUser)
        {         
            var externalID = userID < friendID ? userID + ":" + friendID : friendID + ":" + userID;
            var result = CreateAccessRestrictedVoiceSessionV2(apiKey, ipAddress, clientVersion, userID, username, externalID, VoiceInstanceType.Friend, VoiceInstanceMode.Audio, forceNewSession);

            if (result.Status == CreateVoiceSessionStatus.Successful && result.VoiceInstance != null && pendingUser != null)
            {
                var pendingResult = AddPendingVoiceUsers(apiKey, ipAddress, result.VoiceInstance.CallID, userID, new[] { pendingUser });

                if (pendingResult.Status != BasicVoiceServiceStatus.Successful)
                {
                    Logger.Warn("Created a voice session, but was unable to add the pending user.", pendingResult);
                }
            }

            return result;
        }


        public ProvisionVoiceSessionResponse FriendVoiceSessionV3(string apiKey, IPAddress ipAddress, Version clientVersion, int userID, string username, int friendID, bool forceNewSession, PendingUserRequest pendingUser, VoiceInstanceMode mode)
        {            
            var externalID = userID < friendID ? userID + ":" + friendID : friendID + ":" + userID;
            var result = CreateAccessRestrictedVoiceSessionV2(apiKey, ipAddress, clientVersion, userID, username, externalID, VoiceInstanceType.Friend, mode, forceNewSession);

            if (result.Status == CreateVoiceSessionStatus.Successful && result.VoiceInstance != null && pendingUser != null)
            {
                var pendingResult = AddPendingVoiceUsers(apiKey, ipAddress, result.VoiceInstance.CallID, userID, new[] { pendingUser });

                if (pendingResult.Status != BasicVoiceServiceStatus.Successful)
                {
                    Logger.Warn("Created a voice session, but was unable to add the pending user.", new { userID, username, friendID, forceNewSession, mode, pendingResult });
                }
            }

            return result;
        }

        public ProvisionVoiceSessionResponse GroupVoiceSessionV2(string apiKey, IPAddress ipAddress, Version clientVersion, int userID, string groupName, Guid groupID, bool forceNewSession, int preferredRegion = 0)
        {            
            return CreateAccessRestrictedVoiceSessionV2(apiKey, ipAddress, clientVersion, userID, groupName, groupID.ToString(),
                VoiceInstanceType.Group, VoiceInstanceMode.Audio, forceNewSession, preferredRegion);
        }

        public ProvisionVoiceSessionResponse GroupVoiceSessionV3(string apiKey, IPAddress ipAddress, Version clientVersion, int userID, string groupName, Guid groupID, bool forceNewSession, int preferredRegion, VoiceInstanceMode mode)
        {            
            return CreateAccessRestrictedVoiceSessionV2(apiKey, ipAddress, clientVersion, userID, groupName, groupID.ToString(), VoiceInstanceType.Group, mode, forceNewSession, preferredRegion);
        }

        public ProvisionVoiceSessionResponse AdHocVoiceSessionV2(string apiKey, string userDisplayName, int? gameID, int userID, bool destroyExistingSession, Version clientVersion, IPAddress ipAddress = null)
        {

            try
            {                
                if (VoiceBanManager.IsBanned(userID))
                {
                    return new ProvisionVoiceSessionResponse
                    {
                        Status = CreateVoiceSessionStatus.Throttled
                    };
                }

                if (IsUserSpamming(userID))
                {
                    VoiceBanManager.BanForSpamming(userID);

                    return new ProvisionVoiceSessionResponse
                    {
                        Status = CreateVoiceSessionStatus.Throttled
                    };
                }

                VoiceInstance instance;
                ipAddress = ipAddress ?? OperationContext.Current.GetClientIPAddress();

                var status = DoCreateVoiceSession(false, userDisplayName, clientVersion, VoiceInstanceType.AdHoc, VoiceInstanceMode.Audio, gameID, userID, null, null, null, ipAddress, out instance);

                if (status == CreateVoiceSessionStatus.Successful)
                {
                    return new ProvisionVoiceSessionResponse
                    {
                        Status = CreateVoiceSessionStatus.Successful,
                        VoiceInstance = instance.ToContract()
                    };
                }

                return new ProvisionVoiceSessionResponse
                {
                    Status = status
                };

            }
            catch (Exception ex)
            {
                return new ProvisionVoiceSessionResponse
                {
                    Status = CreateVoiceSessionStatus.Error,
                    StatusMessage = ex.Message + ", Stack: " + ex.StackTrace
                };
            }

        }

        private ProvisionVoiceSessionResponse CreateAccessRestrictedVoiceSessionV2(string apiKey, IPAddress ipAddress, Version clientVersion,
            int userID, string displayName, string externalID, VoiceInstanceType type, VoiceInstanceMode mode, bool forceNewSession, int preferredRegion = 0)
        {

            try
            {

                Logger.Debug("CreateAccessRestrictedVoiceSessionV2", new { userID, displayName, externalID, type, mode, forceNewSession });

                if (!TestApiKey(apiKey, ipAddress))
                {
                    return new ProvisionVoiceSessionResponse { Status = CreateVoiceSessionStatus.Throttled };
                }

                VoiceInstance instance;
                             
                if (type == VoiceInstanceType.Group)
                {                    
                    instance = VoiceInstance.GetActiveByExternalIDAndType(externalID, type);                   
                }
                else
                {
                    instance = VoiceInstance.GetActiveByExternalIDAndTypeAndMode(externalID, type, mode);    
                }                

                var createNewSession = false;

                // If we cannot find one, try to provision one
                if (instance == null)
                {
                    createNewSession = true;
                    Logger.Debug("Creating a new voice session, as an active existing instance was not found.", new { externalID, type, mode });
                }
                else if (instance.VoiceHost == null)
                {
                    Logger.Warn("An active voice instance was found, but the voice host is null. A new one will be created.", new { externalID, type, mode });
                    instance.Status = VoiceInstanceStatus.Shutdown;
                    instance.Save();
                    createNewSession = true;
                }
                else if (!forceNewSession)
                {
                    // Ensure the instance is still hosted on the voice host
                    if (instance.VoiceHost.IsInstanceStale(instance.Code))
                    {
                        instance.Status = VoiceInstanceStatus.Shutdown;
                        instance.Save();
                        createNewSession = true;
                        Logger.Info("Found stale voice instance. A new one will be created.", new { type, mode });
                    }
                    else
                    {
                        Logger.Debug("Existing voice instance is still actively hosting. It will be used.", new { instance.Type, instance.Mode, instance.Code, instance.ExternalID });
                    }
                }
                else
                {
                    Logger.Debug("Forcefully creating a new voice session.");

                    // Get the session details from the host itself, to ensure things sync up in the DB
                    InstanceDetailsResponse instanceDetails;
                    if (!instance.VoiceHost.TryGetInstanceDetails(instance, out instanceDetails) || instanceDetails.Status != InstanceDetailsResponseStatus.Successful)
                    {
                        Logger.Info("A new voice instance is being created forcefully, due to disconnectivity.");

                        // Update this instance, to inactive
                        instance.Status = VoiceInstanceStatus.Shutdown;
                        instance.Save();
                        createNewSession = true;
                    }
                }

                if (createNewSession)
                {
                    var status = DoCreateVoiceSession(true, displayName, clientVersion, type, mode, null, userID, null, externalID, null, ipAddress, out instance, preferredRegion);

                    // Incompatible Client
                    if (status == CreateVoiceSessionStatus.IncompatibleClient)
                    {
                        return new ProvisionVoiceSessionResponse { Status = CreateVoiceSessionStatus.IncompatibleClient };
                    }

                    // Already exists (likely created by the user that just checked in)
                    if (status == CreateVoiceSessionStatus.AlreadyExists)
                    {
                        instance = VoiceInstance.GetActiveByExternalIDAndTypeAndMode(externalID, type, mode);

                        if (instance == null || instance.Status != VoiceInstanceStatus.Active)
                        {
                            Logger.Warn("Unable to create an access restricted voice session. After a status of 'AlreadyExists', an active voice session could not be retrieved from the database.");

                            return new ProvisionVoiceSessionResponse
                            {
                                Status = CreateVoiceSessionStatus.Error
                            };
                        }

                        status = CreateVoiceSessionStatus.Successful;
                    }

                    if (status != CreateVoiceSessionStatus.Successful)
                    {
                        return new ProvisionVoiceSessionResponse
                        {
                            Status = status
                        };
                    }
                }

                return new ProvisionVoiceSessionResponse
                {
                    Status = CreateVoiceSessionStatus.Successful,
                    VoiceInstance = instance.ToContract()
                };


            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to create an access restricted voice session.");
                return new ProvisionVoiceSessionResponse
                {
                    Status = CreateVoiceSessionStatus.Error,
                    StatusMessage = ex.Message + ", Stack: " + ex.StackTrace
                };
            }
        }

        public AddPendingVoiceUsersResponse AddPendingVoiceUsersV2(string apiKey, IPAddress ipAddress, string instanceGuid, int userID, PendingUserRequest[] pendingUsers)
        {
            if (!TestApiKey(apiKey, ipAddress))
            {
                return new AddPendingVoiceUsersResponse { Status = BasicVoiceServiceStatus.Forbidden };
            }

            try
            {
                var instance = VoiceInstance.GetByCode(instanceGuid);

                if (instance == null)
                {
                    Logger.Warn("User attempted to add a pending user to a non-existent voice session.");
                    return new AddPendingVoiceUsersResponse { Status = BasicVoiceServiceStatus.NotFound };
                }

                if (instance.Status != VoiceInstanceStatus.Active)
                {
                    Logger.Debug("User attempted to add a pending user to an inactive voice session.");
                    return new AddPendingVoiceUsersResponse { Status = BasicVoiceServiceStatus.NotFound };
                }

                var status = instance.VoiceHost.AddPendingUsers(instance, pendingUsers);

                if (status == AddPendingUsersStatus.Success)
                {
                    return new AddPendingVoiceUsersResponse { Status = BasicVoiceServiceStatus.Successful, VoiceDetails = instance.ToContract() };
                }

                switch (status)
                {
                    case AddPendingUsersStatus.Forbidden:
                        return new AddPendingVoiceUsersResponse { Status = BasicVoiceServiceStatus.Forbidden };
                    case AddPendingUsersStatus.NotConnected:
                        return new AddPendingVoiceUsersResponse { Status = BasicVoiceServiceStatus.Error };
                    case AddPendingUsersStatus.NotFound:
                        return new AddPendingVoiceUsersResponse { Status = BasicVoiceServiceStatus.NotFound };
                    default:
                        return new AddPendingVoiceUsersResponse { Status = BasicVoiceServiceStatus.Error };
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to add pending users to voice host.");
                return new AddPendingVoiceUsersResponse { Status = BasicVoiceServiceStatus.Error };
            }
        }
      

        public VoiceInstanceDetailsResponse GetVoiceInstanceDetails(string apiKey, string callID)
        {
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " +
                            OperationContext.Current.GetClientIPAddress());
                return null;
            }

            var voiceInstance = VoiceInstance.GetByCode(callID);
            if (voiceInstance == null)
            {
                return null;
            }

            return voiceInstance.ToContract();
        }

        public VoiceInstanceDetailsResponse FindVoiceSessionV2(string apiKey, string inviteCode)
        {
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " +
                            OperationContext.Current.GetClientIPAddress());
                return null;
            }

            try
            {
                var voiceInstance = VoiceInstance.GetByInviteCode(inviteCode);
                if (voiceInstance == null)
                {
                    return null;
                }

                return voiceInstance.ToContract();
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
                return null;
            }
        }

        public BasicVoiceServiceResponse UpdatePermissionsV2(string apiKey, Guid groupID, VoiceUserPermissions[] userPermissions)
        {
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("[UpdatePermissionsV2] Attempt to access API protected method with invalid API key");
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Forbidden };
            }

            try
            {
                var voiceInstance = VoiceInstance.GetActiveByExternalIDAndType(groupID.ToString(), VoiceInstanceType.Group);

                if (voiceInstance == null || voiceInstance.Status != VoiceInstanceStatus.Active)
                {
                    Logger.Warn("[UpdatePermissionsV2]  Attempt to update inactive voice session for group: " + groupID);
                    return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.NotFound };
                }

                if (!voiceInstance.VoiceHost.UpdateUserPermissions(voiceInstance, userPermissions))
                {
                    return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Failed };
                }

                Logger.Trace("[UpdatePermissionsV2]  Successfully voice permissions for host: " + groupID);
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Successful };
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Error };
            }
        }

        public BasicVoiceServiceResponse KickUserV2(string apiKey, string callID, int requestorUserID, int kickedUserID)
        {
            if (!TestApiKey(apiKey, IPAddress.None))
            {
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Forbidden };
            }

            try
            {
                var instance = VoiceInstance.GetByCode(callID);
                if (instance == null || instance.Status != VoiceInstanceStatus.Active)
                {
                    return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.NotFound };
                }

                if (!instance.VoiceHost.RemoveUser(instance, requestorUserID, kickedUserID, UserDisconnectReason.Kicked))
                {
                    return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Failed };
                }

                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Successful };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to remove users from group voice session.");
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Error };
            }
        }

        public BasicVoiceServiceResponse MuteUserV2(string apiKey, string callID, int requestorUserID, int mutedUserID, bool mute)
        {
            if (!TestApiKey(apiKey, IPAddress.None))
            {
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Forbidden };
            }

            try
            {
                var instance = VoiceInstance.GetByCode(callID);
                if (instance == null || instance.Status != VoiceInstanceStatus.Active)
                {
                    return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.NotFound };
                }

                if (!instance.VoiceHost.MuteUser(instance, requestorUserID, mutedUserID, mute))
                {
                    return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Failed };
                }

                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Successful };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to remove users from group voice session.");
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Error };
            }
        }

        public BasicVoiceServiceResponse DeafenUserV2(string apiKey, string callID, int requestorUserID, int deafenedUserID, bool deafen)
        {
            if (!TestApiKey(apiKey, IPAddress.None))
            {
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Forbidden };
            }

            try
            {
                var instance = VoiceInstance.GetByCode(callID);
                if (instance == null || instance.Status != VoiceInstanceStatus.Active)
                {
                    return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.NotFound };
                }

                if (!instance.VoiceHost.DeafenUser(instance, requestorUserID, deafenedUserID, deafen))
                {
                    return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Failed };
                }

                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Successful };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to remove users from group voice session.");
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Error };
            }
        }

        public ProvisionVoiceSessionResponse AutoMatchVoiceSessionV2(string userDisplayName, int? gameID, IPAddress ipAddress, Version clientVersion, Int64 autoMatchKey)
        {
            var response = new ProvisionVoiceSessionResponse { Status = CreateVoiceSessionStatus.Error };

            // Prevent invalid auto match keys
            if (autoMatchKey < 0)
            {
                return response;
            }

            try
            {
                // Try to find this instance (only if it is active)
                var instance = VoiceInstance.GetActiveByAutoMatchKey(autoMatchKey);

                if (instance == null)
                {
                    response.Status = DoCreateVoiceSession(false, userDisplayName, clientVersion, VoiceInstanceType.AutoMatch, VoiceInstanceMode.Audio, gameID, 0, autoMatchKey, null, null, ipAddress, out instance);

                    // Auto-match already exists (likely created by the user that just checked in)
                    if (response.Status == CreateVoiceSessionStatus.AlreadyExists)
                    {
                        instance = VoiceInstance.GetActiveByAutoMatchKey(autoMatchKey);
                    }
                }

                if (instance != null)
                {
                    response = new ProvisionVoiceSessionResponse
                    {
                        Status = CreateVoiceSessionStatus.Successful,
                        VoiceInstance = instance.ToContract()
                    };
                }

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "AutoMatchVoiceSession failed");
            }

            return response;
        }

        #endregion

        private bool TestApiKey(string apiKey, IPAddress ipAddress)
        {
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                Logger.Warn("Attempt was made to access an API restricted method from: " + ipAddress);
                return false;
            }
            return true;
        }

    }
}
