﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Data;
using System.Data.SqlClient;
using System.IO;
using System.Linq;
using System.Net;
using System.Text.RegularExpressions;
using System.Threading;
using System.Web;
using Curse.Logging;
using Curse.Voice.Helpers;
using ICSharpCode.SharpZipLib.Zip;
using System.Diagnostics;
using System.Threading.Tasks;

namespace Curse.Voice.Service.GeoCoding
{
    public class IPDatabaseUpdater
    {
        private static readonly Regex _csvSplit;        
#if DEBUG_GEO
        //private const string DownloadUrl = "http://localhost/GeoIp.zip";
        private const string DownloadUrl = "https://www.maxmind.com/app/geoip_download?edition_id=134&suffix=zip&license_key=2bPFVu3togLd";
#else
        private const string DownloadUrl = "https://www.maxmind.com/app/geoip_download?edition_id=134&suffix=zip&license_key=2bPFVu3togLd";
#endif
        private const string LocationFilename = "GeoIPCity-134-Location.csv";
        private const string BlocksFilename = "GeoIPCity-134-Blocks.csv";
        
        static IPDatabaseUpdater()
        {
            _csvSplit = new Regex(",(?!(?:[^\",]|[^\"],[^\"])+\")", RegexOptions.Compiled);            
        }
        public static void Initialize()
        {
#if DEBUG_GEO
            try
            {
                Task.Factory.StartNew(UpdateIPDatabase);
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
           
#else
    // Only run this on the 'Job' machine
            if ((System.Environment.MachineName.ToLower() != CoreServiceConfiguration.Instance.JobMachineName.ToLower()))
            {
                return;
            }

            new Thread(UpdateIPDatabaseThread) { IsBackground = true }.Start();
#endif

        }

        private static void UpdateIPDatabaseThread()
        {
            while (true)
            {
                try
                {
                    UpdateIPDatabase();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unable to update IP database");
                }

                Thread.Sleep(TimeSpan.FromHours(12));
            }
        }

        private static Dictionary<string, IPLocation> ProcessLocationFile(FileInfo fileInfo)
        {
            string fileContents = null;
            using (var reader = new StreamReader(fileInfo.FullName))
            {
                fileContents = reader.ReadToEnd();
            }

            // File format is: locid, country, region, city, postalcode, latitude, longitude, metroCode, areaCode
            var newLineRegex = new Regex("\n", RegexOptions.Compiled);

            var locationIndex = new Dictionary<string, IPLocation>();

            // Split the file contents by new line
           var lines = newLineRegex.Split(fileContents);

            for (var i = 2; i < lines.Length; i++)
            {
                var line = lines[i];
                var fields = _csvSplit.Split(line);

                if (fields.Length != 9)
                {
                    continue;
                }

                // Remove the quotes
                for (int j = 0; j < fields.Length; j++)
                {
                    fields[j] = fields[j].Replace("\"", string.Empty);
                }

                Int32 locationID = Int32.Parse(fields[0]);
                string countryCode = fields[1];
                string regionCode = fields[2];

                string key = IPLocation.GenerateKey(countryCode, regionCode);

                IPLocation ipLocation = null;
                if (!locationIndex.TryGetValue(key, out ipLocation))
                {
                    ipLocation = new IPLocation(countryCode, regionCode);
                    locationIndex.Add(ipLocation.Key, ipLocation);
                }
                ipLocation.AddLocationID(locationID);
            }

            var dt = GenerateIPLocationTable();
            dt.BeginLoadData();

            foreach (var kvp in locationIndex)
            {
                var row = dt.NewRow();
                row[1] = kvp.Value.Key;
                row[2] = kvp.Value.CountryCode;
                row[3] = kvp.Value.RegionCode;
                dt.Rows.Add(row); 
            }

            using (var conn = DatabaseHelper.Instance.GetConnection())
            {
                var txn = conn.BeginTransaction();

                var cmd = conn.CreateCommand();
                cmd.Transaction = txn;

                cmd.CommandText = "truncate table IPLocation";

                try
                {
                    cmd.ExecuteNonQuery();
                }
                catch (SqlException ex)
                {
                    Logger.Error(ex);
                    throw;
                }

                using (SqlBulkCopy bulk = new SqlBulkCopy(conn, SqlBulkCopyOptions.TableLock, txn))
                {
                    bulk.DestinationTableName = "IPLocation";

                    try
                    {
                        bulk.WriteToServer(dt);
                    }
                    catch (SqlException ex)
                    {
                        Debug.WriteLine(ex.Message);
                        throw;
                    }
                }

                txn.Commit();
            }

            return locationIndex;
        }

        private class IPLocation
        {
            private static HashSet<string> _countryCodesWithRegions = new HashSet<string>(StringComparer.InvariantCultureIgnoreCase) { "US", "CA" };

            public string Key
            {
                get;
                set;
            }

            public string CountryCode
            {
                get; set; 
            }

            public string RegionCode
            {
                get;
                set;
            }

            public HashSet<Int32> LocationIDs { get; private set; }

            public IPLocation(string countryCode, string regionCode)
            {               
                LocationIDs = new HashSet<int>();
                Key = GenerateKey(countryCode, regionCode);
                CountryCode = countryCode;
                if (_countryCodesWithRegions.Contains(countryCode))
                {
                    RegionCode = regionCode;
                }                
            }

            public void AddLocationID(int id)
            {
                LocationIDs.Add(id);
            }

            public static string GenerateKey(string countryCode, string regionCode)
            {
                string key = null;

                if (_countryCodesWithRegions.Contains(countryCode) && !string.IsNullOrEmpty(regionCode))
                {
                    key = countryCode + "-" + regionCode;
                }
                else
                {
                    key = countryCode;
                }

                if (key == null || key.Length > 6)
                {
                    throw new InvalidOperationException("Generated location key is invald: " + key);
                }

                return key;
            }
            
        }
        
        private static void ProcessBlocksFile(Dictionary<int, string> locationToKey, FileInfo fileInfo)
        {
            string fileContents = null;
            var dt = GenerateIPTable();
            dt.BeginLoadData();

            using (var reader = new StreamReader(fileInfo.FullName))
            {               

                string line;
                var counter = 0;
                while ((line = reader.ReadLine()) != null)
                {

                    if (++counter < 3)
                    {
                        continue;                        
                    }

                                         
                    var fields = _csvSplit.Split(line);

                    if (fields.Length != 3)
                    {
                        continue;
                    }

                    // Remove the quotes
                    for (int j = 0; j < fields.Length; j++)
                    {
                        fields[j] = fields[j].Replace("\"", string.Empty);
                    }

                    var startIp = Int64.Parse(fields[0]);
                    var endIp = Int64.Parse(fields[1]);
                    var locationID = Int32.Parse(fields[2]);
                    string locationKey = null;
                    // Look for this location in the index
                    if (!locationToKey.TryGetValue(locationID, out locationKey))
                    {
                        continue;
                    }

                    var row = dt.NewRow();
                    row[1] = locationKey;
                    row[2] = startIp;
                    row[3] = endIp;

                    dt.Rows.Add(row);
                    
                }
                
            }
            

            using (var conn = DatabaseHelper.Instance.GetConnection())
            {
                var txn = conn.BeginTransaction();

                var cmd = conn.CreateCommand();
                cmd.Transaction = txn;

                cmd.CommandText = "truncate table IPRange";

                try
                {
                    cmd.ExecuteNonQuery();
                }
                catch (SqlException ex)
                {            
                    Debug.WriteLine(ex.Message);
                    throw;
                }                

                using (SqlBulkCopy bulk = new SqlBulkCopy(conn, SqlBulkCopyOptions.TableLock, txn))
                {
                    bulk.DestinationTableName = "IPRange";

                    try
                    {
                        bulk.WriteToServer(dt);
                    }
                    catch (SqlException ex)
                    {
                        Debug.WriteLine(ex.Message);
                        throw;
                    }                   
                }

                txn.Commit();
            }
        }

        private static void UpdateIPDatabase()
        {
            Logger.Info("IP Database Update Started");

            using (var client = new WebClient())
            {
                client.Proxy = null;
                var tempZipFile = Path.Combine(Path.GetTempPath(), DateTime.UtcNow.Ticks + ".zip");
                client.DownloadFile(DownloadUrl, tempZipFile);


                var unzipPath = Path.Combine(Path.GetTempPath(), "IPDatabase", DateTime.UtcNow.Ticks.ToString());

                var unzipper = new FastZip();                
                unzipper.ExtractZip(tempZipFile, unzipPath, FastZip.Overwrite.Always, null, null, null, false);

                var di = new DirectoryInfo(unzipPath);
                if (!di.Exists)
                {
                    throw new Exception("GeoIP Database Failed to Extract at " + unzipPath);
                }

                var fi = di.GetFiles("GeoIPCity-134-Location.csv", SearchOption.AllDirectories).FirstOrDefault();

                if (fi == null)
                {
                    throw new Exception("GeoIP Database Failed to Extract GeoIPCity-134-Location.csv");
                }


                // Process the location file
                var locationIndex = ProcessLocationFile(fi);

                var locationIDToKey = new Dictionary<int, string>();
                // Build an index of location ID to Location Key
                foreach (var kvp in locationIndex)
                {
                    var ipLocation = kvp.Value;
                    foreach (var locationID in ipLocation.LocationIDs)
                    {
                        locationIDToKey[locationID] = ipLocation.Key;
                    }
                }

                var blocksFileInfo = di.GetFiles("GeoIPCity-134-Blocks.csv", SearchOption.AllDirectories).FirstOrDefault();

                if (blocksFileInfo == null)
                {
                    throw new Exception("GeoIP Database Failed to Extract GeoIPCity-134-Blocks.csv");
                }

                // Process the ip range file
                ProcessBlocksFile(locationIDToKey, blocksFileInfo);
            }
            
            
            Logger.Info("IP Database Update Finished");
        }        

        private static DataTable GenerateIPTable()
        {
            DataTable dt = new DataTable("IPRange");
            dt.Columns.Add("ID", typeof(Int32));
            dt.Columns.Add("LocationKeyCode", typeof(string));
            dt.Columns.Add("IPRangeStart", typeof(Int64));
            dt.Columns.Add("IPRangeEnd", typeof(Int64));
            return dt;
        }

        private static DataTable GenerateIPLocationTable()
        {
            DataTable dt = new DataTable("IPLocation");
            dt.Columns.Add("ID", typeof(Int32));
            dt.Columns.Add("KeyCode", typeof(string));
            dt.Columns.Add("CountryCode", typeof(string));
            dt.Columns.Add("RegionCode", typeof(string));
            return dt;
        }
    }
}