﻿using Curse.Logging;
using Curse.Voice.Helpers;
using Curse.Voice.Service.Models;
using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Diagnostics;
using System.Linq;
using System.Web;

namespace Curse.Voice.Service.GeoCoding
{
    public class IPRange
    {
        public Int64 Start { get; set; }
        public Int64 End { get; set; }        
        public Int32 VoiceRegionID { get; set; }

        public string CountryCode { get; set; }
        public string RegionCode { get; set; }
                
        private static Dictionary<Int64, IPRange> _values;
        private static Int64[] _binarySearchIndex;

        private static bool _isInitialized;

        public static void Initialize()
        {
            PopulateGeoData();
            _isInitialized = true;
        }

        public static void PopulateGeoData()
        {
            var startTime = DateTime.UtcNow;

            var countryIPRanges = new List<IPRange>();

            using (var conn = DatabaseHelper.Instance.GetConnection())
            {

                using (var command = conn.CreateCommand())
                {
                    command.CommandType = CommandType.StoredProcedure;
                    command.CommandText = "GetIPRanges";

                    using (SqlDataReader reader = command.ExecuteReader())
                    {

                        while (reader.Read())
                        {
                            countryIPRanges.Add(new IPRange(reader));
                        }
                    }

                    BuildRanges(countryIPRanges);
                }                
            }

            Logger.Info("IP Ranges Built in " + (DateTime.UtcNow - startTime).TotalSeconds.ToString("###,##0.00") + " seconds.");
        }
                
        public IPRange(SqlDataReader reader)
        {
            Start = (Int64)reader["IPRangeStart"];
            End = (Int64)reader["IPRangeEnd"];
            CountryCode = (string)reader["CountryCode"];
            
            if (reader["RegionCode"] != DBNull.Value)
            {
                RegionCode = (string)reader["RegionCode"];
            }

            if (reader["PreferredRegionID"] != DBNull.Value)
            {
                VoiceRegionID = (Int32)reader["PreferredRegionID"];
            }
            else if (reader["DefaultRegionID"] != DBNull.Value)
            {
                VoiceRegionID = (Int32)reader["DefaultRegionID"];
            }
            else
            {
                VoiceRegionID = VoiceRegion.DefaultRegionID;
            }

            
        }               

        private static void BuildRanges(IEnumerable<IPRange> ranges)
        {
            _values = new Dictionary<Int64, IPRange>();

            foreach (var item in ranges)
            {
                _values[item.Start] = item;
            }

            _binarySearchIndex = _values.Keys.ToArray();

            Array.Sort(_binarySearchIndex);
        }

        public static IPRange GetRange(Int64 value)
        {
            if (!_isInitialized)
            {
                return null;
            }

            int searchIndex = Array.BinarySearch(_binarySearchIndex, value);
            if (searchIndex < 0)
                searchIndex = ~searchIndex - 1;
            if (searchIndex < 0)
                return null;
            IPRange proposedRange = _values[_binarySearchIndex[searchIndex]];
            
            if (proposedRange.End >= value)
            {
                return proposedRange;
            }
            
            return null;            
        }
    }
}