﻿using System;
using System.Collections.Concurrent;
using System.Threading;
using Curse.Voice.Service.ServiceModels;
using Curse.Logging;
using System.Net.Http;
using Newtonsoft.Json;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Curse.Voice.Service.Helpers
{
    public static class LoggingHelper
    {
        private static LogCategory ThrottledLogger = new LogCategory("LoggingHelper") { Throttle = TimeSpan.FromMinutes(5) };
        private static readonly ConcurrentQueue<LogDataPayload> Queue = new ConcurrentQueue<LogDataPayload>();
        private static HttpClient client; 
        private const int MaxMessageLength = 1024;
      
        private static Dictionary<int, string> appTypeMapping = new Dictionary<int, string>
        {
            { 1 ,  "VoIP" },
            { 2 ,  "Voice"},
            { 3 ,  "Legacy Notifications"},
            { 4 ,  "Legacy Friends"},
            { 5 ,  "General Worker"},
            { 6 ,  "Group Worker"},
            { 7 ,  "Job Service"},
            { 8 ,  "Authentication"},
            { 9 ,  "Legacy Client"},
            { 10,  "Download Tracker"},
            { 11,  "Job Runner Service"},
            { 12,  "Addons"},
            { 13,  "Billing"},
            { 14,  "Achievements"},
            { 15,  "Images"},
            { 16,  "Avatars"},
            { 17,  "Files"},
            { 18,  "Sync"},
            { 19,  "Giveaways Worker"},
            { 20,  "Twitch Worker"},
            { 21,  "Polls Worker"},
            { 22,  "Groups"},
            { 23,  "Contacts"},
            { 24,  "Polls"},
            { 25,  "Giveaways"},
            { 26,  "Accounts"},
            { 27,  "Sessions"},
            { 28,  "Conversations"},
            { 29,  "Calls"},
            { 30,  "Logins"},
            { 31,  "Notifications"},
            { 32,  "Stats"},
            { 33,  "Bugs"},
            { 34,  "Reporting"},
            { 35,  "BattleNet Worker" },
            { 36,  "Twitch Interop Worker" },
            { 37,  "Twitch Interop Web" }
        };


        public static void Initialize()
        {
            client = new HttpClient();
            client.DefaultRequestHeaders.Add("ApiKey", CoreServiceConfiguration.Instance.LogServiceApiKey);
            new Thread(WorkerThread) { IsBackground = true }.Start();
        }

        public static bool AddPayload(LogDataPayload payload)
        {
            if (Queue.Count > 100)
            {
                return false;
            }

            Queue.Enqueue(payload);
            return true;
        }

        private static void WorkerThread()
        {
            while (true)
            {
                Thread.Sleep(1000);
                try
                {
                    LogDataPayload payload;
                    while (Queue.TryDequeue(out payload))
                    {
                        try
                        {
                            SaveToService(payload);
                        }
                        catch (Exception ex)
                        {
                            Logger.Error(ex, "Failed to save log data payload to service!", new { payload.Type, payload.TypeID, payload.HostName, payload.ExternalID });
                        }
                    }
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Logging helper worker thread failed!");
                    Thread.Sleep(TimeSpan.FromMinutes(1));
                }
            }
        }

        private static string GetApplicationName(int typeID)
        {
            string typeName = null;
            appTypeMapping.TryGetValue(typeID, out typeName);
            return typeName;
        }

        private static void SaveToService(LogDataPayload payload)
        {
            var jsonPayload = JsonConvert.SerializeObject(new
            {
                hostName = payload.HostName,
                applicationName = GetApplicationName(payload.TypeID),
                logEntries = payload.SerializedLogEntries,                
            });

            client.PostAsync(CoreServiceConfiguration.Instance.LogServiceUploadUri,
                new StringContent(jsonPayload, Encoding.UTF8, "application/json"))
                .ContinueWith(res =>
                {
                    var httpResponse = res.Result;
                    if (httpResponse == null || httpResponse.StatusCode != System.Net.HttpStatusCode.OK)
                    {
                        Logger.Warn("Unsuccessful response sending logs to logging service", new
                        {
                            statusCode = httpResponse.StatusCode,
                            content = httpResponse.Content.ReadAsStringAsync().Result
                        });
                    } 
                });
        }
    }
}