﻿using System.Data;
using System.Data.SqlClient;
using Curse.Voice.Service.Models;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Curse.Logging;

namespace Curse.Voice.Service.Helpers
{
    public static class ReportingHelper
    {
        private static readonly ConcurrentQueue<GameSessionMetric> MetricsQueue = new ConcurrentQueue<GameSessionMetric>();
        private static readonly LogCategory ThrottledLogger = new LogCategory("ReportingHelper") { Throttle = TimeSpan.FromMinutes(1) };
        
        public static void Initialize()
        {
            new Thread(MetricsWorkerThread) { IsBackground = true }.Start();         
        }

        public static void AddMetrics(GameSessionMetric[] metrics)
        {
            foreach (var metric in metrics)
            {
                MetricsQueue.Enqueue(metric);
            }
        }        

        private static void MetricsWorkerThread()
        {
            while (true)
            {
                Thread.Sleep(5000);
                try
                {
                    SaveGameMetrics();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to save game metrics!");
                    Thread.Sleep(TimeSpan.FromMinutes(1));
                }
            }
        }
      

        private static void SaveGameMetrics()
        {
            if (MetricsQueue.Count == 0)
            {
                return;
            }

            var minute = GetMinute();
            var endDate = new DateTime(DateTime.UtcNow.Year, DateTime.UtcNow.Month, DateTime.UtcNow.Day, DateTime.UtcNow.Hour, minute, 0, DateTimeKind.Utc);
            var startDate = endDate.AddMinutes(-15);

            var metrics = new List<GameSessionMetric>();
            GameSessionMetric metric;
            while (MetricsQueue.TryDequeue(out metric))
            {
                metrics.Add(metric);
            }

            // Raw, by user
            using (var conn = new SqlConnection(CoreServiceConfiguration.Instance.ReportingConnectionString))
            {
                conn.Open();
                
                foreach (var m in metrics)
                {
                    if (string.IsNullOrEmpty(m.GameStateName))
                    {
                        continue;                        
                    }

                    try
                    {
                        using (var command = conn.CreateCommand())
                        {
                            command.CommandText = "UpdateGameSessionsByExitCodeAndGameStateAndUserID";
                            command.CommandType = CommandType.StoredProcedure;
                            command.Parameters.AddWithValue("@GameID", m.GameID);
                            command.Parameters.AddWithValue("@ExitCode", m.ExitCode);
                            command.Parameters.AddWithValue("@GameState", m.GameState);
                            command.Parameters.AddWithValue("@GameStateName", m.GameStateName);
                            command.Parameters.AddWithValue("@UserID", m.UserID);
                            command.Parameters.AddWithValue("@StartDate", startDate);
                            command.Parameters.AddWithValue("@EndDate", endDate);
                            command.ExecuteNonQuery();
                        }
                    }
                    catch (Exception ex)
                    {
                        ThrottledLogger.Error(ex, "Failed to save game session metrics by User ID.");
                    }
                   
                }
            }

            var byGame = metrics.GroupBy(p => p.GameID).ToArray();

            // By Game
            foreach (var group in byGame)
            {
                var startedCount = group.Count();
                var finishedCount = group.Count(p => p.Successful);
                var crashedCount = group.Count(p => !p.Successful);

                using (var conn = new SqlConnection(CoreServiceConfiguration.Instance.ReportingConnectionString))
                {
                    conn.Open();
                    using (var command = conn.CreateCommand())
                    {
                        command.CommandText = "UpdateGameSessions";
                        command.CommandType = CommandType.StoredProcedure;
                        command.Parameters.AddWithValue("@GameID", group.Key);
                        command.Parameters.AddWithValue("@StartDate", startDate);
                        command.Parameters.AddWithValue("@EndDate", endDate);
                        command.Parameters.AddWithValue("@Started", startedCount);
                        command.Parameters.AddWithValue("@Finished", finishedCount);
                        command.Parameters.AddWithValue("@Crashed", crashedCount);
                        command.ExecuteNonQuery();
                    }
                }
            }

            // By Exit Code
            foreach (var gameGroup in byGame)
            {

                var byExitCode = gameGroup.GroupBy(p => p.ExitCode);

                foreach (var exitCodeGroup in byExitCode)
                {
                    var startedCount = exitCodeGroup.Count();
                    var finishedCount = exitCodeGroup.Count(p => p.Successful);
                    var crashedCount = exitCodeGroup.Count(p => !p.Successful);

                    using (var conn = new SqlConnection(CoreServiceConfiguration.Instance.ReportingConnectionString))
                    {
                        conn.Open();
                        using (var command = conn.CreateCommand())
                        {
                            command.CommandText = "UpdateGameSessionsByExitCode";
                            command.CommandType = CommandType.StoredProcedure;
                            command.Parameters.AddWithValue("@GameID", gameGroup.Key);
                            command.Parameters.AddWithValue("@ExitCode", exitCodeGroup.Key);
                            command.Parameters.AddWithValue("@StartDate", startDate);
                            command.Parameters.AddWithValue("@EndDate", endDate);
                            command.Parameters.AddWithValue("@Started", startedCount);
                            command.Parameters.AddWithValue("@Finished", finishedCount);
                            command.Parameters.AddWithValue("@Crashed", crashedCount);
                            command.ExecuteNonQuery();
                        }
                    }

                    var byGameState = gameGroup.GroupBy(p => p.GameState);
                    foreach (var gameStateGroup in byGameState)
                    {                        
                        startedCount = gameStateGroup.Count();
                        finishedCount = gameStateGroup.Count(p => p.Successful);
                        crashedCount = gameStateGroup.Count(p => !p.Successful);

                        using (var conn = new SqlConnection(CoreServiceConfiguration.Instance.ReportingConnectionString))
                        {
                            conn.Open();
                            using (var command = conn.CreateCommand())
                            {
                                command.CommandText = "UpdateGameSessionsByExitCodeAndGameStateAndGameState";
                                command.CommandType = CommandType.StoredProcedure;
                                command.Parameters.AddWithValue("@GameID", gameGroup.Key);
                                command.Parameters.AddWithValue("@ExitCode", exitCodeGroup.Key);
                                command.Parameters.AddWithValue("@GameState", gameStateGroup.Key);
                                command.Parameters.AddWithValue("@StartDate", startDate);
                                command.Parameters.AddWithValue("@EndDate", endDate);
                                command.Parameters.AddWithValue("@Started", startedCount);
                                command.Parameters.AddWithValue("@Finished", finishedCount);
                                command.Parameters.AddWithValue("@Crashed", crashedCount);
                                command.ExecuteNonQuery();
                            }
                        }
                    }

                }
            }


        }

        private static int GetMinute()
        {
            if (DateTime.UtcNow.Minute < 15)
            {
                return 0;
            }

            if (DateTime.UtcNow.Minute < 30)
            {
                return 15;
            }

            if (DateTime.UtcNow.Minute < 45)
            {
                return 30;
            }

            return 45;
        }
    }
}