﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.IO;
using System.Linq;
using System.Runtime.Serialization;
using System.ServiceModel;
using System.Text;
using System.Threading;
using System.Web.Services.Description;
using Curse.Logging;
using Curse.Voice.Helpers;
using Curse.Voice.HostManagement;
using Curse.Voice.Service.Models;
using Curse.Voice.Service.ServiceModels;
using Curse.Voice.UpdateManagement;
using Curse.Voice.Contracts;
using System.Runtime.Caching;
using System.Threading.Tasks;

namespace Curse.Voice.Service
{
    [ServiceBehavior(AddressFilterMode = AddressFilterMode.Any, ConcurrencyMode = ConcurrencyMode.Single, UseSynchronizationContext = false, InstanceContextMode = InstanceContextMode.PerSession)]
    public class HostUpdateService : IHostUpdateService
    {
        private Task _deploymentTask;

        #region Updater Methods

        public UpdatePayloadResult UploadUpdate(UpdatePayload payload)
        {
            if (payload.ApiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                throw new Exception("Invalid API Key");
            }

            VoiceHostUpdateJob job = null;

            try
            {
                job = ProcessUpload(payload);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process update!");

                return new UpdatePayloadResult() { Success = false, Message = "Error processing update. Check the service logs for details" };
            }

            if (job == null)
            {
                return new UpdatePayloadResult() { Success = true };
            }

            return new UpdatePayloadResult() { Success = true, JobID = job.ID };
        }

        public VoiceHostUpdateResult DeployUpdate(VoiceHostUpdateRequest request)
        {
            if (request.ApiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                throw new Exception("Invalid API Key");
            }

            try
            {
                Logger.Info("Received request to deploy existing voice host version!", request);

                var version = VoiceHostVersion.GetByID(request.ServerVersionID);
                if (version == null)
                {
                    return new VoiceHostUpdateResult { Success = false, Message = "Unable to find voice host version by id: " + request.ServerVersionID };
                }

                var job = CreateUpdateJob(version, request.Environment, request.DisableRollout, request.HostID, request.RegionID, request.HostModes, request.AbortOnFailure);
                return new VoiceHostUpdateResult { Success = true, JobID = job.ID };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to create deploy job!");
                return new VoiceHostUpdateResult { Success = false, Message = "Failed to create deploy job. View the logs for details." };
            }
        }

        private VoiceHostUpdateJob ProcessUpload(UpdatePayload payload)
        {
            var serverVersion = Version.Parse(payload.ServerVersionString);

            if (string.IsNullOrEmpty(payload.MinimumClientVersionString))
            {
                try
                {
                    Logger.Info("Determining most recent version for environment: " + payload.Environment);

                    var voiceHostVersion = VoiceHostVersion.GetLatestByEnvironment(payload.Environment);

                    if (voiceHostVersion != null)
                    {
                        Logger.Info("Found most recent version: " + voiceHostVersion.MinimumClientVersionString);
                        payload.MinimumClientVersionString = voiceHostVersion.MinimumClientVersionString;
                    }
                    else
                    {
                        Logger.Info("Unable to find a recent version!");
                        payload.MinimumClientVersionString = new Version(1, 0, 0, 0).ToString();
                    }
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to get latest version, during upload of update!");
                }

            }

            var minimumClientVersion = Version.Parse(payload.MinimumClientVersionString);

            var version = VoiceHostVersion.GetByVersion(serverVersion) ?? new VoiceHostVersion()
            {
                Version = serverVersion,
                MinimumClientVersion = minimumClientVersion,
                Environment = payload.Environment,
                Status = VoiceHostVersionStatus.Pending,
            };

            using (var conn = DatabaseHelper.Instance.GetConnection())
            {
                using (var transaction = conn.BeginTransaction())
                {
                    // Save the model to the database, but don't commit yet
                    version.SaveToDatabase(conn, transaction);

                    // Save the payload to the database
                    var voiceHostPayload = new VoiceHostPayload();
                    voiceHostPayload.VersionID = version.ID;

                    using (var memoryStream = new MemoryStream())
                    {
                        payload.ZipFile.CopyTo(memoryStream);
                        voiceHostPayload.Data = memoryStream.ToArray();
                    }

                    voiceHostPayload.SaveToDatabase(conn, transaction);

                    transaction.Commit();
                }

            }

            return CreateUpdateJob(version, payload.Environment, payload.DisableRollout, payload.HostID, payload.RegionID, payload.HostModes, payload.AbortOnFailure);
        }


        private VoiceHostUpdateJob CreateUpdateJob(VoiceHostVersion version, VoiceHostEnvironment environment, bool disableRollout, int hostID, int regionID, int hostModes, bool abortOnFailure)
        {
            Logger.Info("Creating voice host update job.", new { version, environment, disableRollout, hostID, regionID, hostModes, abortOnFailure });

            var hosts = VoiceHost.GetAllByEnvironment(environment).OrderBy(p => p.RegionID).ToArray();

            string message = null;
            if (disableRollout)
            {
                if (hostID > 0)
                {
                    message = "This deploy will only be deployed to a single voice host: " + hostID;
                    hosts = hosts.Where(p => p.ID == hostID).ToArray();
                }
                else if (regionID > 0)
                {
                    message = "This deploy will only be deployed to a single voice region.";
                    hosts = hosts.Where(p => p.RegionID == regionID && p.Status != VoiceHostStatus.Offline).ToArray();
                }

                if (hostModes != 0)
                {
                    hosts = hosts.Where(p => p.SupportedModes.HasFlag((VoiceInstanceMode)hostModes)).ToArray();
                }
            }

            // Get a list of all hosts, for this environment
            if (!hosts.Any())
            {
                Logger.Warn("This service update will not be applied to any hosts. There are none!");
                return null;
            }

            VoiceHostUpdateJob job = null;

            // Create a job, to roll this out
            using (var conn = DatabaseHelper.Instance.GetConnection())
            {
                using (var transaction = conn.BeginTransaction())
                {

                    job = new VoiceHostUpdateJob()
                    {
                        Status = VoiceHostUpdateJobStatus.Pending,
                        DateCreated = DateTime.UtcNow,
                        DateModified = DateTime.UtcNow,
                        TotalHosts = hosts.Length,
                        CurrentProgress = 0,
                        VersionID = version.ID

                    };

                    job.SaveToDatabase(conn, transaction);

                    foreach (var host in hosts)
                    {
                        var target = new VoiceHostUpdateTarget()
                        {
                            HostID = host.ID,
                            JobID = job.ID,
                            Status = VoiceHostUpdateTargetStatus.Pending
                        };
                        target.SaveToDatabase(conn, transaction);
                    }

                    transaction.Commit();
                }
            }

            if (!string.IsNullOrWhiteSpace(message))
            {
                UpdateDeploymentStatus(job, false, message);
            }

            // Kick off a thread to roll out the update

            _deploymentTask = Task.Factory.StartNew(() =>
            {
                try
                {
                    RolloutUpdate(job, abortOnFailure);

                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Server update failed to be rolled out!");
                    version.Status = VoiceHostVersionStatus.Failed;
                    version.SaveToDatabase();
                }
            });

            return job;
        }

        private static void UpdateDeploymentStatus(VoiceHostUpdateJob job, bool complete, params string[] messages)
        {
            if (job == null || messages == null || messages.Length == 0)
            {
                return;
            }

            try
            {
                string message = messages[0];
                if (messages.Length > 1)
                {
                    message = messages.Aggregate((agg, next) => agg + next + Environment.NewLine);
                }

                var memCache = MemoryCache.Default;
                var deploymentStatus = memCache.Get(job.ID.ToString()) as VoiceHostDeploymentStatus;
                if (deploymentStatus == null)
                {
                    deploymentStatus = new VoiceHostDeploymentStatus();
                    memCache.Set(job.ID.ToString(), deploymentStatus, new CacheItemPolicy
                    {
                        SlidingExpiration = TimeSpan.FromMinutes(10)
                    });
                }

                deploymentStatus.MessageLog += message;
                deploymentStatus.DeploymentComplete = complete;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Voice Deployment Status Update", job);
            }
        }

        private static void RolloutUpdate(VoiceHostUpdateJob job, bool abortOnFailure = false)
        {
            var startTimestamp = DateTime.UtcNow;
            var version = VoiceHostVersion.GetByID(job.VersionID);

            var targets = VoiceHostUpdateTarget.GetAllByJobID(job.ID);
            var payload = VoiceHostPayload.GetByVersionID(job.VersionID);

            if (payload == null)
            {
                job.ChangeStatus(VoiceHostUpdateJobStatus.Failed);
                UpdateDeploymentStatus(job, true, "No payload provided.");
                return;
            }

            job.ChangeStatus(VoiceHostUpdateJobStatus.Started);

            UpdateDeploymentStatus(job, false, "Update rollout has started. It will be deployed to " + targets.Length.ToString("###,##0") + " hosts.");
            var currentProgress = 0;

            // Ok, now to disperse this to all of our connected hosts
            foreach (var target in targets.OrderBy(p => p.ID).ToArray())
            {
                ++currentProgress;

                if (job.Status == VoiceHostUpdateJobStatus.Cancelled) // Our job was cancelled!
                {
                    UpdateDeploymentStatus(job, false, "Host update rollout of version '" + job.Version.Version + "' has been cancelled!");
                    break;
                }

                // // Get the voice host model from the database, and save it's status
                var host = VoiceHost.GetByID(target.HostID);
                if (host == null)
                {
                    target.Status = VoiceHostUpdateTargetStatus.Failed;
                    target.SaveToDatabase();
                    Logger.Warn("Host update rollout failed for host ID '" + target.HostID + "'. It could not be found in the database.");
                    continue;
                }


                UpdateDeploymentStatus(job, false, "Updating " + currentProgress.ToString("###,##0") + " of " + targets.Length.ToString("###,##0") + " " + host.HostName + " in " + host.Region.Name + "... ");
                host.ChangeStatus(VoiceHostStatus.Updating);
                host.ChangeUpdateStatus(VoiceHostUpdateStatus.UpdateInProgress);
                target.ChangeStatus(VoiceHostUpdateTargetStatus.TransferringPayload);
                job.IncrementProgress();

                // Establish a connection to this host's updater service

                try
                {
                    if (DeployUpdate(job, host, version, payload))
                    {
                        job.TrackSuccess();
                        UpdateDeploymentStatus(job, false, "Succeeded!");
                    }
                    else
                    {
                        throw new Exception("Failed to push update to host!");
                    }

                }
                catch (Exception ex)
                {
                    UpdateDeploymentStatus(job, false, "Failed!");
                    Logger.Error(ex, "Host update rollout failed for host ID '" + target.HostID + "', due to an unhandled exception.");
                    host.ChangeUpdateStatus(VoiceHostUpdateStatus.UpdateFailed);
                    target.ChangeStatus(VoiceHostUpdateTargetStatus.Failed);
                    job.TrackFailure();

                    if (abortOnFailure)
                    {
                        UpdateDeploymentStatus(job, true, "---------------------------------------------------------------------",
                            "Rollout has failed after " + (DateTime.UtcNow - startTimestamp).TotalSeconds.ToString("###,##0") + " second(s).",
                            "Successful: " + job.Successes.ToString("###,##0") + ". Failed: " + job.Failures.ToString("###,##0"));
                        job.ChangeStatus(VoiceHostUpdateJobStatus.Failed);
                        return;
                    }

                    continue;
                }


                target.ChangeStatus(VoiceHostUpdateTargetStatus.Successful);
                host.ChangeUpdateStatus(VoiceHostUpdateStatus.UpdateSucceeded);
                job = VoiceHostUpdateJob.GetByID(job.ID); // Refresh our job

                UpdateDeploymentStatus(job, false, "Waiting 5 seconds for failover...");
                Thread.Sleep(TimeSpan.FromSeconds(5));

            }

            job = VoiceHostUpdateJob.GetByID(job.ID); // Refresh our job

            // Update the version, to mark it as active
            version.Status = VoiceHostVersionStatus.Active;
            version.SaveToDatabase();

            UpdateDeploymentStatus(job, true, "---------------------------------------------------------------------",
                "Rollout has completed in " + (DateTime.UtcNow - startTimestamp).TotalSeconds.ToString("###,##0") + " second(s).",
                "Successful: " + job.Successes.ToString("###,##0") + ". Failed: " + job.Failures.ToString("###,##0"));
            job.ChangeStatus(VoiceHostUpdateJobStatus.Completed);
        }


        public static bool DeployUpdate(VoiceHostUpdateJob job, VoiceHost host, VoiceHostVersion version, VoiceHostPayload payload)
        {
            UpdateDeploymentStatus(job, false, "Starting host failover");

            try
            {
                host.Failover();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failover was unsuccessful. Continuing update.");
            }

            Logger.Info("Connecting to update service at " + host.IPAddress + ":" + CoreServiceConfiguration.Instance.HostPortNumber);

            UpdateDeploymentStatus(job, false, "Deploying server version '" + version.VersionString + "' to '" + host.IPAddress + "'");

            var result = false;
            using (var client = new VoiceUpdaterClient(host.IPAddress, CoreServiceConfiguration.Instance.HostPortNumber))
            {
                client.Open();
                result = client.PushUpdate(CoreServiceConfiguration.Instance.ApiKey, version.Version, payload.Data);
                client.Close();
            }

            return result;
        }

        public bool CancelUpdate(string apiKey, int jobID)
        {
            try
            {
                if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
                {
                    throw new Exception("Invalid API Key");
                }

                VoiceHostUpdateJob job = VoiceHostUpdateJob.GetByID(jobID);
                job.ChangeStatus(VoiceHostUpdateJobStatus.Cancelled);
                UpdateDeploymentStatus(job, true, "Job Cancelled");
                return true;
            }
            catch (Exception ex)
            {

                Logger.Error(ex, "Failed to cancel update!");
                return false;
            }
        }

        public UpdateRolloutResult GetUpdateStatus(string apiKey, int jobID)
        {
            try
            {
                if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
                {
                    throw new Exception("Invalid API Key");
                }

                VoiceHostUpdateJob job = VoiceHostUpdateJob.GetByID(jobID);

                if (job == null)
                {
                    return new UpdateRolloutResult() { Status = UpdateRolloutStatus.NotFound };
                }

                return new UpdateRolloutResult() { Status = UpdateRolloutStatus.Successful, Job = job };
            }
            catch (Exception ex)
            {

                Logger.Error(ex, "Failed to cancel update!");
                return new UpdateRolloutResult() { Status = UpdateRolloutStatus.Error };
            }
        }


        #endregion

        #region Client Updater Methods

        public bool RegisterClientVersion(string apiKey, VoiceHostEnvironment environment, string versionString)
        {
            throw new Exception("No longer supported from Central Voice Service");
        }

        #endregion

        public VoiceHostDeploymentStatus GetDeploymentStatus(string apiKey, int jobID)
        {
            if (apiKey != CoreServiceConfiguration.Instance.ApiKey)
            {
                throw new Exception("Invalid API Key");
            }

            VoiceHostDeploymentStatus deployStatus = null;
            try
            {
                var memCache = MemoryCache.Default;
                deployStatus = memCache.Get(jobID.ToString()) as VoiceHostDeploymentStatus;              
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error getting deployment status.", jobID);
            }
            return deployStatus;
        }
    }
}
