﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Globalization;
using System.Linq;
using System.Text;
using Curse.Hashing;
using Curse.ServiceAuthentication.AuthenticationService;
using Curse.Voice.Helpers;

namespace Curse.Voice.Service.Models
{
    public class VoiceHandshake
    {
                
        public bool HasMatches { get; set; }
        public int PlayerCount { get; set; }
        public VoiceHandshakeUser[] Users { get; set; }

        public VoiceHandshake()
        {
            
        }

        public VoiceHandshake(SqlDataReader reader)
        {
            HasMatches = (bool)reader["HasMatches"];
            PlayerCount = (int)reader["PlayerCount"];
        }

        public static VoiceHandshake CreateLegacy(long key)
        {
            using (var conn = DatabaseHelper.Instance.GetConnection())
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "CreateVoiceHandshake";
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.AddWithValue("@Key", key);

                    using (var reader = cmd.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return new VoiceHandshake(reader);
                        }
                        
                        return new VoiceHandshake() { HasMatches = false };
                    }                    
                }
            }
        }

        public static VoiceHandshake Create(long key, string display, int userID)
        {
            using (var conn = DatabaseHelper.Instance.GetConnection())
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "CreateVoiceHandshakeV2";
                    cmd.CommandType = CommandType.StoredProcedure;
                    cmd.Parameters.AddWithValue("@Key", key);
                    cmd.Parameters.AddWithValue("@DisplayName", display);
                    cmd.Parameters.AddWithValue("@UserID", userID);
                    
                    using (var reader = cmd.ExecuteReader())
                    {
                        if (!reader.HasRows)
                        {
                            return new VoiceHandshake() { HasMatches = false };    
                        }

                        var users = new List<VoiceHandshakeUser>();

                        while (reader.Read())
                        {
                            users.Add(new VoiceHandshakeUser(reader));
                        }

                        return new VoiceHandshake() { Users = users.ToArray(), HasMatches = users.Count > 1, PlayerCount = users.Count };
                    }
                }
            }
        }

        public static VoiceHandshake CheckLegacy(long key)
        {
            using (var conn = DatabaseHelper.Instance.GetConnection())
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "SELECT [HasMatches], [PlayerCount] FROM [VoiceHandshake] WHERE [Key] = @Key";                    
                    cmd.Parameters.AddWithValue("@Key", key);

                    using (var reader = cmd.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return new VoiceHandshake(reader);
                        }                        
                        return new VoiceHandshake() { HasMatches = false };
                    }
                }
            }
        }

        public static VoiceHandshake Check(long key)
        {
            using (var conn = DatabaseHelper.Instance.GetConnection())
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "SELECT [UserID], [DisplayName], [Timestamp], [Declined] FROM [VoiceHandshake] INNER JOIN [VoiceHandshakeUser] ON [VoiceHandshakeUser].[HandshakeKey] = [VoiceHandshake].[Key] WHERE [Key] = @Key";
                    cmd.Parameters.AddWithValue("@Key", key);

                    using (var reader = cmd.ExecuteReader())
                    {
                        if (!reader.HasRows)
                        {
                            return new VoiceHandshake() { HasMatches = false };
                        }

                        var users = new List<VoiceHandshakeUser>();

                        while (reader.Read())
                        {
                            users.Add(new VoiceHandshakeUser(reader));
                        }

                        return new VoiceHandshake() { Users = users.ToArray(), HasMatches = users.Count > 1, PlayerCount = users.Count };
                    }
                }
            }
        }

        public static void Delete(long key)
        {
            using (var conn = DatabaseHelper.Instance.GetConnection())
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "DELETE FROM [VoiceHandshake] WHERE [Key] = @Key";
                    cmd.Parameters.AddWithValue("@Key", key);
                    cmd.ExecuteNonQuery();
                }
            }
        }

        public static Int64 CalculateFingerprint(int? gameID, VoiceHandshakeUser[] users)
        {
            // Hash the game ID, and the numerically sorted user list:
            var sb = new StringBuilder();

            sb.Append(gameID.HasValue ? gameID.Value.ToString(CultureInfo.InvariantCulture) : "None");
            foreach (var user in users.OrderBy(p => p.UserID))
            {
                sb.Append(";");
                sb.Append(user.UserID.ToString(CultureInfo.InvariantCulture));
            }
            var sig = sb.ToString();            
            return MurmurHash2.ComputeHash(Encoding.UTF8.GetBytes(sig));
        }

        public static void UpdateFingerprint(Int64 key, Int64 fingerprint)
        {
            using (var conn = DatabaseHelper.Instance.GetConnection())
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "UPDATE [VoiceHandshake] SET [Fingerprint] = @Fingerprint WHERE [Key] = @Key";
                    cmd.Parameters.AddWithValue("@Fingerprint", fingerprint);
                    cmd.Parameters.AddWithValue("@Key", key);
                    cmd.ExecuteNonQuery();
                }
            }
        }

        public static void Decline(int userID, Int64 key)
        {
            using (var conn = DatabaseHelper.Instance.GetConnection())
            {
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "UPDATE [VoiceHandshakeUser] SET [Declined] = 1 WHERE [HandshakeKey] = @HandshakeKey and [UserID] = @UserID";
                    cmd.Parameters.AddWithValue("@HandshakeKey", key);
                    cmd.Parameters.AddWithValue("@UserID", userID);
                    cmd.ExecuteNonQuery();
                }
            }
        }
    }
}