﻿using System.Diagnostics;
using System.ServiceModel;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Configuration;
using System.Data.SqlClient;
using System.Linq;
using Curse.Logging;
using Curse.Voice.Helpers;
using Curse.Voice.HostManagement;
using Curse.Voice.HostManagement.Models;
using Curse.Voice.HostManagement.Responses;
using System.Net;
using System.Data.SqlTypes;
using Curse.Voice.Service.Responses;
using System.Threading;
using Curse.Voice.Contracts;
using System.Security.Cryptography;
using System.Text;

namespace Curse.Voice.Service.Models
{
    public enum VoiceHostStatus
    {
        Online = 1,
        Offline,
        ShuttingDown,
        Unhealthy,
        Updating
    }

    public enum VoiceHostUpdateStatus
    {
        None = 1,
        UpdateInProgress,
        UpdateFailed,
        UpdateSucceeded
    }

    public enum VoiceHostAvailability
    {
        Available,
        NotOnline,
        NotConnected,
        StaleStatistics,
        MaxedConnections,
        MaxedInboundBandwidth,
        MaxedOutboundBandwidth
    }

    public class VoiceHostProvisionResult
    {
        public bool IsSuccessful { get; set; }
        public string LogMessage { get; set; }
        public Exception Exception { get; set; }
        public object LogData { get; set; }
    }

    public class VoiceHost : IDisposable
    {
        private static readonly ConcurrentDictionary<int, VoiceHost> _hosts = new ConcurrentDictionary<int, VoiceHost>();
        private static DateTime? _lastHostUpdate;

        static VoiceHost()
        {

        }

        public static int HostCount
        {
            get { return _hosts.Count; }
        }

        public static void UpdateFromDatabase()
        {
            var startTime = DateTime.UtcNow;

            // Get all hosts for this region.
            var allHosts = GetAll(_lastHostUpdate);

            // Reconnect any hosts which have new IP addresses (happens anytime a VM reboots, on AWS)
            var modifiedHosts = _hosts.Values.Join(allHosts, d => d.ID, n => n.ID, (a, b) => new { CurrentHost = a, NewHost = b }).ToArray();

            foreach (var pair in modifiedHosts)
            {
                if (pair.NewHost.DateUpdated <= pair.CurrentHost.DateUpdated)
                {
                    continue; // Already up to date
                }

                if (pair.NewHost.Status == VoiceHostStatus.Offline)
                {
                    pair.CurrentHost.Statistics = new VoiceHostStatistics();
                }

                pair.CurrentHost.Status = pair.NewHost.Status;
                pair.CurrentHost.DateUpdated = pair.NewHost.DateUpdated;
                pair.CurrentHost.RegionID = pair.NewHost.RegionID;
                pair.CurrentHost.VersionID = pair.NewHost.VersionID;
                pair.CurrentHost.UpdateStatus = pair.NewHost.UpdateStatus;
                pair.CurrentHost.DateOnline = pair.NewHost.DateOnline;
                pair.CurrentHost.DateOffline = pair.NewHost.DateOffline;
                pair.CurrentHost.DateUnhealthy = pair.NewHost.DateUnhealthy;
                pair.CurrentHost.PublicHostName = pair.NewHost.PublicHostName;
                pair.CurrentHost.SupportedModes = pair.NewHost.SupportedModes;
                
                if (pair.CurrentHost.IPAddress != pair.NewHost.IPAddress)
                {
                    pair.CurrentHost.IPAddress = pair.NewHost.IPAddress;
                    pair.CurrentHost.Disconnect();
                }
            }

            // Add any new hosts, which are not yet in the list
            var addedHosts = allHosts.Where(p => !_hosts.ContainsKey(p.ID));
            foreach (var addedHost in addedHosts)
            {
                Logger.Debug("Discovered new host '" + addedHost.IPAddress + "'");
                _hosts.TryAdd(addedHost.ID, addedHost);
            }

            _lastHostUpdate = startTime.AddMinutes(-1);
        }


        public int ID { get; set; }
        public DateTime DateCreated { get; set; }
        public DateTime DateUpdated { get; set; }
        public string IPAddress { get; set; }
        public string HostName { get; set; }

        public string FullHostName
        {
            get
            {
                return PublicHostName + "." + CoreServiceConfiguration.Instance.HostDomainName;                
            }            
        }

        public string PublicHostName { get; set; }
        public VoiceHostStatus Status { get; set; }
        public int RegionID { get; set; }
        public VoiceHostEnvironment Environment { get; set; }
        public Guid ExternalID { get; set; }
        public DateTime DateOnline { get; set; }
        public DateTime DateOffline { get; set; }
        public int VersionID { get; set; }
        public VoiceHostUpdateStatus UpdateStatus { get; set; }
        public int ConnectionFailures { get; set; }
        public int? VoiceConfigurationID { get; set; }
        public int? VideoConfigurationID { get; set; }
        public DateTime? DateUnhealthy { get; set; }
        public VoiceInstanceMode SupportedModes { get; set; }

        public VoiceHostVersion Version
        {
            get { return VoiceHostVersion.GetByID(VersionID); }
        }

        public string VersionString
        {
            get
            {
                return Version != null ? Version.VersionString : "Unknown";
            }
            set
            {

            }
        }

        public string RegionName
        {
            get
            {
                return Region != null ? Region.Name : "Unknown";
            }
            set
            {

            }
        }

        public int DataRegionID
        {
            get
            {
                var region = Region;
                if (region == null)
                {
                    Logger.Warn("Voice host does not have a region: " + IPAddress, new { ID, HostName });
                    return 1;
                }

                return region.DataRegionID;
            }
        }

        public bool IsConnected { get; private set; }

        public VoiceHostStatistics Statistics { get; set; }

        public VoiceRegion Region
        {
            get { return VoiceRegion.GetByID(RegionID); }
        }

        public VoiceHostConfiguration Configuration
        {
            get { return (VoiceConfigurationID.HasValue ? VoiceHostConfiguration.GetByID(VoiceConfigurationID.Value) : VoiceHostConfiguration.DefaultVoice) ?? VoiceHostConfiguration.DefaultVoice; }
            set
            {
                if (value != null)
                {
                    VoiceConfigurationID = value.ID;
                }
                else
                {
                    VoiceConfigurationID = null;
                }
            }
        }

        public VoiceHostConfiguration VideoConfiguration
        {
            get { return (VideoConfigurationID.HasValue ? VoiceHostConfiguration.GetByID(VideoConfigurationID.Value) : VoiceHostConfiguration.DefaultVideo) ?? VoiceHostConfiguration.DefaultVideo; }
            set
            {
                if (value != null)
                {
                    VideoConfigurationID = value.ID;
                }
                else
                {
                    VideoConfigurationID = null;
                }
            }
        }

        public VoiceHost() { }

        private VoiceHost(SqlDataReader reader)
        {
            ID = reader.GetInt32(0);
            DateCreated = reader.GetDateTime(1);
            DateUpdated = reader.GetDateTime(2);
            HostName = reader.GetString(3);            
            IPAddress = reader.GetString(4);
            Status = (VoiceHostStatus)reader.GetByte(5);
            RegionID = reader.GetInt32(6);
            Environment = (VoiceHostEnvironment)reader.GetByte(7);
            ExternalID = reader.GetGuid(8);
            DateOnline = reader.GetDateTime(9);
            DateOffline = reader.GetDateTime(10);
            VersionID = reader.GetInt32(11);
            UpdateStatus = (VoiceHostUpdateStatus)reader.GetByte(12);

            if (reader[13] != DBNull.Value)
            {
                VoiceConfigurationID = reader.GetInt32(13);
            }

            if (reader[14] != DBNull.Value)
            {
                DateUnhealthy = reader.GetDateTime(14);
            }
            else
            {
                DateUnhealthy = null;
            }

            PublicHostName = reader.GetString(15);
            SupportedModes = (VoiceInstanceMode)reader.GetByte(16);

            if (reader[17] != DBNull.Value)
            {
                VoiceConfigurationID = reader.GetInt32(17);
            }

        }

        public void SaveToDatabase()
        {
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    if (ID > 0)
                    {
                        command.CommandText = "UPDATE [VoiceHost] SET HostName = @HostName, IPAddress = @IPAddress, Status = @Status, RegionID = @RegionID, Environment = @Environment, ExternalID = @ExternalID, DateUpdated = GETUTCDATE(), "
                                              + "DateOnline = @DateOnline, DateOffline = @DateOffline, VersionID = @VersionID, UpdateStatus = @UpdateStatus, ConfigurationID = @VoiceConfigurationID, VideoConfigurationID = @VideoConfigurationID,"
                                              + " DateUnhealthy = @DateUnhealthy, PublicHostName = @PublicHostName, SupportedModes = @SupportedModes WHERE ID = @ID";

                        command.Parameters.AddWithValue("@ID", ID);

                        command.Parameters.AddWithValue("@DateOnline",
                            DateOnline >= SqlDateTime.MinValue ? DateOnline : DateTime.UtcNow);

                        command.Parameters.AddWithValue("@DateOffline",
                            DateOffline >= SqlDateTime.MinValue ? DateOffline : DateTime.UtcNow);

                        if (DateUnhealthy.HasValue)
                        {
                            command.Parameters.AddWithValue("@DateUnhealthy", DateUnhealthy.Value);
                        }
                        else
                        {
                            command.Parameters.AddWithValue("@DateUnhealthy", DBNull.Value);
                        }                        
                    }
                    else
                    {
                        command.CommandText = "INSERT INTO [VoiceHost] (HostName, IPAddress, Status, RegionID, Environment, ExternalID, VersionID, ConfigurationID, VideoConfigurationID, PublicHostName, SupportedModes) VALUES(@HostName, @IPAddress, @Status, @RegionID, @Environment, @ExternalID, @VersionID, @VoiceConfigurationID, @VideoConfigurationID, @PublicHostName, @SupportedModes)";
                    }

                    command.Parameters.AddWithValue("@PublicHostName", PublicHostName);
                    command.Parameters.AddWithValue("@HostName", HostName);
                    command.Parameters.AddWithValue("@IPAddress", IPAddress);
                    command.Parameters.AddWithValue("@Status", (byte)Status);
                    command.Parameters.AddWithValue("@RegionID", RegionID);
                    command.Parameters.AddWithValue("@Environment", (byte)Environment);
                    command.Parameters.AddWithValue("@ExternalID", ExternalID);
                    command.Parameters.AddWithValue("@VersionID", VersionID);
                    command.Parameters.AddWithValue("@UpdateStatus", (byte)UpdateStatus);
                    
                    if (VoiceConfigurationID.HasValue)
                    {
                        command.Parameters.AddWithValue("@VoiceConfigurationID", VoiceConfigurationID);
                    }
                    else
                    {
                        command.Parameters.AddWithValue("@VoiceConfigurationID", DBNull.Value);
                    }

                    if (VideoConfigurationID.HasValue)
                    {
                        command.Parameters.AddWithValue("@VideoConfigurationID", VoiceConfigurationID);
                    }
                    else
                    {
                        command.Parameters.AddWithValue("@VideoConfigurationID", DBNull.Value);
                    }

                    command.Parameters.AddWithValue("@SupportedModes", (byte)SupportedModes);
                    
                    command.ExecuteNonQuery();
                }
            }
        }

        public void ChangeUpdateStatus(VoiceHostUpdateStatus status)
        {
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {

                    command.CommandText = "UPDATE [VoiceHost] SET UpdateStatus = @UpdateStatus, DateUpdated = GETUTCDATE() WHERE ID = @ID";
                    command.Parameters.AddWithValue("@ID", ID);
                    command.Parameters.AddWithValue("@UpdateStatus", (byte)status);
                    command.ExecuteNonQuery();
                }
            }

            UpdateStatus = status;
        }

        public void ChangeStatus(VoiceHostStatus status)
        {
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {

                    command.CommandText = "UPDATE [VoiceHost] SET Status = @Status, DateUpdated = GETUTCDATE() WHERE ID = @ID";
                    command.Parameters.AddWithValue("@ID", ID);
                    command.Parameters.AddWithValue("@Status", (byte)status);
                    command.ExecuteNonQuery();
                }
            }

            Status = status;
        }

        public void MarkInstancesOffline()
        {
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "UPDATE [VoiceInstance] SET Status = @Status, DateEnded = GETUTCDATE() WHERE Status <> @Status AND VoiceHostID = @VoiceHostID";
                    command.Parameters.AddWithValue("@Status", (byte)VoiceInstanceStatus.Shutdown);
                    command.Parameters.AddWithValue("@DateCreated", DateOnline);
                    command.Parameters.AddWithValue("@VoiceHostID", ID);
                    var rowsAffected = command.ExecuteNonQuery();
                    Logger.Info("Voice Host '" + HostName + "' has come online, and marked " + rowsAffected.ToString("###,##0") + " active voices instances as offline.");
                    VoiceInstance.ExpireAll();
                }
            }
        }

        private static VoiceHost[] GetAll(DateTime? since)
        {
            var voiceHosts = new List<VoiceHost>();
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [VoiceHost] WHERE Environment = @Environment";
                    command.Parameters.AddWithValue("@Environment", (int)CoreServiceConfiguration.Instance.Environment);
                    if (since.HasValue)
                    {
                        command.CommandText += " and DateUpdated >= @DateUpdated";
                        command.Parameters.AddWithValue("@DateUpdated", since);
                    }

#if CONFIG_DEBUG
                    command.CommandText += " and HostName = @HostName";
                    command.Parameters.AddWithValue("@HostName", System.Environment.MachineName);
#endif

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            voiceHosts.Add(new VoiceHost(reader));
                        }
                    }
                }
            }

            return voiceHosts.ToArray();
        }

        private static VoiceHost GetFromDatabaseByID(int id)
        {
            VoiceHost host = null;
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [VoiceHost] WHERE ID = @ID";
                    command.Parameters.AddWithValue("@ID", id);               

                    using (var reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            host = new VoiceHost(reader);
                        }
                    }
                }
            }

            if (host != null)
            {
                _hosts.AddOrUpdate(host.ID, host, (i, voiceHost) => host);
            }

            return host;
        }

        public static VoiceHost GetByExternalID(Guid externalID)
        {
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT ID FROM [VoiceHost] where ExternalID = @ExternalID";
                    command.Parameters.AddWithValue("@ExternalID", externalID);

                    using (var reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            int id = reader.GetInt32(0);
                            return GetByID(id);
                        }
                    }
                }
            }

            return null;
        }

        public static VoiceHost GetByIPAddressOrExternalID(IPAddress ipAddress, Guid externalID, bool bypassCache = false)
        {
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT ID FROM [VoiceHost] where ExternalID = @ExternalID or IPAddress = @IPAddress";
                    command.Parameters.AddWithValue("@ExternalID", externalID);
                    command.Parameters.AddWithValue("@IPAddress", ipAddress.ToString());

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            var id = reader.GetInt32(0);

                            if (id > 0)
                            {
                                return bypassCache ? GetFromDatabaseByID(id) : GetByID(id);    
                            }
                            
                        }
                    }
                }
            }

            return null;
        }

        public static VoiceHost[] GetAllByRegion(VoiceRegion voiceRegion)
        {
            var voiceHosts = new List<VoiceHost>();
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT ID FROM [VoiceHost] where RegionID = @RegionID";
                    command.Parameters.AddWithValue("@RegionID", voiceRegion.ID);

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            var id = reader.GetInt32(0);
                            var host = GetByID(id);
                            if (host != null)
                            {
                                voiceHosts.Add(host);
                            }
                        }
                    }
                }
            }

            return voiceHosts.ToArray();
        }

        public static VoiceHost[] GetAllByEnvironment(VoiceHostEnvironment environment)
        {
            var voiceHosts = new List<VoiceHost>();
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT ID FROM [VoiceHost] where Environment = @Environment";
                    command.Parameters.AddWithValue("@Environment", (byte)environment);

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            int id = reader.GetInt32(0);
                            var host = GetByID(id);
                            if (host != null)
                            {
                                voiceHosts.Add(host);
                            }
                        }
                    }
                }
            }

            return voiceHosts.ToArray();
        }

        public static VoiceHost GetByID(int id)
        {
            VoiceHost host;

            if (_hosts.TryGetValue(id, out host))
            {
                return host;
            }

            return null;
        }

        public void Disconnect()
        {
            IsConnected = false;
            if (_voiceManagementClient == null)
            {
                return;
            }
            var conn = _voiceManagementClient;
            _voiceManagementClient = null;

            try
            {
                conn.Close();
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to disconnect from: " + IPAddress);
            }
        }


        public void Dispose()
        {
            Disconnect();
        }


        #region Network

        private VoiceManagementClient _voiceManagementClient;

        private VoiceManagementClient CreateNewClient()
        {
            return new VoiceManagementClient(new InstanceContext(new VoiceHostCallbackClient(ID)), IPAddress, CoreServiceConfiguration.Instance.HostPortNumber);
        }

        public bool FailoverInstance(FailoverInstructions instructions)
        {
            if (!EnsureConnectivity())
            {
                return false;
            }

            try
            {
                _voiceManagementClient.FailoverInstance(CoreServiceConfiguration.Instance.ApiKey, instructions);
                return true;
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to failover instance.", instructions);
                return false;
            }            
        }

        public VoiceHostProvisionResult ProvisionInstance(VoiceInstance instance)
        {
            if (!EnsureConnectivity())
            {
                return new VoiceHostProvisionResult
                {
                    IsSuccessful = false,
                    LogMessage = "Failed to provision voice instance. Host is not connected to the management service.",
                    LogData = LogData
                };                
            }

            try
            {

                switch (instance.Type)
                {
                    
                    case VoiceInstanceType.Group:

                        if (Version.Version.Minor >= 3)
                        {
                            _voiceManagementClient.ProvisionGroupVoiceInstanceV2(CoreServiceConfiguration.Instance.ApiKey, instance.Code.ToString(), Guid.Parse(instance.ExternalID), instance.InviteCode, instance.Mode);
                        }
                        else
                        {
                            _voiceManagementClient.ProvisionGroupVoiceInstance(instance.Code.ToString(), Guid.Parse(instance.ExternalID), instance.InviteCode, CoreServiceConfiguration.Instance.ApiKey);    
                        }                        
                        
                        break;

                    default:

                        if (Version.Version.Minor >= 3)
                        {
                            _voiceManagementClient.ProvisionTypedVoiceInstanceV2(CoreServiceConfiguration.Instance.ApiKey, instance.Code.ToString(), instance.UserID, instance.Type, instance.Mode);                                
                        }
                        else
                        {
                            _voiceManagementClient.ProvisionTypedVoiceInstance(instance.Code.ToString(), instance.UserID, instance.Type, CoreServiceConfiguration.Instance.ApiKey);
                        }                        
                        
                        break;
                }

            }
            catch (Exception ex)
            {
                return new VoiceHostProvisionResult
                {
                    IsSuccessful = false,
                    Exception = ex,
                    LogMessage = "Failed to provision voice instance, due to an unhandled exception",
                    LogData = LogData
                }; 
            }

            return new VoiceHostProvisionResult
            {
                IsSuccessful = true,                
            }; 
        }

        public bool UnlockInstance(VoiceInstance instance)
        {
            if (!EnsureConnectivity())
            {
                Logger.Error("Failed to unlock voice instance. Host is not connected to the management service.", LogData);
                return false;
            }

            if (instance.Type != VoiceInstanceType.Friend)
            {
                // can only unlock friend voice instances
                return false;
            }

            try
            {
                var response = _voiceManagementClient.UnlockVoiceInstance(instance.Code.ToString(),
                    CoreServiceConfiguration.Instance.ApiKey);
                return response.Status == UnlockInstanceStatus.Successful;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to unlock instance.", LogData);
                return false;
            }
        }

        private static readonly Version MinimumRemoveUserVersion = new Version("1.2.0.0");

        public bool RemoveUsers(VoiceInstance instance, int requestorID, int[] usersToRemove, UserDisconnectReason reason)
        {
            if (!EnsureConnectivity())
            {
                Logger.Error("Failed to kick users from voice instance. Host is not conencted to the management service.", LogData);
                return false;
            }
            if (instance.Type != VoiceInstanceType.Group)
            {
                // can only kick from group voice sessions
                return false;
            }


            try
            {
                if (Version.Version < MinimumRemoveUserVersion) //Legacy API
                {
                    var resp = _voiceManagementClient .KickUsers(instance.Code.ToString(), requestorID, usersToRemove, CoreServiceConfiguration.Instance.ApiKey);
                    return resp.Status == KickUsersStatus.Successful;
                }
                else
                {
                    var resp = _voiceManagementClient.RemoveUsers(instance.Code.ToString(), requestorID, usersToRemove, reason,
                    CoreServiceConfiguration.Instance.ApiKey);
                    return resp.Status == RemoveUsersStatus.Successful;
                } 
                
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to kick users.", LogData);
                return false;
            }
        }

        private static readonly Version MinimumModerateVersion = new Version(1, 2, 0, 0);

        public bool RemoveUser(VoiceInstance instance, int requestorID, int userToRemove, UserDisconnectReason reason)
        {
            if (!EnsureConnectivity())
            {
                return false;
            }

            try
            {
                if (Version.Version < MinimumModerateVersion)
                {
                    var resp = _voiceManagementClient.KickUsers(instance.Code.ToString(), requestorID, new[] {userToRemove}, CoreServiceConfiguration.Instance.ApiKey);
                    return resp.Status == KickUsersStatus.Successful;
                }
                else
                {
                    var resp = _voiceManagementClient.RemoveUsers(instance.Code.ToString(), requestorID, new[] {userToRemove}, reason, CoreServiceConfiguration.Instance.ApiKey);
                    return resp.Status == RemoveUsersStatus.Successful;
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to remove user", LogData);
                return false;
            }
        }

        public bool UpdateUserPermissions(VoiceInstance voiceInstance, VoiceUserPermissions[] userPermissions)
        {
            if (!EnsureConnectivity())
            {
                Logger.Warn("Failed to update voice permissions for host. It is not connected.", new { voiceInstance.ExternalID });
                return false;
            }

            try
            {
                if (Version.Version < MinimumModerateVersion)
                {
                    Logger.Warn("Unable update voice permissions for host. It is an incompatible version.", new { voiceInstance.ExternalID });
                    return false;
                }

                var resp = _voiceManagementClient.UpdateUserPermissions(voiceInstance.Code.ToString(), userPermissions, CoreServiceConfiguration.Instance.ApiKey);

                if (resp.Status != UpdateUserPermissionsStatus.Successful)
                {
                    Logger.Warn("Unable update voice permissions for host: " + resp.Status, new { voiceInstance.ExternalID });
                }

                return resp.Status == UpdateUserPermissionsStatus.Successful;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to update user permissions", LogData);
                return false;
            }
        }

        public bool MuteUser(VoiceInstance instance, int requestorID, int userToMute, bool mute)
        {
            if (!EnsureConnectivity())
            {
                return false;
            }

            try
            {
                if (Version.Version < MinimumModerateVersion)
                {
                    return false;
                }

                var resp = _voiceManagementClient.MuteUser(instance.Code.ToString(), requestorID, userToMute, mute, CoreServiceConfiguration.Instance.ApiKey);
                return resp.Status == MuteUserStatus.Successful;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to mute user", LogData);
                return false;
            }
        }

        public bool DeafenUser(VoiceInstance instance, int requestorID, int userToDeafen, bool deafen)
        {
            if (!EnsureConnectivity())
            {
                return false;
            }

            try
            {
                if (Version.Version < MinimumModerateVersion)
                {
                    return false;
                }

                var resp = _voiceManagementClient.DeafenUser(instance.Code.ToString(), requestorID, userToDeafen, deafen, CoreServiceConfiguration.Instance.ApiKey);
                return resp.Status == DeafenUserStatus.Successful;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to deafen user", LogData);
                return false;
            }
        }


        public AddPendingUsersStatus AddPendingUsers(VoiceInstance instance, PendingUserRequest[] pendingUsers)
        {
            if (!EnsureConnectivity())
            {
                Logger.Error("Failed to add pending users to voice instance. Host is not connected to the management service.", LogData);
                return AddPendingUsersStatus.NotConnected;
            }

            try
            {
                var response = _voiceManagementClient.AddPendingUsers(instance.Code.ToString(), pendingUsers, CoreServiceConfiguration.Instance.ApiKey);
                if (response.Status == AddPendingUsersStatus.Error)
                {                    
                    Logger.Warn("Failed to add pending users to voice host.", new {response, instance.Code, instance.Status, instance.VoiceHostID, instance.Type});                    
                }

                return response.Status;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to add pending users due to an exception.", LogData);
                return AddPendingUsersStatus.Error;
            }
        }

        public RemovePendingUserStatus RemovePendingUser(VoiceInstance instance, int curseUserID)
        {
            if (!EnsureConnectivity())
            {
                Logger.Error("Failed to remove pending user from voice instance. Host is not connected to the management service.", LogData);
                return RemovePendingUserStatus.NotConnected;
            }

            try
            {
                var response = _voiceManagementClient.RemovePendingUser(instance.Code.ToString(), curseUserID, CoreServiceConfiguration.Instance.ApiKey);
                if (response.Status == RemovePendingUserStatus.Error)
                {                    
                    Logger.Warn("Failed to remove pending user from voice host.", new { response, instance.Code, instance.Status, instance.VoiceHostID, instance.Type });                                        
                }

                return response.Status;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to remove pending user due to an exception.", LogData);
                return RemovePendingUserStatus.Error;
            }
        }

        public bool DestroyInstance(VoiceInstance instance)
        {

            if (!EnsureConnectivity())
            {
                return false;
            }

            try
            {
                _voiceManagementClient.DestroyVoiceInstance(instance.Code.ToString(), CoreServiceConfiguration.Instance.ApiKey);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to destroy instance!", LogData);
                return false;
            }

            return true;

        }

        private object LogData
        {
            get
            {
                try
                {
                    return new { HostName, IPAddress, RegionName };
                }
                catch (Exception ex)
                {
                    return "Failed to get log data: " + ex.Message;
                }                
            }
        }


        public bool TryGetInstanceDetails(VoiceInstance instance, out InstanceDetailsResponse response)
        {

            if (!EnsureConnectivity())
            {
                response = null;
                return false;
            }

            try
            {
                response = _voiceManagementClient.GetInstanceDetails(CoreServiceConfiguration.Instance.ApiKey, instance.Code);
                return true;                
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to get instance details from host!", LogData);
                response = null;
                return false;
            }
        }

        public bool IsInstanceStale(Guid instanceCode)
        {
            if (!EnsureConnectivity())
            {                
                return false;
            }

            try
            {
                var response = _voiceManagementClient.GetInstanceDetails(CoreServiceConfiguration.Instance.ApiKey, instanceCode);
                return response != null && response.Status == InstanceDetailsResponseStatus.NotFound;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to get instance details from host!", LogData);                
                return false;
            }
        }

        private readonly object _connectionLock = new object();

        private bool EnsureConnectivity()
        {
            var lockAcquired = false;

            try
            {
                if (!Monitor.TryEnter(_connectionLock, 1000))
                {
                    Logger.Warn("Failed to acquire connection lock after 1 second: " + IPAddress + " (" + RegionName + ")");
                    return false;
                }

                lockAcquired = true;

                if (_voiceManagementClient == null)
                {
                    _voiceManagementClient = CreateNewClient();
                    _voiceManagementClient.Open();
                }
                else if (_voiceManagementClient.State != CommunicationState.Opened)
                {
                    if (_voiceManagementClient.State == CommunicationState.Faulted)
                    {
                        _voiceManagementClient.Abort();
                    }
                    else if (_voiceManagementClient.State != CommunicationState.Closed)
                    {
                        try
                        {
                            _voiceManagementClient.Close();
                        }
                        catch
                        {
                            _voiceManagementClient.Abort();
                        }
                    }

                    _voiceManagementClient = CreateNewClient();
                    _voiceManagementClient.Open();
                }

                return true;
            }
            catch (CommunicationException ex)
            {
                Logger.Warn(ex, "Communication error while trying to establish a management service connection to: " + IPAddress + " (" + RegionName + ")");
                return false;

            }
            catch (TimeoutException ex)
            {
                Logger.Warn(ex, "Timeout when establishing a management service connection to: " + IPAddress + " (" + RegionName + ")");
                return false;
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Unhandled error when tyring to establish a management service connection to: " + IPAddress + " (" + RegionName + ")", ex);
                return false;
            }
            finally
            {
                if (lockAcquired)
                {
                    Monitor.Exit(_connectionLock);
                }
            }

        }

        #endregion


        private int MaximumConnections
        {
            get
            {
                if (SupportedModes == VoiceInstanceMode.Audio)
                {
                    return Configuration.MaximumConnections;
                }

                if (SupportedModes == VoiceInstanceMode.Video)
                {
                    return VideoConfiguration.MaximumConnections;                    
                }

                return Math.Min(Configuration.MaximumConnections, VideoConfiguration.MaximumConnections);
            }
        }

        private int MaximumOutboundBandwidth
        {
            get
            {
                if (SupportedModes == VoiceInstanceMode.Audio)
                {
                    return Configuration.MaximumOutboundBbps;
                }

                if (SupportedModes == VoiceInstanceMode.Video)
                {
                    return VideoConfiguration.MaximumOutboundBbps;
                }

                return Math.Min(Configuration.MaximumOutboundBbps, VideoConfiguration.MaximumOutboundBbps);
            }
        }

        private int MaximumInboundBandwidth
        {
            get
            {
                if (SupportedModes == VoiceInstanceMode.Audio)
                {
                    return Configuration.MaximumInboundBbps;
                }

                if (SupportedModes == VoiceInstanceMode.Video)
                {
                    return VideoConfiguration.MaximumInboundBbps;
                }

                return Math.Min(Configuration.MaximumInboundBbps, VideoConfiguration.MaximumInboundBbps);
            }
        }

        public VoiceHostAvailability Availability
        {

            get
            {
                if (Status != VoiceHostStatus.Online)
                {
                    return VoiceHostAvailability.NotOnline;
                }

                if (!IsConnected)
                {
                    return VoiceHostAvailability.NotConnected;
                }

                if (Statistics == null || Statistics.LastUpdated < DateTime.UtcNow.AddMinutes(-5))
                {
                    return VoiceHostAvailability.StaleStatistics;
                }

                if (Statistics.ActiveConnections >= MaximumConnections)
                {
                    return VoiceHostAvailability.MaxedConnections;
                }

                if (Statistics.SentBitsPerSecond >= MaximumOutboundBandwidth)
                {
                    return VoiceHostAvailability.MaxedOutboundBandwidth;
                }

                if (Statistics.ReceivedBitsPerSecond >= MaximumInboundBandwidth)
                {
                    return VoiceHostAvailability.MaxedInboundBandwidth;
                }

                return VoiceHostAvailability.Available;
            }
        }

        private DateTime _lastStatsAttempt = DateTime.MinValue;

        public void UpdateStatistics()
        {
            // Only try unhealthy hosts every minute
            if (Status == VoiceHostStatus.Unhealthy && DateTime.UtcNow.Subtract(_lastStatsAttempt) < TimeSpan.FromMinutes(1))
            {
                return;
            }

            // Skip any hosts in the middle of an update or offline
            if (Status == VoiceHostStatus.Offline)
            {
                return;
            }

            _lastStatsAttempt = DateTime.UtcNow;

            if (!EnsureConnectivity())
            {
                IsConnected = false;
                TrackConnectionFailure();
                return;
            }

            IsConnected = true;

            try
            {
                var stats = _voiceManagementClient.GetStatistics(CoreServiceConfiguration.Instance.ApiKey);
                if (stats != null)
                {
                    Statistics = Statistics ?? new VoiceHostStatistics();
                    Statistics.ActiveConnections = stats.ActiveConnections;

                    Statistics.ActiveSessions = stats.ActiveSessions;
                    Statistics.MultiUserSessions = stats.MultiUserSessions;
                    Statistics.TotalSessions = stats.TotalSessions;

                    Statistics.ReceivedBitsPerSecond = stats.ReceivedBitsPerSecond;
                    Statistics.SentBitsPerSecond = stats.SentBitsPerSecond;
                    Statistics.TotalBitsPerSecond = stats.ReceivedBitsPerSecond + stats.SentBitsPerSecond;
                    Statistics.ProcessorUtilization = stats.ProcessorUtilization;
                    Statistics.LastUpdated = DateTime.UtcNow;
                    TrackConnectionSuccess();
                }
                else
                {
                    Logger.Warn("Failed to get statistics for host '" + IPAddress + "'");
                    TrackConnectionFailure();
                }
            }
            catch (CommunicationException ex)
            {
                Logger.Warn(ex, "Communication error when trying to get statistics from a voice host: " + IPAddress + " (" + RegionName + ")");

                IsConnected = false;
                TrackConnectionFailure();
                try
                {
                    var client = _voiceManagementClient;
                    if (client != null)
                    {
                        client.Abort();
                        _voiceManagementClient = null;    
                    }                    
                }
                catch (Exception ex2)
                {
                    Logger.Warn(ex2, "Unable to destroy voice management client!");
                }

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled error when trying to get statistics from a voice host: " + IPAddress + " (" + RegionName + ")");
                TrackConnectionFailure();
            }
        }

        public InstanceStatisticsResponse GetInstanceStatistics()
        {
            if (!IsConnected)
            {
                Logger.Warn("Unable to get instance statistics for host '" + IPAddress + "'. The management client is not connected.");
                return null;
            }

            return _voiceManagementClient.GetInstanceStatistics(CoreServiceConfiguration.Instance.ApiKey);
        }

        public ConnectionDiagnosticsResponse GetConnectionDiagnostics()
        {
            if (!IsConnected)
            {
                Logger.Warn("Unable to get connection diagnostics for host '" + IPAddress + "'. The management client is not connected.");
                return null;
            }

            return _voiceManagementClient.GetConnectionDiagnostics(CoreServiceConfiguration.Instance.ApiKey);
        }

        private void TrackConnectionSuccess()
        {
            if (Status == VoiceHostStatus.Unhealthy) // If we were considered unhealth before, consider ourselves healthy now
            {
                Logger.Info("Status of '" + HostName + "' (" + IPAddress + ") is once again healthy.");
                Status = VoiceHostStatus.Online;
                SaveToDatabase();
            }

            ConnectionFailures = 0;
        }

        private void TrackConnectionFailure()
        {
            if (Status == VoiceHostStatus.Online)
            {
                Logger.Warn("Setting status of '" + HostName + "' (" + IPAddress + ") to unhealthy, due to management service connection failures.");
                DateUnhealthy = DateTime.UtcNow;
                Status = VoiceHostStatus.Unhealthy;
                SaveToDatabase();
            }

            ++ConnectionFailures;
        }

        public int ActiveSessions
        {
            get
            {
                var stats = Statistics;
                return stats != null ? stats.ActiveSessions : 0;
            }
        }

        #region Failover

        public FailoverInstructions[] GetFailoverInstructions(VoiceHostStatus status)
        {
            var startTime = DateTime.UtcNow;

            // Mark the host as updating or shutting down
            ChangeStatus(status);

            // See if the region has any available hosts
            if (!Region.GetAvailableHosts("Failover").Any() && !Region.FailoverRegion.GetAvailableHosts("Failover").Any())
            {
                Logger.Warn("Planned failover has failed for host '" + HostName + "', as it has no available failover targets!");
                return null;
            }

            // Get all instances on the host, which are active, and provision them on other hosts
            var activeInstances = VoiceInstance.GetAllActiveByHost(this);

            // This is to deal with hosts that are pwnd
            if (Statistics != null && Statistics.ActiveConnections < activeInstances.Length)
            {
                Logger.Warn("Failover will exclude old instances, due to likelihood of failure.");
                activeInstances = activeInstances.Where(p => DateTime.UtcNow.Subtract(p.DateCreated) < TimeSpan.FromHours(3)).ToArray();
            }

            Logger.Info("Failing over " + activeInstances.Count().ToString("###,##0") + " instances for host '" + HostName + "'");

            var currentRegion = Region;

            var instructions = new List<FailoverInstructions>();

            foreach (var instance in activeInstances)
            {
                try
                {
                    // Find the best failover target for this instance
                    var sw = Stopwatch.StartNew();
                    var newHost = VoiceHostManager.Instance.FindBestFailoverHost(currentRegion, instance.Mode);

                    // If one cannot be found, continue
                    if (newHost == null)
                    {
                        Logger.Warn("Unable to find suitable failover host for instance '" + instance.Code + "' on host '" + HostName + "'");
                        instructions.Add(new FailoverInstructions { Status = FailoverStatus.Failed });
                        continue;
                    }

                    Logger.Info("Found failover host in " + sw.ElapsedMilliseconds + " ms");
                    sw.Restart();

                    // Otherwise, provision the instance on the target
                    var provisionResult = newHost.ProvisionInstance(instance);
                    if (!provisionResult.IsSuccessful)
                    {
                        Logger.Warn(provisionResult.Exception, string.Format("Unable to provision instance '{0}' on failover target target host '{1}'", instance.Code, newHost.HostName));
                        instructions.Add(new FailoverInstructions { Status = FailoverStatus.Failed });
                        continue;
                    }


                    Logger.Info("Provisioned failover instance in " + sw.ElapsedMilliseconds + " ms");
                    sw.Restart();

                    // Save thew new host info
                    instance.VoiceHostID = newHost.ID;
                    instance.Save();

                    Logger.Info("Saved failover instance in " + sw.ElapsedMilliseconds + " ms");

                    // We've provisioned the instance at the new host!
                    instructions.Add(new FailoverInstructions
                    {
                        Status = FailoverStatus.Successful,
                        HostName = newHost.FullHostName,
                        IPAddress = newHost.IPAddress,
                        InstanceCode = instance.Code.ToString(),
                        Port = CoreServiceConfiguration.Instance.VoicePortNumber,
                        Reason = FailoverReason.HostUpdating
                    });
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Instance failover generated an exception!", LogData);
                    instructions.Add(new FailoverInstructions { Status = FailoverStatus.Error });
                }
            }

            Logger.Info("Failover completed for host '" + HostName + "' in " + (DateTime.UtcNow - startTime).TotalSeconds.ToString("####,##0.00") + " seconds.");

            return instructions.ToArray();

        }


        public void PerformFailover(VoiceHostStatus status)
        {
            var startTime = DateTime.UtcNow;
            // Mark the host as updating or shutting down
            ChangeStatus(status);

            // See if the region has any available hosts
            if (!Region.GetAvailableHosts("Failover").Any() && !Region.FailoverRegion.GetAvailableHosts("Failover").Any())
            {
                Logger.Warn("Planned failover has failed for host '" + HostName + "', as it has no available failover targets!");
                return;
            }

            // Get all instances on the host, which are active, and provision them on other hosts
            var activeInstances = VoiceInstance.GetAllActiveByHost(this);

            Logger.Info("Failing over " + activeInstances.Count().ToString("###,##0") + " instances for host '" + HostName + "'");

            var currentRegion = Region;

            var managementClient = CreateNewClient();
            managementClient.Open();

            var counter = 0;

            foreach (var instance in activeInstances)
            {
                try
                {
                    var desc = "[" + (++counter + " of " + activeInstances.Length) + "] ";

                    Logger.Info(desc + "Starting failover...");

                    // Find the best failover target for this instance                   
                    var newHost = VoiceHostManager.Instance.FindBestFailoverHost(currentRegion, instance.Mode);

                    // If one cannot be found, continue
                    if (newHost == null)
                    {
                        Logger.Warn(desc + "Unable to find suitable failover host for instance '" + instance.Code + "' on host '" + HostName + "'");
                        continue;
                    }

                    // Otherwise, provision the instance on the target
                    var provisionReuslt = newHost.ProvisionInstance(instance);
                    if (!provisionReuslt.IsSuccessful)
                    {
                        Logger.Warn(provisionReuslt.Exception, desc + string.Format("Unable to find provision instance '{0}' on failover target target host '{1}'", instance.Code, newHost.HostName));
                        continue;
                    }

                    var sw = Stopwatch.StartNew();
                    Logger.Info(desc + "Provisioned failover instance in " + sw.ElapsedMilliseconds + " ms");

                    // Save thew new host info
                    instance.VoiceHostID = newHost.ID;
                    instance.Save();

                    // We've provisioned the instance at the new host!
                    var instructions = new FailoverInstructions
                    {
                        Status = FailoverStatus.Successful,
                        HostName = newHost.FullHostName,
                        IPAddress = newHost.IPAddress,
                        InstanceCode = instance.Code.ToString(),
                        Port = CoreServiceConfiguration.Instance.VoicePortNumber,
                        Reason = FailoverReason.HostUpdating
                    };

                    sw.Restart();
                    var resp = managementClient.FailoverInstance(CoreServiceConfiguration.Instance.ApiKey, instructions);
                    Logger.Info(desc + "Completed instance failover in " + sw.ElapsedMilliseconds + " ms. Status: " + resp.Status);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Instance failover generated an exception!", LogData);

                    if (managementClient.State != CommunicationState.Opened)
                    {
                        Logger.Info("Attempting to reopen management client...");
                        try
                        {
                            managementClient.Abort();
                        }
                        catch { }

                        try
                        {
                            managementClient = CreateNewClient();
                            managementClient.Open();
                        }
                        catch (Exception ex2)
                        {
                            Logger.Error(ex2, "Failed to reopen management client!");
                            throw;
                        }
                    }
                }
            }

            Logger.Info("Failover completed for host '" + HostName + "' in " + (DateTime.UtcNow - startTime).TotalSeconds.ToString("####,##0.00") + " seconds.");

        }

        private static readonly Version NewFailoverVersion = new Version("1.1.0.0");

        public FailoverServerResponse Failover(VoiceHostStatus status = VoiceHostStatus.Updating)
        {
            if (Version.Version >= NewFailoverVersion)
            {
                Logger.Info("Beginning new failover for host '" + HostName + "'");
                try
                {
                    PerformFailover(status);
                    return new FailoverServerResponse { Status = FailoverServerStatus.Successful };
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failover error!", LogData);
                    return new FailoverServerResponse { Status = FailoverServerStatus.Error, StatusMessage = ex.Message };
                }

            }
            else
            {
                Logger.Info("Beginning legacy failover for host '" + HostName + "'");
                var instructions = GetFailoverInstructions(VoiceHostStatus.Updating);
                if (instructions == null || !instructions.Any())
                {
                    return new FailoverServerResponse { Status = FailoverServerStatus.Successful };
                }
                var managementClient = CreateNewClient();
                managementClient.Open();
                return managementClient.Failover(CoreServiceConfiguration.Instance.ApiKey, instructions);
            }

        }


        public ShutdownHostResponse Shutdown()
        {

            var failoverInstructions = GetFailoverInstructions(VoiceHostStatus.ShuttingDown);

            Status = VoiceHostStatus.Offline;
            DateOffline = DateTime.UtcNow;
            Statistics = new VoiceHostStatistics();
            SaveToDatabase();

            return new ShutdownHostResponse { Status = ShutdownHostStatus.Successful, FailoverInstructions = failoverInstructions };
        }

        #endregion


        public static string GeneratePublicHostName(string machineName)
        {
#if CONFIG_DEBUG || CONFIG_STAGING
            return machineName.Split('.')[0];
#endif

            byte[] hash;
            using (var md5 = MD5.Create())
            {                
                hash = md5.ComputeHash(Encoding.UTF8.GetBytes(machineName));
            }

            var sb = new StringBuilder();            
            foreach (byte t in hash)
            {
                sb.Append(t.ToString("x2"));
            }

            var hostName = sb.ToString();

            return hostName;
        }

        public VoiceHostConfigurationContract GetConfigurationContract()
        {            
            var voiceConfiguration = Configuration;
            var videoConfiguration = VideoConfiguration;

            return new VoiceHostConfigurationContract
            {
                MaxUsersPerVoiceSession = voiceConfiguration.MaxUsersPerSession,
                MaxUsersPerVideoSession = videoConfiguration.MaxUsersPerSession,
            };            
        }
    }
}