﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Linq;
using System.Web;
using Curse.Logging;
using Curse.Voice.Helpers;

namespace Curse.Voice.Service.Models
{
    public enum VoiceHostVersionStatus
    {
        Pending = 1,
        Active,
        RolledBack,
        Failed
    }

    public class VoiceHostVersion
    {
        private static ConcurrentDictionary<int, VoiceHostVersion> _versionsByID;
        private static ConcurrentDictionary<Version, VoiceHostVersion> _versionsByVersion;


        public int ID { get; set; }

        private string _versionString;

        public string VersionString
        {
            get
            {
                return _versionString;
            }
            set
            {
                _versionString = value;
                _version = Version.Parse(value);
            }
        }

        private Version _version;

        public Version Version
        {
            get { return _version; }
            set
            {
                _version = value;
                _versionString = value.ToString();
            }
        }


        private string _minimumClientVersionString;

        public string MinimumClientVersionString
        {
            get { return _minimumClientVersionString; }
            set
            {
                _minimumClientVersionString = value;
                MinimumClientVersion = Version.Parse(value);
            }
        }

        private Version _minimumClientVersion;
        public Version MinimumClientVersion
        {
            get { return _minimumClientVersion; }
            set { 
                _minimumClientVersion = value;
                _minimumClientVersionString = value.ToString();
            }
        }

        public DateTime DateCreated { get; set; }
        public DateTime DateModified { get; set; }
        public VoiceHostEnvironment Environment { get; set; }
        public VoiceHostVersionStatus Status { get; set; }

        public VoiceHostVersion()
        {
        }

        public VoiceHostVersion(SqlDataReader reader)
        {
            ID = reader.GetInt32(0);
            VersionString = reader.GetString(1);
            Version = Version.Parse(VersionString);
            MinimumClientVersionString = reader.GetString(2);
            MinimumClientVersion = Version.Parse(MinimumClientVersionString);
            DateCreated = reader.GetDateTime(3);
            DateModified = reader.GetDateTime(4);
            Environment = (VoiceHostEnvironment)reader.GetByte(5);
            Status = (VoiceHostVersionStatus)reader.GetByte(6);
        }

        public void SaveToDatabase(SqlConnection conn, SqlTransaction transaction = null)
        {
            using (SqlCommand command = conn.CreateCommand())
            {
                command.Transaction = transaction;

                command.Parameters.AddWithValue("@MinimumClientVersion", MinimumClientVersionString);
                command.Parameters.AddWithValue("@DateModified", DateTime.UtcNow);
                command.Parameters.AddWithValue("@Version", VersionString);
                command.Parameters.AddWithValue("@Environment", (byte)Environment);
                command.Parameters.AddWithValue("@Status", (byte)Status);

                if (ID > 0)
                {
                    command.CommandText = "UPDATE [VoiceHostVersion] SET MinimumClientVersion = @MinimumClientVersion, Status = @Status, DateModified = @DateModified WHERE ID = @ID";
                    command.Parameters.AddWithValue("@ID", ID);
                    command.ExecuteNonQuery();
                }
                else
                {
                    command.Parameters.AddWithValue("@DateCreated", DateTime.UtcNow);
                    command.CommandText = "INSERT INTO [VoiceHostVersion] (Version, MinimumClientVersion, DateCreated, DateModified, Environment, Status) OUTPUT INSERTED.ID VALUES(@Version, @MinimumClientVersion, @DateCreated, @DateModified, @Environment, @Status)";
                    ID = (int)command.ExecuteScalar();

                    _versionsByID.TryAdd(ID, this);
                    _versionsByVersion.TryAdd(Version, this);
                }
            }
        }

        public void SaveToDatabase()
        {
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                SaveToDatabase(connection);
            }
            
        }

        static VoiceHostVersion()
        {

            try
            {
                _versionsByID = new ConcurrentDictionary<int, VoiceHostVersion>(GetAll(null).ToDictionary(p => p.ID));
                _versionsByVersion = new ConcurrentDictionary<Version, VoiceHostVersion>(_versionsByID.Values.ToDictionary(p => p.Version));
            }
            catch (Exception ex)
            {
                _versionsByID = new ConcurrentDictionary<int, VoiceHostVersion>();
                Logger.Error(ex, "Failed to retrieve any versions from the database!");
            }

        }

        private static DateTime? _lastHostUpdate = null;

        public static void UpdateFromDatabase()
        {
            DateTime startTime = DateTime.UtcNow;

            // Get all versions
            var modifiedVersions = GetAll(_lastHostUpdate);

            foreach (var version in modifiedVersions)
            {
                VoiceHostVersion currentVersion = version;
                _versionsByID.AddOrUpdate(version.ID, i => currentVersion, (i, hostVersion) => currentVersion);
                _versionsByVersion.AddOrUpdate(version.Version, i => currentVersion, (i, hostVersion) => currentVersion);

            }

            _lastHostUpdate = startTime.AddMinutes(-1);
        }



        public static VoiceHostVersion GetLatestByEnvironment(VoiceHostEnvironment environment)
        {            
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT TOP 1 * FROM [VoiceHostVersion] WHERE [Environment] = @Environment and [Status] = @Status ORDER BY [DateCreated] desc";
                    command.Parameters.AddWithValue("@Environment", (byte)environment);
                    command.Parameters.AddWithValue("@Status", (byte)VoiceHostVersionStatus.Active);

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return new VoiceHostVersion(reader);
                        }
                    }
                }
            }

            return null;
        }

        private static VoiceHostVersion[] GetAll(DateTime? since)
        {
            List<VoiceHostVersion> versions = new List<VoiceHostVersion>();
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [VoiceHostVersion]";
                    if (since.HasValue)
                    {
                        command.CommandText += " WHERE DateModified >= @DateModified";
                        command.Parameters.AddWithValue("@DateModified", since);
                    }

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            versions.Add(new VoiceHostVersion(reader));
                        }
                    }
                }
            }

            return versions.ToArray();
        }

        public static VoiceHostVersion GetByID(int id)
        {
            VoiceHostVersion model = null;

            if (_versionsByID.TryGetValue(id, out model))
            {
                return model;
            }
            else
            {
                return null;
            }
        }

        public static VoiceHostVersion GetByVersion(Version version)
        {
            VoiceHostVersion model = null;

            if (_versionsByVersion.TryGetValue(version, out model))
            {
                return model;
            }
            else
            {
                return null;
            }
        }
    }
}