﻿using System.Data;
using Curse.Caching;
using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using Curse.Logging;
using Curse.Voice.Contracts;
using Curse.Voice.Helpers;
using Curse.Voice.Service.Responses;
using Curse.Voice.HostManagement.Models;

namespace Curse.Voice.Service.Models
{

    public enum VoiceInstanceStatus
    {
        Active = 1,
        Inactive = 2,
        Shutdown = 3,
    }



    public enum VoiceInstanceSaveStatus
    {
        Success,
        Error,
        DuplicateInviteCode,
        DuplicateAutoMatchKey,
        DuplicateExternalID
    }

    public class VoiceInstance
    {
        public int ID { get; set; }
        public VoiceInstanceStatus Status { get; set; }
        public int UserID { get; set; }
        public Guid Code { get; set; }
        public DateTime DateCreated { get; set; }
        public int VoiceHostID { get; set; }
        public int? GameID { get; set; }
        public DateTime? DateEnded { get; set; }
        public string ExternalID { get; set; }
        public VoiceInstanceType Type { get; set; }
        public VoiceInstanceMode Mode { get; set; }

        public string InviteCode
        {
            get;
            set;
        }

        public string InviteDisplayName
        {
            get;
            set;
        }

        public Int64? AutoMatchKey
        {
            get;
            set;
        }

        public VoiceHost VoiceHost
        {
            get { return VoiceHost.GetByID(VoiceHostID); }
        }

        public string InviteUrl
        {
            get
            {
                return string.Format(CoreServiceConfiguration.Instance.InviteUrlFormat, InviteCode);
            }
        }

        public VoiceInstance() { }

        public VoiceInstance(SqlDataReader reader)
        {
            ID = reader.GetInt32(0);
            Status = (VoiceInstanceStatus)reader.GetByte(1);
            UserID = reader.GetInt32(2);
            Code = reader.GetGuid(3);
            DateCreated = reader.GetDateTime(4);
            VoiceHostID = reader.GetInt32(5);

            if (reader[6] != DBNull.Value)
            {
                GameID = reader.GetInt32(6);
            }

            if (reader[7] != DBNull.Value)
            {
                DateEnded = reader.GetDateTime(7);
            }

            if (reader[8] != DBNull.Value)
            {
                InviteCode = reader.GetString(8);
            }

            if (reader[9] != DBNull.Value)
            {
                InviteDisplayName = reader.GetString(9);
            }

            if (reader[10] != DBNull.Value)
            {
                AutoMatchKey = reader.GetInt64(10);
            }

            if (reader[11] != DBNull.Value)
            {
                ExternalID = reader.GetString(11);
            }

            Type = (VoiceInstanceType)reader.GetByte(12);
            Mode = (VoiceInstanceMode)reader.GetByte(14);
        }

        public static VoiceInstance GetFromDatabaseByID(int id)
        {
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [VoiceInstance] WHERE ID = @ID";
                    command.Parameters.AddWithValue("@ID", id);

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return new VoiceInstance(reader);
                        }

                        return null;
                    }
                }
            }
        }

        public static int? GetIDFromDatabaseByUserID(int userID)
        {
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT ID FROM [VoiceInstance] WHERE [Status] = 1 AND [UserID] = @UserID";
                    command.Parameters.AddWithValue("@UserID", userID);

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return reader.GetInt32(0);
                        }

                        return null;
                    }
                }
            }
        }

        public static int? GetIDFromDatabaseByCode(string code)
        {
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT ID FROM [VoiceInstance] WHERE [Code] = @Code";
                    command.Parameters.AddWithValue("@Code", code);

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return reader.GetInt32(0);
                        }

                        return null;
                    }
                }
            }
        }

        public static int? GetActiveIDFromDatabaseByAutoMatchKey(Int64 autoMatchKey)
        {
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT ID FROM [VoiceInstance] WHERE [AutoMatchKey] = @AutoMatchKey AND [Status] = @Status";
                    command.Parameters.AddWithValue("@AutoMatchKey", autoMatchKey);
                    command.Parameters.AddWithValue("@Status", (int)VoiceInstanceStatus.Active);


                    using (var reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return reader.GetInt32(0);
                        }

                        return null;
                    }
                }
            }
        }

        public static int? GetIDFromDatabaseByAutoMatchFingerprint(Int64 fingerprint)
        {
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandType = CommandType.StoredProcedure;
                    command.CommandText = "FindInstanceByFingerprint";
                    command.Parameters.AddWithValue("@Fingerprint", fingerprint);

                    using (var reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            var result = reader[0];
                            if (result != DBNull.Value)
                            {
                                return (int)result;
                            }
                        }

                        return null;
                    }
                }
            }
        }

        public static int? GetIDFromDatabaseByInviteCode(string inviteCode)
        {
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT ID FROM [VoiceInstance] WHERE [InviteCode] = @InviteCode";
                    command.Parameters.AddWithValue("@InviteCode", inviteCode);

                    using (var reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return reader.GetInt32(0);
                        }

                        return null;
                    }
                }
            }
        }

        public static int? GetActiveIDFromDatabaseByExternalID(string groupID, VoiceInstanceType type, VoiceInstanceMode mode)
        {
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT ID FROM [VoiceInstance] WHERE [ExternalID] = @ExternalID AND [Status] = @Status AND [Type] = @Type AND [Mode] = @Mode";
                    command.Parameters.AddWithValue("@ExternalID", groupID);
                    command.Parameters.AddWithValue("@Status", (int)VoiceInstanceStatus.Active);
                    command.Parameters.AddWithValue("@Type", (byte)type);
                    command.Parameters.AddWithValue("@Mode", (byte)mode);

                    using (var reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return reader.GetInt32(0);
                        }

                        return null;
                    }
                }
            }
        }

        public static int? GetActiveIDFromDatabaseByExternalID(string groupID, VoiceInstanceType type)
        {
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT ID FROM [VoiceInstance] WHERE [ExternalID] = @ExternalID AND [Status] = @Status AND [Type] = @Type";
                    command.Parameters.AddWithValue("@ExternalID", groupID);
                    command.Parameters.AddWithValue("@Status", (int)VoiceInstanceStatus.Active);
                    command.Parameters.AddWithValue("@Type", (byte)type);
                    
                    using (var reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return reader.GetInt32(0);
                        }

                        return null;
                    }
                }
            }
        }

        public VoiceInstanceSaveStatus Save()
        {
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                return Save(connection);
            }
        }

        /// <summary>
        /// Change the type of this instance to ad hoc, and remove the external ID
        /// </summary>
        public void Unlock()
        {
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.Parameters.AddWithValue("@Type", (byte)VoiceInstanceType.MultiFriend);
                    command.Parameters.AddWithValue("@ID", ID);
                    command.CommandText = "UPDATE [VoiceInstance] SET Type = @Type, ExternalID = null WHERE ID = @ID";
                    command.ExecuteNonQuery();
                }
            }
        }

        public const int InviteDisplayNameMaxLength = 128;

        public VoiceInstanceSaveStatus Save(SqlConnection connection, SqlTransaction transaction = null)
        {
            using (var command = connection.CreateCommand())
            {
                if (transaction != null)
                {
                    command.Transaction = transaction;
                }


                if (InviteDisplayName != null && InviteDisplayName.Length > InviteDisplayNameMaxLength)
                {
                    InviteDisplayName = InviteDisplayName.Substring(0, InviteDisplayNameMaxLength);
                }

                command.Parameters.AddWithValue("@Code", Code);
                command.Parameters.AddWithValue("@Status", (byte)Status);
                command.Parameters.AddWithValue("@UserID", UserID);
                command.Parameters.AddWithValue("@DateCreated", DateCreated);
                command.Parameters.AddWithValue("@VoiceHostID", VoiceHostID);
                command.Parameters.AddWithValue("@GameID", GameID.HasValue ? (object)GameID.Value : 0);
                command.Parameters.AddWithValue("@DateEnded", DateEnded.HasValue ? (object)DateEnded.Value : DBNull.Value);
                command.Parameters.AddWithValue("@InviteCode", InviteCode != null ? (object)InviteCode : DBNull.Value);
                command.Parameters.AddWithValue("@InviteDisplayName", InviteDisplayName != null ? (object)InviteDisplayName : DBNull.Value);
                command.Parameters.AddWithValue("@AutoMatchKey", AutoMatchKey.HasValue ? (object)AutoMatchKey.Value : DBNull.Value);
                command.Parameters.AddWithValue("@ExternalID", ExternalID != null ? (object)ExternalID : DBNull.Value);
                command.Parameters.AddWithValue("@Type", (byte)Type);
                command.Parameters.AddWithValue("@Mode", (byte)Mode);

                if (ID > 0)
                {
                    command.Parameters.AddWithValue("@ID", ID);
                    command.CommandText = "UPDATE [VoiceInstance] SET Status = @Status, UserID = @UserID, DateCreated = @DateCreated, VoiceHostID = @VoiceHostID, GameID = @GameID, DateEnded = @DateEnded, InviteCode = @InviteCode,  InviteDisplayName = @InviteDisplayName, AutoMatchKey = @AutoMatchKey, Type = @Type, Mode = @Mode where ID = @ID";
                    command.ExecuteNonQuery();
                }
                else
                {
                    command.CommandText = "INSERT INTO [VoiceInstance] (Code, Status, UserID, DateCreated, VoiceHostID, GameID, InviteCode, InviteDisplayName, AutoMatchKey, ExternalID, Type, Mode) OUTPUT INSERTED.[ID] VALUES(@Code, @Status, @UserID, @DateCreated, @VoiceHostID, @GameID, @InviteCode, @InviteDisplayName, @AutoMatchKey, @ExternalID, @Type, @Mode)";

                    try
                    {
                        ID = (int)command.ExecuteScalar();
                    }
                    catch (SqlException ex)
                    {
                        if (ex.Number == 2601 && ex.Message.ToLower().Contains("ix_voiceinstance_invitecode")) // Unique Index violation. This is expected to happen, from time to time.
                        {
                            return VoiceInstanceSaveStatus.DuplicateInviteCode;
                        }

                        if (ex.Number == 2601 && ex.Message.ToLower().Contains("ix_voiceinstance_automatchkey")) // Unique Index violation. This is expected to happen, from time to time.
                        {
                            return VoiceInstanceSaveStatus.DuplicateAutoMatchKey;
                        }

                        if (ex.Number == 2601 && ex.Message.ToLower().Contains("ix_voiceinstance_externalid_type")) // Unique Index violation. This is expected to happen, from time to time.
                        {
                            return VoiceInstanceSaveStatus.DuplicateExternalID;
                        }

                        throw;
                    }
                }

            }

            CacheManager.Expire(GetCacheKey(ID));

            return VoiceInstanceSaveStatus.Success;

        }

        public static VoiceInstanceSaveStatus UpdateMaxUsersInCall(int userCount, int instanceID)
        {
            try
            {
                using (var conn = DatabaseHelper.Instance.GetConnection())
                {
                    using (var cmd = conn.CreateCommand())
                    {
                        cmd.CommandText = "UPDATE [VoiceInstance] SET MaxUsers = @MaxUsers WHERE [ID] = @ID";
                        cmd.Parameters.AddWithValue("@MaxUsers", userCount);
                        cmd.Parameters.AddWithValue("@ID", instanceID);
                        cmd.ExecuteNonQuery();
                    }
                }
            }
            catch (SqlException ex)
            {
                Logger.Error(ex, "sql exception occured updating maxusers", new { instanceID, userCount });
                return VoiceInstanceSaveStatus.Error;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "exception occured updating maxusers", new { instanceID, userCount });
                return VoiceInstanceSaveStatus.Error;
            }
            return VoiceInstanceSaveStatus.Success;
        }

        public static string GetCacheKey(int id)
        {
            return "VoiceInstance:" + id;
        }

        public const string AllVoiceInstancesDependencyKey = "DependencyKey:AllVoiceInstancesDependencyKey";

        public static void ExpireAll()
        {
            CacheManager.Expire(AllVoiceInstancesDependencyKey);
        }

        public static VoiceInstance GetByID(int id)
        {
            return CacheManager.GetOrAdd(GetCacheKey(id), () => GetFromDatabaseByID(id), TimeSpan.FromHours(1), new[] { AllVoiceInstancesDependencyKey });
        }

        public static VoiceInstance GetByUserID(int userID)
        {
            var id = GetIDFromDatabaseByUserID(userID);
            return id.HasValue ? GetByID(id.Value) : null;
        }

        public static VoiceInstance GetByCode(string code)
        {
            var id = GetIDFromDatabaseByCode(code);
            return id.HasValue ? GetByID(id.Value) : null;
        }

        public static VoiceInstance GetActiveByAutoMatchKey(Int64 autoMatchkey)
        {
            var id = GetActiveIDFromDatabaseByAutoMatchKey(autoMatchkey);
            return id.HasValue ? GetByID(id.Value) : null;
        }

        public static VoiceInstance GetByAutoMatchFingerprint(Int64 autoMatchFingerprint)
        {
            var id = GetIDFromDatabaseByAutoMatchFingerprint(autoMatchFingerprint);
            return id.HasValue ? GetByID(id.Value) : null;
        }

        public static VoiceInstance GetByInviteCode(string inviteCode)
        {
            var id = GetIDFromDatabaseByInviteCode(inviteCode);
            return id.HasValue ? GetByID(id.Value) : null;
        }

        public static VoiceInstance GetActiveByExternalIDAndTypeAndMode(string externalID, VoiceInstanceType type, VoiceInstanceMode mode)
        {
            var id = GetActiveIDFromDatabaseByExternalID(externalID, type, mode);
            return id.HasValue ? GetByID(id.Value) : null;
        }

        public static VoiceInstance GetActiveByExternalIDAndType(string externalID, VoiceInstanceType type)
        {
            var id = GetActiveIDFromDatabaseByExternalID(externalID, type);
            return id.HasValue ? GetByID(id.Value) : null;
        }

        public static VoiceInstance[] GetAllActiveByHost(VoiceHost host)
        {
            var voiceHosts = new List<VoiceInstance>();
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [VoiceInstance] where VoiceHostID = @HostID and Status = @Status";
                    command.Parameters.AddWithValue("@HostID", host.ID);
                    command.Parameters.AddWithValue("@Status", (byte)VoiceInstanceStatus.Active);

                    using (var reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            voiceHosts.Add(new VoiceInstance(reader));
                        }
                    }
                }
            }

            return voiceHosts.ToArray();
        }

        public static int GetCountByUserID(int userID, DateTime since)
        {
            var voiceHosts = new List<VoiceInstance>();
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT COUNT(0) FROM [VoiceInstance] where UserID = @UserID and DateCreated >= @DateCreated";
                    command.Parameters.AddWithValue("@UserID", userID);
                    command.Parameters.AddWithValue("@DateCreated", since);
                    return (int)command.ExecuteScalar();
                }
            }
        }

        public static void MigrateVoiceInstanceKey(Int64 key, int id)
        {
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandType = CommandType.StoredProcedure;
                    command.CommandText = "MigrateVoiceInstanceKey";
                    command.Parameters.AddWithValue("@Key", key);
                    command.Parameters.AddWithValue("@InstanceID", id);
                    command.ExecuteNonQuery();
                }
            }
        }

        public void UpdateGameID(int gameID)
        {
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "UPDATE VoiceInstance SET GameID = @GameID WHERE ID = @ID";
                    command.Parameters.AddWithValue("@GameID", gameID);
                    command.Parameters.AddWithValue("@ID", ID);
                    command.ExecuteNonQuery();
                }
            }

            GameID = gameID;
            CacheManager.Expire(GetCacheKey(ID));
        }

        public VoiceInstanceDetailsResponse ToContract()
        {
            return new VoiceInstanceDetailsResponse
            {
                CallID = Code.ToString(),
                DateCreated = DateCreated,
                DateEnded = DateEnded,
                ExternalID = ExternalID,
                GameID = GameID,
                Status = Status,
                Type = Type,
                UserID = UserID,
                VoiceHostID = VoiceHostID, 
                AutoMatchKey = AutoMatchKey, 
                CreatorName = InviteDisplayName,
                HostID = VoiceHostID,
                HostName = VoiceHost.FullHostName,
                InviteUrl = InviteUrl,
                RegionName = VoiceHost.RegionName,     
                IpAddress = VoiceHost.IPAddress,
                InviteCode = InviteCode,
                Mode = Mode
            };
        }

        public bool ChangeMode(VoiceInstanceMode mode)
        {
            var currentHost = VoiceHost;

            // Find the best failover target for this instance                   
            var newHost = VoiceHostManager.Instance.FindBestHost(VoiceHost.Region, default(Version), mode, new HashSet<int> { VoiceHostID });

            // If one cannot be found, continue
            if (newHost == null)
            {
                Logger.Warn("Unable to change voice instance mode. Could not find a suitable failover target.");
                return false;
            }

            // Otherwise, provision the instance on the target
            var provisionResult = newHost.ProvisionInstance(this);

            if (!provisionResult.IsSuccessful)
            {
                Logger.Warn(provisionResult.Exception, "Unable to change voice instance mode. Could not provision instance on new host.");
                return false;
            }

            // Shutdown lingering voice session, if one exists
            if (ExternalID != null)
            {
                var existing = GetActiveByExternalIDAndTypeAndMode(ExternalID, Type, mode);

                if (existing != null)
                {
                    Logger.Warn("Found an existing instance when changing call mode. It will be set to inactive, to avoid issues.",  new { ExternalID, Type });
                    existing.Status = VoiceInstanceStatus.Inactive;
                    existing.Save();
                }
            }

            // Save thew new host info
            Mode = mode;
            VoiceHostID = newHost.ID;
            Save();

            // We've provisioned the instance at the new host!
            var instructions = new FailoverInstructions
            {
                Status = FailoverStatus.Successful,
                HostName = newHost.FullHostName,
                IPAddress = newHost.IPAddress,
                InstanceCode = Code.ToString(),
                Port = CoreServiceConfiguration.Instance.VoicePortNumber,
                Reason = FailoverReason.ChangingMode
            };

            return currentHost.FailoverInstance(instructions);            
        }
    }
}