﻿using Curse.Logging;
using Curse.Voice.Contracts;
using Curse.Voice.Helpers;
using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Linq;
using System.Threading;

namespace Curse.Voice.Service.Models
{
    public class VoiceRegion
    {

        public static int DefaultRegionID { get; private set; }

#if CONFIG_STAGING
        private int _currentHostIndex = int.MaxValue; // rollover/negative testing
#else
        private int _currentHostIndex = -1;
#endif

        public VoiceHost GetNextHost(Version clientVersion, VoiceInstanceMode mode, HashSet<int> excludeHostIDs = null)
        {
            
            var availableHosts = GetAvailableHostsForClientVersionAndMode(clientVersion, "Primary", mode);

            if (!availableHosts.Any())
            {
                availableHosts = FailoverRegion.GetAvailableHostsForClientVersionAndMode(clientVersion, "Secondary", mode);
            }

            if (!availableHosts.Any() && DefaultRegion != this)
            {
                availableHosts = DefaultRegion.GetAvailableHostsForClientVersionAndMode(clientVersion, "Default", mode);
            }

            if (excludeHostIDs != null && excludeHostIDs.Count > 0)
            {
                availableHosts = availableHosts.Where(p => !excludeHostIDs.Contains(p.ID)).ToArray();
            }
           
            if (!availableHosts.Any())
            {
                Logger.Warn("Unable to find any available hosts for region '" + Name + "' with failover region '" + FailoverRegion.Name);
                return null;
            }

            // Cast values to unsigned so that rollover works properly and modulo never sees a negative value
            var nextHostIndex = (uint)Interlocked.Increment(ref _currentHostIndex);
            nextHostIndex %= (uint)availableHosts.Length;

            return availableHosts[nextHostIndex];
        }

        public VoiceHost GetNextFailoverHost(VoiceInstanceMode mode)
        {

            var availableHosts = GetAvailableHosts("Primary");

            if (!availableHosts.Any())
            {
                availableHosts = FailoverRegion.GetAvailableHosts("Secondary");
            }

            if (!availableHosts.Any() && DefaultRegion != this)
            {
                availableHosts = DefaultRegion.GetAvailableHosts("Default");
            }

            availableHosts = availableHosts.Where(p => p.SupportedModes.HasFlag(mode)).ToArray();

            if (!availableHosts.Any())
            {
                Logger.Warn("Unable to find any available hosts for region '" + Name + "' with failover region '" + FailoverRegion.Name);
                return null;
            }

            return availableHosts.OrderByDescending(p => p.VersionID) // Favor the latest server version
                       .ThenBy(p => p.Statistics.ActiveSessions) // With the fewest sessions
                       .ThenBy(p => p.Statistics.ActiveConnections) // And the fewest connections
                       .FirstOrDefault();
        }

        static VoiceRegion()
        {

            _regions = GetAll().ToDictionary(p => p.ID);
            var defaultRegion = _regions.Values.FirstOrDefault(p => p.IsDefault);

            if (defaultRegion != null)
            {
                DefaultRegionID = defaultRegion.ID;
            }
            
        }

        public static IEnumerable<VoiceRegion> AllRegions
        {
            get { return _regions.Values; }
        }

        public static VoiceRegion DefaultRegion
        {
            get { return GetByID(DefaultRegionID); }
        }

        public string Name { get; set; }
        public int ID { get; set; }
        public bool IsDefault { get; set; }
        public bool IsEnabled { get; set; }
        public int FailoverRegionID { get; set; }
        public string DisplayName { get; set; }
        public int DataRegionID { get; set; }

        public VoiceRegion() 
        {
            
        }

        private VoiceHost[] _hosts = new VoiceHost[0];

        public VoiceHost[] Hosts
        {
            get
            {
                return _hosts;
            }
            
        }
        
        public VoiceRegion(SqlDataReader reader) : this()
        {
            ID = reader.GetInt32(0);
            Name = reader.GetString(1);
            IsDefault = reader.GetBoolean(2);
            IsEnabled = reader.GetBoolean(3);
            FailoverRegionID = reader.GetInt32(4);
            DisplayName = reader.GetString(5);
            DataRegionID = reader.GetInt32(6);
        }

        private static readonly Dictionary<int, VoiceRegion> _regions;

        public static VoiceRegion GetByID(int id)
        {
            VoiceRegion region;
            return _regions.TryGetValue(id, out region) ? region : null;
        }

        private static VoiceRegion[] GetAll()
        {
            var voiceHosts = new List<VoiceRegion>();
            using (var connection = DatabaseHelper.Instance.GetConnection())
            {
                using (var command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [VoiceRegion]";

                    using (var reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {                            
                            voiceHosts.Add(new VoiceRegion(reader));
                        }
                    }
                }
            }

            return voiceHosts.ToArray();
        }

        public int EffectiveRegionID
        {
            get
            {
                // If this region is enabled, simply use it
                if (IsEnabled)
                {
                    return ID;
                }

                // Otherwise try the failover region
                var failoverRegion = GetByID(FailoverRegionID);

                if (failoverRegion != null && failoverRegion.IsEnabled)
                {
                    return failoverRegion.ID;
                }

                // Otherwise, go up the failover chain, until something is found
                return DefaultRegionID;
            }
        }

        public VoiceRegion EffectiveRegion
        {
            get
            {
                if (EffectiveRegionID == ID)
                {
                    return this;
                }
                
                return GetByID(EffectiveRegionID);                
            }
            
        }

        public VoiceRegion FailoverRegion
        {
            get
            {
                if (FailoverRegionID == ID)
                {
                    return this;
                }

                return GetByID(FailoverRegionID);
            }

        }

        public VoiceHost[] GetAvailableHosts(string requestType)
        {
            if (_hosts.Length == 0)
            {
                Logger.Warn("Host list is empty for region: " + Name);
                return _hosts;
            }

            var availableHosts = _hosts.Where(p => p.Availability == VoiceHostAvailability.Available).ToArray();

            if (availableHosts.Any()) return availableHosts;

            Logger.Warn("No available hosts for " + requestType + " region: " + Name);

            foreach (var host in _hosts)
            {
                Logger.Warn("Host '" + host.HostName + "' in region '" + Name + "' is not available: " + host.Availability);
            }

            return availableHosts;
        }

        public VoiceHost[] GetAvailableHostsForClientVersionAndMode(Version clientVersion, string requestType, VoiceInstanceMode mode)
        {
            var availableHosts = GetAvailableHosts(requestType).Where(p => p.SupportedModes.HasFlag(mode));
            
            if (clientVersion != default(Version))
            {
                availableHosts = availableHosts.Where(p => clientVersion >= p.Version.MinimumClientVersion);
            }

#if CONFIG_STAGING
            Logger.Info("AvailableHostsForClientVersionAndMode", new {
                Version = clientVersion != null ? clientVersion.ToString() : "(null)",
                Mode = mode,
                Hosts = availableHosts.Select(h => h.HostName).ToArray(),
            });
#endif

            return availableHosts.ToArray();
        }

        public void UpdateHosts()
        {
            _hosts = VoiceHost.GetAllByRegion(this);
        }


    }
}