﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Web;
using Curse.Logging;
using Curse.Voice.Service.Helpers;
using Curse.Voice.Service.ServiceModels;
using Newtonsoft.Json;

namespace Curse.Voice.Service.Utilities
{
    public static class LogUtility
    {
        private static readonly object SyncRoot = new object();
        private static bool _isInitialized;
        private static bool _isAlive = true;
        private static VoiceLogWriter _logWriter;
        private static string _dnsName;

        internal static void Initialize()
        {
            lock (SyncRoot)
            {
                if (_isInitialized)
                {
                    return;
                }

                _dnsName = System.Net.Dns.GetHostName();
                _isAlive = true;
                Logger.Init(BaseFolderPath, FolderName);
                _logWriter = new VoiceLogWriter();
                Logger.AddLogWriter(_logWriter);
                _isInitialized = true;
            }
        }

        public static string BaseFolderPath
        {
            get { return Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "Logs"); }
        }

        public static string FolderName
        {
            get { return "VoiceService"; }
        }

        public static string FullFolderPath
        {
            get { return Path.Combine(BaseFolderPath, FolderName); }
        }

        public static void Shutdown()
        {
            _isAlive = false;
            try
            {
                Logger.Shutdown();
                _logWriter.UploadToService();
            }
            catch
            {
#if DEBUG
                if (Debugger.IsAttached)
                {
                    Debugger.Break();
                }
#endif
            }
        }

        private class VoiceLogWriter : ILogWriter
        {

            private readonly Queue<LogFileEntry> _entries = new Queue<LogFileEntry>();

            public VoiceLogWriter()
            {
                Task.Factory.StartNew(UploadThread, TaskCreationOptions.LongRunning);
            }

            public void Write(LogFileEntry entry)
            {
                lock (_entries)
                {
                    if (_entries.Count > 5000)
                    {
                        return;
                    }
                    _entries.Enqueue(entry);
                }
            }

            public void Initialize(JsonSerializer serializer)
            {
                // Not used for this implementation
            }

            public void UploadToService()
            {
                lock (SyncRoot)
                {
                    var newEntries = new List<LogFileEntry>();

                    lock (_entries)
                    {
                        var count = _entries.Count;
                        if (count > 5000)
                        {
                            _entries.Clear();
                            _entries.Enqueue(new LogFileEntry
                            {
                                Level = LogLevel.Warn,
                                Message = "[FriendsLogManager] Log data was cleared, due to exceeding limits.",
                                Timestamp = DateTime.UtcNow,
                                Data = new { NumEntries = count }
                            });
                        }

                        // Upload 1,000 entries at a time
                        while (_entries.Count > 0 && newEntries.Count < 1000)
                        {
                            var entry = _entries.Dequeue();
                            if (entry == null)
                            {
                                break;
                            }

                            newEntries.Add(entry);
                        }

                    }

                    if (newEntries.Count == 0)
                    {
                        return;
                    }

                    var success = false;
                    try
                    {
                        SaveToDatabase(newEntries.ToArray());
                        success = true;
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Failed to submit log data to service!");
                    }

                    if (success) return;

                    // Something failed. Add the entries back for uploading later.
                    lock (_entries)
                    {
                        foreach (var entry in newEntries)
                        {
                            _entries.Enqueue(entry);
                        }
                    }
                }
            }

            private static void SaveToDatabase(LogFileEntry[] latestEntries)
            {
                var newLogData = Logger.Serialize(latestEntries);
                var logFilePayload = new LogDataPayload
                {
                    Type = LogDataPayloadType.VoiceService,
                    TypeID = 2,
                    ExternalID = Environment.MachineName,
                    HostName = _dnsName,
                    SerializedLogEntries = newLogData
                };

                LoggingHelper.AddPayload(logFilePayload);
            }

            private void UploadThread()
            {
                while (true)
                {
                    Thread.Sleep(1000);

                    if (!_isAlive)
                    {
                        return;
                    }

                    try
                    {
                        UploadToService();
                    }
                    catch (Exception ex)
                    {
                        if (ex is AggregateException)
                        {
                            ex = (ex as AggregateException).Flatten();
                        }

                        Logger.Error(ex, "Upload Thread Error");
                    }

                }
            }

            public void Shutdown()
            {

            }
        }
    }
}