﻿using Curse.Logging;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Runtime.Caching;
using System.Web;

namespace Curse.Voice.Service.Utilities
{
    public static class TrustChainHelper
    {
        private static readonly string _cacheKey = "TrustedIPMasks";
        private static MemoryCache _memCache = MemoryCache.Default;

        public static string GetClientIP(string ipList)
        {            
            var ipAddresses = ipList.Split(','); 
            if(ipAddresses.Length == 1)
            {
                return ipAddresses[0];
            }

            var trustedCidrs = _memCache.Get(_cacheKey) as IEnumerable<CidrMask>;
            if (trustedCidrs == null)
            {
                trustedCidrs = GetTrustedCidrMasks();
                if(trustedCidrs != null && trustedCidrs.Any())
                {
                    _memCache.Add(_cacheKey, trustedCidrs, new CacheItemPolicy
                    {
                        AbsoluteExpiration = DateTimeOffset.Now.AddHours(1)
                    });
                }
            }
            
            for(int i = ipAddresses.Length - 1; i >= 0; i--)
            {
                IPAddress addr = null; 
                if(IPAddress.TryParse(ipAddresses[i], out addr))
                {
                    var cidr = BitConverter.ToInt32(addr.GetAddressBytes(), 0);
                    if(!trustedCidrs.Any(x => x.IsInRange(cidr)))
                    {
                        return ipAddresses[i];
                    }
                }                
            }

            Logger.Warn("IP list contains only trusted IPs, assuming client is on a trusted domain.", new
            {
                ipAddresses
            });
            return ipAddresses[0];
        }

        private static IEnumerable<CidrMask> GetTrustedCidrMasks()
        {
            var trustedIpRanges = new List<string>();

            var sourcesList = CoreServiceConfiguration.Instance.TrustedCloudflareRangesSource;
            if (!string.IsNullOrWhiteSpace(sourcesList))
            {
                var sources = sourcesList.Split(',');
                using (HttpClient client = new HttpClient())
                {
                    // assumes that these are public lists... cloudflare should be the only one. 
                    foreach (var source in sources)
                    {
                        HttpResponseMessage response = null;
                        try
                        {
                            response = client.GetAsync(source).Result;
                            if (response.IsSuccessStatusCode)
                            {
                                var result = response.Content.ReadAsStringAsync().Result;
                                trustedIpRanges.AddRange(result.Split(new string[] { Environment.NewLine, "\n" }, StringSplitOptions.RemoveEmptyEntries));

                            }
                        }
                        catch(Exception ex)
                        {
                            Logger.Error(ex, "Error getting trusted ip ranges from source",
                                new
                                {
                                    source,
                                    response
                                });
                        }
                    }
                }
            }

            var additionalTrustedRanges = CoreServiceConfiguration.Instance.TrustedIpRanges;
            if(!string.IsNullOrWhiteSpace(additionalTrustedRanges))
            {
                var sources = additionalTrustedRanges.Split(',');
                trustedIpRanges.AddRange(sources);
            }

            Logger.Info("found trusted ip ranegs:", trustedIpRanges);

            // convert the full list to cidr masks
            return trustedIpRanges.Select(x => new CidrMask(x));
        }

        // Implementation ripped from Sage
        private class CidrMask
        {

            private int _ipAddress;
            private int _mask;

            public CidrMask(string mask)
            {
                var parts = mask.Split('/');
                _ipAddress = BitConverter.ToInt32(IPAddress.Parse(parts[0]).GetAddressBytes(), 0);
                _mask = IPAddress.HostToNetworkOrder(-1 << (32 - int.Parse(parts[1])));
            }

            public bool IsInRange(int cidr)
            {
                return ((_ipAddress & _mask) == (cidr & _mask));
            }

            public static bool IsInRange(CidrMask[] cirdMasks, IPAddress ipAddress)
            {
                var cidr = BitConverter.ToInt32(ipAddress.GetAddressBytes(), 0);
                return cirdMasks.Any(x => x.IsInRange(cidr));
            }
        }
    }
}