﻿using Curse.Logging;
using Curse.Voice.Helpers;
using Curse.Voice.Service.Extensions;
using Curse.Voice.Service.GeoCoding;
using Curse.Voice.Service.Models;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Diagnostics;
using System.Linq;
using System.Net;
using System.Runtime.InteropServices;
using System.Threading;
using System.Threading.Tasks;
using Curse.Voice.Contracts;

namespace Curse.Voice.Service
{
    public class VoiceHostManager
    {
        public static readonly VoiceHostManager Instance = new VoiceHostManager();
        private VoiceHost[] _hosts;

        public void Initialize()
        {

            var initStart = DateTime.UtcNow;

            try
            {
                UpdateHostList();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to update host list during initialization!");
            }

            try
            {
                UpdateHostStats();
                Logger.Info("Discovered " + VoiceHost.HostCount + " voice hosts!");
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to update host stats during initialization!");
            }

            new Thread(HostUpdateThread) { IsBackground = true }.Start();
            new Thread(HeartbeatThread) { IsBackground = true }.Start();

            Logger.Info("Voice Host Manager initialized in " + (DateTime.UtcNow - initStart).TotalSeconds.ToString("####,##0.00") + " seconds.");
        }

        public VoiceHost FindBestHost(IPAddress ipAddress, Version clientVersion, VoiceInstanceMode mode, HashSet<int> excludeHostIDs = null)
        {
            // First, get the IP range for this ip address
            var range = IPRange.GetRange(ipAddress.ToInt64());

            // Next, coalesce into a voice region
            var voiceRegion = (range != null ? VoiceRegion.GetByID(range.VoiceRegionID).EffectiveRegion : null) ?? VoiceRegion.DefaultRegion;

            return FindBestHost(voiceRegion, clientVersion, mode, excludeHostIDs);
        }

        public VoiceHost FindBestHost(VoiceRegion voiceRegion, Version clientVersion, VoiceInstanceMode mode, HashSet<int> excludeHostIDs = null)
        {
            return voiceRegion.GetNextHost(clientVersion, mode, excludeHostIDs);
        }

        public VoiceHost FindBestFailoverHost(VoiceRegion voiceRegion, VoiceInstanceMode mode)
        {
            return voiceRegion.GetNextFailoverHost(mode);
        }

        private void HeartbeatThread()
        {
            while (true)
            {
                Thread.Sleep(1000);

                try
                {
                    UpdateHostStats();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Heartbeat thread failure!");
                }
            }
        }

        private class StatsUpdateTimer
        {
            private long TotalCollections { get; set; }
            private double TotalSeconds { get; set; }
            private double MostRecentCollection { get; set; }
            private int NumberOfHosts { get; set; }

            public StatsUpdateTimer(double seconds, int numberOfHosts)
            {
                TrackCollection(seconds, numberOfHosts);
            }

            public void TrackCollection(double seconds, int numberOfHosts)
            {
                MostRecentCollection = seconds;
                TotalSeconds += seconds;
                ++TotalCollections;
                NumberOfHosts = numberOfHosts;
            }

            public string GetStats()
            {
                var average = (TotalSeconds / TotalCollections) * 1000;
                var recent = MostRecentCollection * 1000;
                return "Recent: " + recent.ToString("N2") + " ms, Average: " + average.ToString("N2") + " ms, Number of Hosts: " + NumberOfHosts;
            }
        }

        private readonly ConcurrentDictionary<string, StatsUpdateTimer> _statsUpdateTimerByRegion = new ConcurrentDictionary<string, StatsUpdateTimer>();
        private readonly LogCategory _hostStatsTimer = new LogCategory("Host Stats") { Throttle = TimeSpan.FromMinutes(5), ReleaseLevel = LogLevel.Debug };

        private void UpdateHostStats()
        {
            var allRegions = VoiceRegion.AllRegions.ToArray();
            var hostedRegions = allRegions.Where(p => p.Hosts.Any()).ToArray();
            var totalSw = Stopwatch.StartNew();

            if(!hostedRegions.Any())
            {
                return;
            }

            Parallel.ForEach(hostedRegions, new ParallelOptions { MaxDegreeOfParallelism = hostedRegions.Length }, (region) =>
            {                
                var offlineHosts = region.Hosts.Where(p => p.Status == VoiceHostStatus.Offline);

                foreach (var host in offlineHosts)
                {
                    host.Statistics = new VoiceHostStatistics();
                }

                var sw = Stopwatch.StartNew();

                var activeHosts = region.Hosts.Where(p => p.Status == VoiceHostStatus.Online || p.Status == VoiceHostStatus.Unhealthy).ToArray();
                Parallel.ForEach(activeHosts, host =>
                {
                    try
                    {
                        host.UpdateStatistics();
                    }
                    catch (Exception ex)
                    {
                        Logger.Warn(ex, "Failed to update stats from host: '" + host.IPAddress + "'");
                    }
                });

                sw.Stop();
                StatsUpdateTimer timer;
                if (_statsUpdateTimerByRegion.TryGetValue(region.Name, out timer))
                {
                    timer.TrackCollection(sw.Elapsed.TotalSeconds, activeHosts.Length);
                }
                else
                {
                    _statsUpdateTimerByRegion.TryAdd(region.Name, new StatsUpdateTimer(sw.Elapsed.TotalSeconds, activeHosts.Length));
                }
            });

            var stats = _statsUpdateTimerByRegion.Select(p => new {Region = p.Key, Stats = p.Value.GetStats()});
            _hostStatsTimer.Debug("Finished collecting host stats", stats);

        }

        private void HostUpdateThread()
        {
            while (true)
            {
                Thread.Sleep(1000); // Update our hosts every second
                UpdateHostList();
            }
        }

        private void UpdateHostList()
        {
            try
            {
                VoiceHostVersion.UpdateFromDatabase();
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to retrieve voice host versions from database.");
            }

            try
            {
                VoiceHost.UpdateFromDatabase();
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to retrieve voice hosts from database.");
            }

            try
            {
                foreach (var region in VoiceRegion.AllRegions)
                {
                    try
                    {
                        region.UpdateHosts();
                    }
                    catch (Exception ex)
                    {
                        Logger.Warn(ex, "Failed to update host list for region: '" + region.Name + "'");
                    }

                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to update hosts for region.");
            }

            try
            {
                _hosts = VoiceRegion.AllRegions.SelectMany(r => r.Hosts).ToArray();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to update hosts in voice host manager!");
            }

        }

        public VoiceHost[] Hosts
        {
            get
            {
                return _hosts;
            }
        }
    }
}