﻿using System;
using System.Collections.Concurrent;
using System.Text.RegularExpressions;
using Curse.Logging;
using Curse.ServiceAuthentication;
using Curse.Voice.Service.Models;
using Curse.Voice.Service.Responses;
using Curse.Voice.Service.Helpers;

namespace Curse.Voice.Service
{
    public static class VoiceSessionManager
    {
        private readonly static ConcurrentDictionary<int, int> FailedSessionAttempts = new ConcurrentDictionary<int, int>();

        public static BasicVoiceServiceResponse ReportGameState(string sessionGuid, int gameID)
        {
            try
            {
                var instance = VoiceInstance.GetByCode(sessionGuid);

                if (instance == null)
                {
                    return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.NotFound };
                }

                // No game defined, or they already match
                if (gameID == 0 || instance.GameID == gameID || instance.GameID == -1)
                {
                    return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Successful };
                }
                
                // If the existing game is 0, always overwrite it
                if (instance.GameID == 0)
                {
                    instance.UpdateGameID(gameID);
                }
                // If it's a different game set it to multi-game
                else if (instance.GameID != gameID) 
                {
                    instance.UpdateGameID(-1);
                }
                
                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Successful };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to report game state.");

                return new BasicVoiceServiceResponse { Status = BasicVoiceServiceStatus.Error };
            }
        }


        private static readonly Regex _inviteRegex = new Regex(@"^http\://.*?/join/(?<Code>.*?)$", RegexOptions.IgnoreCase);
        private static readonly Regex _shortCodeRegex = new Regex(@"^http\://cv\.gg/(?<Code>.*?)$", RegexOptions.IgnoreCase);

        public static GetVoiceSessionResponse GetVoiceSession(string inviteCode, Version clientVersion)
        {
            try
            {                
                if (inviteCode.StartsWith("http", StringComparison.InvariantCultureIgnoreCase))
                {
                    // Try to get the invite code
                    var inviteCodeMatch = _inviteRegex.Match(inviteCode);
                    
                    if (!inviteCodeMatch.Success)
                    {
                        inviteCodeMatch = _shortCodeRegex.Match(inviteCode);
                    }

                    if (inviteCodeMatch.Success)
                    {
                        inviteCode = inviteCodeMatch.Groups["Code"].Value;
                    }
                    else
                    {
                        return new GetVoiceSessionResponse { Status = GetVoiceSessionStatus.NotFound };
                    }
                }

                var session = AuthenticationProvider.CurrentSession;
                var instance = VoiceInstance.GetByInviteCode(inviteCode);

                if (instance == null)
                {
                    var failureCount = FailedSessionAttempts.AddOrUpdate(session.UserID, i => 1, (key, val) => val + 1);

                    if (failureCount >= CoreServiceConfiguration.Instance.MaxJoinFailures)
                    {

                        VoiceBanManager.Ban(session.UserID, failureCount % 10 == 0, new { FailureCount = failureCount, InviteCode = inviteCode, ClientVersion = clientVersion.ToString() });
                        return new GetVoiceSessionResponse
                        {
                            Status = GetVoiceSessionStatus.Throttled
                        };
                    }

                    return new GetVoiceSessionResponse
                    {
                        Status = GetVoiceSessionStatus.NotFound
                    };
                }

                if (VoiceBanManager.IsBanned(session.UserID))
                {
                    return new GetVoiceSessionResponse
                    {
                        Status = GetVoiceSessionStatus.Throttled
                    };
                }

                // Reset the user's failed attempt count, if it exists
                if (FailedSessionAttempts.ContainsKey(session.UserID))
                {
                    int removedUserID;
                    FailedSessionAttempts.TryRemove(session.UserID, out removedUserID);
                }


                // Ensure the voice host isn't null
                if (instance.VoiceHost == null)
                {
                    instance.Status = VoiceInstanceStatus.Inactive;
                    instance.Save();
                    return new GetVoiceSessionResponse { Status = GetVoiceSessionStatus.NotFound };
                }

                // Ensure that the user has a compatible client
                if (clientVersion < instance.VoiceHost.Version.MinimumClientVersion)
                {
                    return new GetVoiceSessionResponse
                    {
                        Status = GetVoiceSessionStatus.IncompatibleClient
                    };
                }                

                return new GetVoiceSessionResponse
                {
                    Status = GetVoiceSessionStatus.Successful,
                    CreatorName = instance.InviteDisplayName,
                    GameID = instance.GameID > 0 ? instance.GameID : null,
                    HostName = instance.VoiceHost.HostName,
                    IPAddress = instance.VoiceHost.IPAddress,
                    Port = CoreServiceConfiguration.Instance.VoicePortNumber,
                    InviteUrl = instance.InviteUrl,
                    InstanceCode = instance.Code.ToString(),
                    HostID = instance.VoiceHostID,
                    IsAutoMatch = instance.AutoMatchKey.HasValue,
                    AutoMatchKey = instance.AutoMatchKey,
                    RegionName = instance.VoiceHost.Region.DisplayName
                };

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to get voice session.");

                return new GetVoiceSessionResponse
                {
                    Status = GetVoiceSessionStatus.Error,
                    StatusMessage = ex.Message + ", Stack: " + ex.StackTrace
                };
            }
        }
    }
}