﻿using System.ServiceModel;
using Curse.Caching;
using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Linq;
using System.Web;
using Curse.Voice.Helpers;
using Curse.Voice.HostManagement;
using Curse.Voice.HostManagement.Responses;
using System.Net;
using System.Data.SqlTypes;
using Curse.Voice.Service.Responses;

namespace Curse.Voice.Service.Models
{
    public enum VoiceHostStatus 
    {
        Online = 1,
        Offline = 2,
        ShuttingDown = 3
    }

    public class VoiceHost : IDisposable
    {
        private static Dictionary<int, VoiceHost> _hosts;

        static VoiceHost()
        {

            try
            {
                _hosts = GetAll(null).ToDictionary(p => p.ID);                       
            }
            catch (Exception ex)
            {
                _hosts = new Dictionary<int, VoiceHost>();
                Logger.ErrorException("Failed to provision any hosts!", ex);
            }
            
        }

        private static DateTime? _lastHostUpdate = null;

        public static void UpdateFromDatabase()
        {
            DateTime startTime = DateTime.UtcNow;

            // Get all hosts for this region.
            var allHosts = GetAll(_lastHostUpdate);

            // Reconnect any hosts which have new IP addresses (happens anytime a VM reboots, on AWS)
            var modifiedHosts = _hosts.Values.Join(allHosts, d => d.ID, n => n.ID, (a, b) => new { CurrentHost = a, NewHost = b }).ToArray();

            foreach (var pair in modifiedHosts)
            {
                if (pair.NewHost.DateUpdated <= pair.CurrentHost.DateUpdated)
                {
                    continue; // Already up to date
                }
                pair.CurrentHost.Status = pair.NewHost.Status;
                pair.CurrentHost.DateUpdated = pair.NewHost.DateUpdated;

                if (pair.CurrentHost.IPAddress != pair.NewHost.IPAddress)
                {
                    pair.CurrentHost.IPAddress = pair.NewHost.IPAddress;
                    pair.CurrentHost.Disconnect();
                }
            }

            // Add any new hosts, which are not yet in the list
            var addedHosts = allHosts.Where(p => !_hosts.ContainsKey(p.ID));
            foreach (var addedHost in addedHosts)
            {
                _hosts.Add(addedHost.ID, addedHost);
            }

            _lastHostUpdate = startTime.AddMinutes(-1);
        }

        public int ID { get; set; }
        public DateTime DateCreated { get; set; }
        public DateTime DateUpdated { get; set; }
        public string IPAddress { get; set; }
        public string HostName { get; set; }
        public VoiceHostStatus Status { get; set; }
        public int RegionID { get; set; }
        public VoiceHostEnvironment Environment { get; set; }
        public Guid ExternalID { get; set; }
        public DateTime DateOnline { get; set; }
        public DateTime DateOffline { get; set; }

        public bool IsConnected { get; private set; }

        public VoiceHostStatistics Statistics
        {
            get;
            private set;
        }

        public VoiceRegion Region
        {
            get { return VoiceRegion.GetByID(RegionID); }
        }

        public VoiceHost() { }

        private VoiceHost(SqlDataReader reader)
        {
            ID = reader.GetInt32(0);
            DateCreated = reader.GetDateTime(1);
            DateUpdated = reader.GetDateTime(2);
            HostName = reader.GetString(3);
            IPAddress = reader.GetString(4);            
            Status = (VoiceHostStatus)reader.GetByte(5);
            RegionID = reader.GetInt32(6);
            Environment = (VoiceHostEnvironment)reader.GetByte(7);
            ExternalID = reader.GetGuid(8);
            DateOnline = reader.GetDateTime(9);
            DateOffline = reader.GetDateTime(10);
        }

        public void SaveToDatabase()
        {
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    if (ID > 0)
                    {
                        command.CommandText = "UPDATE [VoiceHost] SET HostName = @HostName, IPAddress = @IPAddress, Status = @Status, RegionID = @RegionID, Environment = @Environment, ExternalID = @ExternalID, DateUpdated = GETUTCDATE(), DateOnline = @DateOnline, DateOffline = @DateOffline WHERE ID = @ID";
                        command.Parameters.AddWithValue("@ID", ID);

                        if (DateOnline >= SqlDateTime.MinValue)
                        {
                            command.Parameters.AddWithValue("@DateOnline", DateOnline);
                        }
                        else
                        {
                            command.Parameters.AddWithValue("@DateOnline", DateTime.UtcNow);
                        }

                        if (DateOffline >= SqlDateTime.MinValue)
                        {
                            command.Parameters.AddWithValue("@DateOffline", DateOnline);
                        }
                        else
                        {
                            command.Parameters.AddWithValue("@DateOffline", DateTime.UtcNow);
                        }
                    }
                    else
                    {
                        command.CommandText = "INSERT INTO [VoiceHost] (HostName, IPAddress, Status, RegionID, Environment, ExternalID) VALUES(@HostName, @IPAddress, @Status, @RegionID, @Environment, @ExternalID)";
                    }

                    command.Parameters.AddWithValue("@HostName", HostName);
                    command.Parameters.AddWithValue("@IPAddress", IPAddress);
                    command.Parameters.AddWithValue("@Status", (byte)Status);
                    command.Parameters.AddWithValue("@RegionID", RegionID);
                    command.Parameters.AddWithValue("@Environment", (byte)Environment);
                    command.Parameters.AddWithValue("@ExternalID", ExternalID);                    
                    command.ExecuteNonQuery();
                }
            }
        }

        public void MarkInstancesOffline()
        {
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    command.CommandText = "UPDATE [VoiceInstance] SET Status = @Status where Status <> @Status AND DateCreated < @DateCreated and VoiceHostID = @VoiceHostID";
                    command.Parameters.AddWithValue("@Status", (byte)VoiceInstanceStatus.Shutdown);
                    command.Parameters.AddWithValue("@DateCreated", DateOnline);
                    command.Parameters.AddWithValue("@VoiceHostID", ID);
                    int rowsAffected = (int)command.ExecuteNonQuery();
                    Logger.Info("Voice Host '" + HostName + "' has come online, and marked " + rowsAffected.ToString("###,##0") + " active voices instances as offline.");
                    VoiceInstance.ExpireAll();
                }
            }
        }      

        private static VoiceHost[] GetAll(DateTime? since)
        {
            List<VoiceHost> voiceHosts = new List<VoiceHost>();
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [VoiceHost] WHERE Environment = @Environment";
                    command.Parameters.AddWithValue("@Environment", (int)CoreServiceConfiguration.Instance.Environment);
                    if (since.HasValue)
                    {
                        command.CommandText += " and DateUpdated >= @DateUpdated";
                        command.Parameters.AddWithValue("@DateUpdated", since);
                    }

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {                            
                            voiceHosts.Add(new VoiceHost(reader));
                        }                        
                    }
                }
            }

            return voiceHosts.ToArray();
        }
        
        public static VoiceHost GetByIPAddressOrExternalID(IPAddress ipAddress, Guid externalID)
        {            
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT ID FROM [VoiceHost] where ExternalID = @ExternalID or IPAddress = @IPAddress";
                    command.Parameters.AddWithValue("@ExternalID", externalID);
                    command.Parameters.AddWithValue("@IPAddress", ipAddress.ToString());
                    
                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            int id = reader.GetInt32(0);
                            return GetByID(id);
                        }
                    }
                }
            }

            return null;
        }

        public static VoiceHost[] GetAllByRegion(VoiceRegion voiceRegion)
        {
            List<VoiceHost> voiceHosts = new List<VoiceHost>();
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT ID FROM [VoiceHost] where RegionID = @RegionID";
                    command.Parameters.AddWithValue("@RegionID", voiceRegion.ID);
                    
                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            int id = reader.GetInt32(0);
                            var host = GetByID(id);
                            if (host != null)
                            {
                                voiceHosts.Add(host);
                            }
                        }
                    }
                }
            }

            return voiceHosts.ToArray();
        }

        public static VoiceHost GetByID(int id)
        {
            VoiceHost host = null;

            if (_hosts.TryGetValue(id, out host))
            {
                return host;
            }
            else
            {
                return null;
            }                        
        }
              
        public void Disconnect()
        {
            IsConnected = false;
            if (_voiceManagementClient != null)
            {                
                var conn = _voiceManagementClient;
                _voiceManagementClient = null;
                try
                {
                    conn.Close();
                }
                catch (Exception ex)
                {
                    Logger.Debug("Failed to disconnect from: " + IPAddress);
                }
                
            }
        }
        
        
        public void Dispose()
        {
            Disconnect();
        }


        #region Network 

        private VoiceManagementClient _voiceManagementClient = null;        
       
        private VoiceManagementClient CreateNewClient()
        {
            return new VoiceManagementClient(new InstanceContext(new VoiceHostCallbackClient()), IPAddress, CoreServiceConfiguration.Instance.HostPortNumber);   
        }
        
        public bool ProvisionInstance(VoiceInstance instance)
        {
            if (!IsConnected)
            {
                return false;
            }

            try
            {
                _voiceManagementClient.ProvisionVoiceInstance(instance.Code.ToString(), instance.UserID);                
            }
            catch (Exception ex)
            {
                Logger.ErrorException("Failed to provision instance: " + ex.Message, ex);
                return false;
            }

            return true;

        }

        public bool DestroyInstance(VoiceInstance instance)
        {

            if (!IsConnected)
            {
                return false;
            }

            try
            {
                _voiceManagementClient.DestroyVoiceInstance(instance.Code.ToString());
            }
            catch (Exception ex)
            {
                Logger.ErrorException("Failed to destroy instance: " + ex.Message, ex);
                return false;
            }

            return true;

        }

        public void SendHeartbeat()
        {
            if (!IsConnected)
            {
                return;
            }

            try
            {
                _voiceManagementClient.SendHeartbeat();
            }
            catch (Exception ex)
            {
                Logger.ErrorException("Failed to send heartment to host!", ex);                
            }
        }       
        
        private bool EnsureConnectivity()
        {
            try
            {
                if (_voiceManagementClient == null)
                {
                    _voiceManagementClient = CreateNewClient();
                    _voiceManagementClient.Open();                    
                }
                else if (_voiceManagementClient.State != CommunicationState.Opened)
                {
                    if (_voiceManagementClient.State == CommunicationState.Faulted)
                    {
                        _voiceManagementClient.Abort();
                    }
                    else if (_voiceManagementClient.State != CommunicationState.Closed)
                    {
                        _voiceManagementClient.Close();
                    }

                    _voiceManagementClient = CreateNewClient();
                    _voiceManagementClient.Open();
                }

                return true;
            }
            catch (CommunicationException)
            {                
                return false;
            }
            catch (Exception ex)
            {
                Logger.WarnException("Unable to establish a management service connection to: " + IPAddress, ex);
                return false;
            }
           
        }

        #endregion

        public bool IsAvailable
        {
            get
            {
                if (Status != VoiceHostStatus.Online)
                {
                    Logger.Debug("IsAvailable: Voice host is not online!");
                    return false;
                }

                if (!IsConnected)
                {
                    Logger.Debug("IsAvailable: Voice host is not connected!");
                    return false;
                }

                if (Statistics == null || Statistics.LastUpdated < DateTime.UtcNow.AddSeconds(-10))
                {
                    Logger.Debug("IsAvailable: Voice host has missing our out of date statistics!");
                    return false;
                }

                if (Statistics.ActiveConnections >= CoreServiceConfiguration.Instance.MaximumConnectionsPerHost)
                {
                    Logger.Debug("IsAvailable: Voice host is at capacity!");
                    return false;
                }

                if (Statistics.CurrentBandwidthUsage >= CoreServiceConfiguration.Instance.MaximumBandwithUtilization)
                {
                    Logger.Debug("IsAvailable: Voice host has no available bandwidth!");
                    return false;
                }

                return true;
            }
        }

        public void UpdateStatistics()
        {
            if (!EnsureConnectivity())
            {
                IsConnected = false;
                return;
            }
            
            IsConnected = true;
            
            try
            {
                var stats = _voiceManagementClient.GetStatistics();
                Statistics = Statistics ?? new VoiceHostStatistics();
                Statistics.ActiveConnections = stats.ActiveConnections;
                Statistics.CurrentBandwidthUsage = stats.CurrentBandwidthUsage;
                Statistics.LastUpdated = DateTime.UtcNow;
            }
            catch (CommunicationException ex)
            {
                Logger.WarnException("Communication error when trying get voice host statistics from: " + IPAddress, ex);
                IsConnected = false;
            }
            catch (Exception ex)
            {
                Logger.ErrorException("Unhandled error when trying to get voice host statistics from: " + IPAddress, ex);                
            }
        }


        #region Failover


        public ShutdownHostResponse FailoverInstances()
        {
            DateTime failoverStartTime = DateTime.UtcNow;
            Logger.Info("Beginning failover for host '" + HostName + "'");

            // Mark the host as shutting down
            Status = VoiceHostStatus.ShuttingDown;
            SaveToDatabase();

            // See if the region has any available hosts
            if (!Region.AvailableHosts.Any() && !Region.FailoverRegion.AvailableHosts.Any())
            {
                Logger.Info("Planned failover has failed for host '" + HostName + "', as it has no available failover targets!");
                return new ShutdownHostResponse() {Status = ShutdownHostStatus.Failed};
            }

            // Get all instances on the host, which are active, and provision them on other hosts
            var activeInstances = VoiceInstance.GetAllActiveByHost(this);

            Logger.Info("Failing over " + activeInstances.Count().ToString("###,##0") + " instances for host '" + HostName + "'");

            var currentRegion = this.Region;

            List<VoiceInstanceFailoverResult> results = new List<VoiceInstanceFailoverResult>();

            foreach (var instance in activeInstances)
            {
                try
                {
                    // Find the best failover target for this instance
                    var newHost = VoiceHostManager.Instance.FindBestHost(currentRegion);

                    // If one cannot be found, continue
                    if (newHost == null)
                    {
                        Logger.Info("Unable to find suitable failover host for instance '" + instance.Code + "' on host '" + HostName + "'");
                        results.Add(new VoiceInstanceFailoverResult(VoiceInstanceFailoverStatus.Failed));
                        continue;
                    }

                    // Otherwise, provision the instance on the target
                    if (!newHost.ProvisionInstance(instance))
                    {
                        Logger.Info("Unable to find provision instance '{0}' on failover target target host '{1}'", instance.Code, newHost.HostName);
                        results.Add(new VoiceInstanceFailoverResult(VoiceInstanceFailoverStatus.Failed));
                        continue;
                    }

                    // Save thew new host info
                    instance.VoiceHostID = newHost.ID;
                    instance.Save();

                    // We've provisioned the instance at the new host!
                    results.Add(new VoiceInstanceFailoverResult(VoiceInstanceFailoverStatus.Successful)
                        {
                            HostName = newHost.HostName,
                            IPAddress = newHost.IPAddress,
                            InstanceCode = instance.Code.ToString(),
                            Port = CoreServiceConfiguration.Instance.VoicePortNumber
                        });
                }
                catch (Exception ex)
                {
                    Logger.ErrorException("Instance failover generated an exception!", ex);
                    results.Add(new VoiceInstanceFailoverResult(VoiceInstanceFailoverStatus.Error));
                }               
            }

            return new ShutdownHostResponse() { Status = ShutdownHostStatus.Successful, Results = results.ToArray() };
        }

        #endregion
    }
}