﻿using Curse.Caching;
using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Linq;
using System.Web;
using Curse.Voice.Helpers;

namespace Curse.Voice.Service.Models
{

    public enum VoiceInstanceStatus
    {
        Active = 1,
        Inactive = 2,
        Shutdown = 3
    }

    public class VoiceInstance
    {
        public int ID { get; set; }
        public VoiceInstanceStatus Status { get; set; }
        public int UserID { get; set; }
        public Guid Code { get; set; }
        public DateTime DateCreated { get; set; }
        public int VoiceHostID { get; set; }
        public int? GameID { get; set; }

        public VoiceHost VoiceHost
        {
            get { return VoiceHost.GetByID(VoiceHostID); }
        }

        public VoiceInstance() { }

        public VoiceInstance(SqlDataReader reader)
        {
            ID = reader.GetInt32(0);
            Status = (VoiceInstanceStatus)reader.GetByte(1);
            UserID = reader.GetInt32(2);
            Code = reader.GetGuid(3);
            DateCreated = reader.GetDateTime(4);
            VoiceHostID = reader.GetInt32(5);

            if (reader[6] != DBNull.Value)
            {
                GameID = reader.GetInt32(6);
            }
        }

        public static VoiceInstance GetFromDatabaseByID(int id)
        {
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [VoiceInstance] WHERE ID = @ID";
                    command.Parameters.AddWithValue("@ID", id);

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return new VoiceInstance(reader);
                        }

                        return null;
                    }
                }
            }
        }
        
        public static int? GetIDFromDatabaseByUserID(int userID)
        {
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT ID FROM [VoiceInstance] WHERE [Status] = 1 AND [UserID] = @UserID";
                    command.Parameters.AddWithValue("@UserID", userID);

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return reader.GetInt32(0);
                        }

                        return null;
                    }
                }
            }
        }

        public static int? GetIDFromDatabaseByCode(string code)
        {
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT ID FROM [VoiceInstance] WHERE [Code] = @Code";
                    command.Parameters.AddWithValue("@Code", code);

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return reader.GetInt32(0);
                        }

                        return null;
                    }
                }
            }
        }

        public void Save()
        {
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                Save(connection);
            }
        }

        public void Save(SqlConnection connection, SqlTransaction transaction = null)
        {
            using (SqlCommand command = connection.CreateCommand())
            {
                if(transaction != null)
                {
                    command.Transaction = transaction;
                }

                command.Parameters.AddWithValue("@Status", (byte)Status);
                command.Parameters.AddWithValue("@UserID", UserID);
                command.Parameters.AddWithValue("@Code", Code);
                command.Parameters.AddWithValue("@DateCreated", DateCreated);
                command.Parameters.AddWithValue("@VoiceHostID", VoiceHostID);
                command.Parameters.AddWithValue("@GameID", GameID.HasValue ? (object)GameID.Value : DBNull.Value);

                if (ID > 0)
                {
                    command.Parameters.AddWithValue("@ID", ID);
                    command.CommandText = "UPDATE [VoiceInstance] SET Status = @Status, UserID = @UserID, Code = @Code, DateCreated = @DateCreated, VoiceHostID = @VoiceHostID, GameID = @GameID where ID = @ID";
                    command.ExecuteNonQuery();
                }
                else
                {
                    command.CommandText = "INSERT INTO [VoiceInstance] (Status, UserID, Code, DateCreated, VoiceHostID, GameID) OUTPUT INSERTED.ID VALUES(@Status, @UserID, @Code, @DateCreated, @VoiceHostID, @GameID)";
                    ID = (int)command.ExecuteScalar();
                }
            }


            CacheManager.Expire(GetCacheKey(ID));

        }

        public static string GetCacheKey(int id)
        {
            return "VoiceInstance:" + id;
        }

        public const string AllVoiceInstancesDependencyKey = "DependencyKey:AllVoiceInstancesDependencyKey";

        public static void ExpireAll()
        {
            CacheManager.Expire(AllVoiceInstancesDependencyKey);
        }

        public static VoiceInstance GetByID(int id)
        {
            return CacheManager.GetOrAdd(GetCacheKey(id), () => GetFromDatabaseByID(id), TimeSpan.FromHours(1), new[] { AllVoiceInstancesDependencyKey  });
        }

        public static VoiceInstance GetByUserID(int userID)
        {
            int? id = GetIDFromDatabaseByUserID(userID);

            if (!id.HasValue)
            {
                return null;
            }

            return GetByID(id.Value);
        }

        public static VoiceInstance GetByCode(string code)
        {
            int? id = GetIDFromDatabaseByCode(code);

            if (!id.HasValue)
            {
                return null;
            }

            return GetByID(id.Value);
        }

        public static VoiceInstance[] GetAllActiveByHost(VoiceHost host)
        {
            List<VoiceInstance> voiceHosts = new List<VoiceInstance>();
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [VoiceInstance] where VoiceHostID = @HostID and Status = @Status";                    
                    command.Parameters.AddWithValue("@HostID", host.ID);
                    command.Parameters.AddWithValue("@Status", (byte)VoiceInstanceStatus.Active);


                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {                            
                            voiceHosts.Add(new VoiceInstance(reader));
                        }
                    }
                }
            }

            return voiceHosts.ToArray();
        }
    }
}