﻿using Curse.Caching;
using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Linq;
using System.Web;
using Curse.Voice.Helpers;

namespace Curse.Voice.Service.Models
{

    public class VoiceInvitation
    {
        public int ID { get; set; }
        public int VoiceInstanceID { get; set; }
        public string Code { get; set; }
        public DateTime DateCreated { get; set; }
        public string CreatorName { get; set; }

        public string Url
        {
            get
            {
                return string.Format(CoreServiceConfiguration.Instance.InviteUrlFormat, Code);
            }
        }

        public VoiceInstance VoiceInstance
        {
            get { return VoiceInstance.GetByID(VoiceInstanceID); }
        }

        public VoiceInvitation() { }

        public VoiceInvitation(SqlDataReader reader)
        {
            ID = reader.GetInt32(0);
            VoiceInstanceID = reader.GetInt32(1);
            Code = reader.GetString(2);
            DateCreated = reader.GetDateTime(3);
            CreatorName = reader.GetString(4);

        }

        public static VoiceInvitation GetFromDatabaseByID(int id)
        {
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [VoiceInvitation] WHERE ID = @ID";
                    command.Parameters.AddWithValue("@ID", id);

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return new VoiceInvitation(reader);
                        }

                        return null;
                    }
                }
            }
        }

        public static int? GetIDFromDatabaseByCode(string code)
        {
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT ID FROM [VoiceInvitation] WHERE Code = @Code";
                    command.Parameters.AddWithValue("@Code", code);

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return reader.GetInt32(0);
                        }

                        return null;
                    }
                }
            }
        }

        public static int? GetIDFromDatabaseByInstance(int id)
        {
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT ID FROM [VoiceInvitation] WHERE VoiceInstanceID = @InstanceID";
                    command.Parameters.AddWithValue("@InstanceID", id);

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        if (reader.Read())
                        {
                            return reader.GetInt32(0);
                        }

                        return null;
                    }
                }
            }
        }

        public bool Save(SqlConnection connection, SqlTransaction transaction)
        {            
            using (SqlCommand command = connection.CreateCommand())
            {
                command.Transaction = transaction;
                command.Parameters.AddWithValue("@VoiceInstanceID", VoiceInstanceID);
                command.Parameters.AddWithValue("@Code", Code);
                command.Parameters.AddWithValue("@DateCreated", DateCreated);
                command.Parameters.AddWithValue("@CreatorName", CreatorName);                    

                if (ID > 0)
                {
                    command.Parameters.AddWithValue("@ID", ID);
                    command.CommandText = "UPDATE [VoiceInvitation] SET VoiceInstanceID = @VoiceInstanceID, Code = @Code, DateCreated = @DateCreated, CreatorName = @CreatorName where ID = @ID";
                    command.ExecuteNonQuery();
                }
                else
                {
                    command.CommandText = "INSERT INTO [VoiceInvitation] (VoiceInstanceID, Code, DateCreated, CreatorName) OUTPUT INSERTED.ID VALUES(@VoiceInstanceID, @Code, @DateCreated, @CreatorName)";
                    try
                    {
                        ID = (int)command.ExecuteScalar();
                    }
                    catch (SqlException ex)
                    {
                        if (ex.Number == 2601) // Unique Index violation. This is expected to happen, from time to time.
                        {
                            Logger.Error("Failed to create an invitation. The code is not unique: " + Code);
                            return false;
                        }                            
                        throw;
                            
                    }
                        
                }
            }
            

            CacheManager.Expire(GetCacheKey(ID));

            return true;
        }

        public static string GetCacheKey(int id)
        {
            return "VoiceInvitationByID:" + id;
        }

        public static VoiceInvitation GetByInstance(int instanceId)
        {
            int? id = CacheManager.GetOrAdd("VoiceInvitationByInstance:" + instanceId, () => GetIDFromDatabaseByInstance(instanceId));

            if (!id.HasValue)
            {
                return null;
            }

            return GetByID(id.Value);
        }

        public static VoiceInvitation GetByID(int id)
        {
            return CacheManager.GetOrAdd(GetCacheKey(id), () => GetFromDatabaseByID(id));
        }

        public static VoiceInvitation GetByCode(string code)
        {
            int? id = CacheManager.GetOrAdd("VoiceInvitationByCode:" + code, () => GetIDFromDatabaseByCode(code));

            if (!id.HasValue)
            {
                return null;
            }

            return GetByID(id.Value);
        }
    }
}