﻿using Curse.Caching;
using Curse.Voice.Helpers;
using Curse.Voice.Service.Utilities;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Linq;
using System.Threading;
using System.Web;

namespace Curse.Voice.Service.Models
{
    public class VoiceRegion
    {

        public static int DefaultRegionID { get; private set; }

        private int _currentHostIndex = -1;

        public VoiceHost GetNextHost()
        {
            
            var availableHosts = AvailableHosts;

            if (!availableHosts.Any())
            {
                Logger.Warn("No available hosts found for region '" + Name + "', trying failover region, '" + FailoverRegion.Name + "'. Total host count is '" + Hosts.Length + "'");
                availableHosts = FailoverRegion.AvailableHosts;
            }

            if (!availableHosts.Any())
            {
                Logger.Warn("No available hosts found for '" + FailoverRegion.Name + "'. Total host count is '" + FailoverRegion.Hosts.Length + "'");
                Logger.Warn("Unable to find any available hosts for region '" + Name + "' with failover region '" + FailoverRegion.Name);
                return null;
            }

            int nextHostIndex = Interlocked.Increment(ref _currentHostIndex);
            if (nextHostIndex > availableHosts.Length - 1)
            {
                nextHostIndex = _currentHostIndex = 0;
            }

            return availableHosts[nextHostIndex];
        }

        static VoiceRegion()
        {

            _regions = GetAll().ToDictionary(p => p.ID);
            var defaultRegion = _regions.Values.FirstOrDefault(p => p.IsDefault);

            if (defaultRegion != null)
            {
                DefaultRegionID = defaultRegion.ID;
            }
            
        }

        public static IEnumerable<VoiceRegion> AllRegions
        {
            get { return _regions.Values; }
        }

        public static VoiceRegion DefaultRegion
        {
            get { return GetByID(DefaultRegionID); }
        }

        public string Name { get; set; }
        public int ID { get; set; }
        public bool IsDefault { get; set; }
        public bool IsEnabled { get; set; }
        public int FailoverRegionID { get; set; }
        
        
        public VoiceRegion() 
        {
            
        }

        private VoiceHost[] _hosts = new VoiceHost[0];

        public VoiceHost[] Hosts
        {
            get
            {
                return _hosts;
            }
            
        }
        
        public VoiceRegion(SqlDataReader reader) : this()
        {
            ID = reader.GetInt32(0);
            Name = reader.GetString(1);
            IsDefault = reader.GetBoolean(2);
            IsEnabled = reader.GetBoolean(3);
            FailoverRegionID = reader.GetInt32(4);
        }
        private static Dictionary<int, VoiceRegion> _regions;

        public static VoiceRegion GetByID(int id)
        {
            VoiceRegion region = null;
            if (_regions.TryGetValue(id, out region))
            {
                return region;
            }
            else
            {
                return null;
            }
        }

        private static VoiceRegion[] GetAll()
        {
            List<VoiceRegion> voiceHosts = new List<VoiceRegion>();
            using (SqlConnection connection = DatabaseHelper.Instance.GetConnection())
            {
                using (SqlCommand command = connection.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM [VoiceRegion]";

                    using (SqlDataReader reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            int id = reader.GetInt32(0);
                            voiceHosts.Add(new VoiceRegion(reader));
                        }
                    }
                }
            }

            return voiceHosts.ToArray();
        }

        public int EffectiveRegionID
        {
            get
            {
                // If this region is enabled, simply use it
                if (IsEnabled)
                {
                    return ID;
                }

                // Otherwise try the failover region
                var failoverRegion = GetByID(FailoverRegionID);

                if (failoverRegion != null && failoverRegion.IsEnabled)
                {
                    return failoverRegion.ID;
                }

                // Otherwise, go up the failover chain, until something is found
                return DefaultRegionID;
            }
        }

        public VoiceRegion EffectiveRegion
        {
            get
            {
                if (EffectiveRegionID == ID)
                {
                    return this;
                }
                
                return GetByID(EffectiveRegionID);                
            }
            
        }

        public VoiceRegion FailoverRegion
        {
            get
            {
                if (FailoverRegionID == ID)
                {
                    return this;
                }

                return GetByID(FailoverRegionID);
            }

        }

        public VoiceHost[] AvailableHosts
        {
            get { return _hosts.Where(p => p.IsAvailable).ToArray(); }
        }

        public void UpdateHosts()
        {
            _hosts = VoiceHost.GetAllByRegion(this);
        }


    }
}