﻿using Curse.Voice.Host.Extensions;
using Curse.Voice.HostInterface;
using Curse.Voice.HostInterface.Models;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Web;

namespace Curse.Voice.Host
{
    public class VoiceInstance
    {

        public DateTime DateCreated = DateTime.UtcNow;
        public string Identifier { get; private set; }
        public int CreatorUserID { get; private set; }
        public int OwnerSessionID { get; private set; }

        private int _idCounter = 0;

        public ConcurrentDictionary<int, VoiceSession> SessionsByID { get; private set; }
        public ConcurrentDictionary<int, VoiceSession> SessionsByCallback { get; private set; }

        public VoiceInstance(string identifer, int creatorUserID)
        {
            Identifier = identifer;
            CreatorUserID = creatorUserID;
            SessionsByID = new ConcurrentDictionary<int, VoiceSession>();
            SessionsByCallback = new ConcurrentDictionary<int, VoiceSession>();
        }

        public VoiceSession GetSession(int id)
        {
            VoiceSession session = null;
            if (SessionsByCallback.TryGetValue(id, out session))
            {
                return session;
            }
            return null;
        }

        public bool IsExpired
        {
            get { return (DateTime.UtcNow - DateCreated > TimeSpan.FromMinutes(5)) && SessionsByID.Count == 0; }
        }

        public IEnumerable<VoiceUser> Users
        {
            get { return SessionsByID.Values.Select(p => p.User); }
        }

        public IEnumerable<VoiceSession> Sessions
        {
            get { return SessionsByID.Values; }
        }

        public VoiceSession CreateSession(int? userID, string displayName, string avatarUrl, ICurseVoiceCallbackClient callbackClient)
        {
            VoiceUser user = new VoiceUser(Interlocked.Increment(ref _idCounter), userID, displayName, avatarUrl);
            VoiceSession session = new VoiceSession(this, user, callbackClient);
            if (session.IsCreator)
            {
                OwnerSessionID = session.ID;
            }
            SessionsByID.TryAdd(session.User.ID, session);
            SessionsByCallback.TryAdd(session.CallbackClient.GetHashCode(), session);
            ClientConnected(session);
            return session;
        }

        public void ClientConnected(VoiceSession connectedSession)
        {                        
            List<Task> notificationTasks = new List<Task>();

            foreach (var session in SessionsByID.Values)
            {
                if (!session.IsConnected || session.ID == connectedSession.ID)
                {
                    continue;
                }

                try
                {
                    var task = session.NotifyUserJoinedAsync(connectedSession.User);
                    notificationTasks.Add(task);
                    //session.NotifyUserJoined(connectedSession.User);
                }
                catch (Exception ex)
                {

                    Debug.WriteLine("Failed to notify client that a user join: " + ex.Message);
                }
            }

            TaskHelper.WaitAllSafely(notificationTasks.ToArray(), TimeSpan.FromSeconds(1));
        }

        public void ClientDisconnected(VoiceSession disconnectedSession)
        {
            // Remove this session from our list
            VoiceSession removedSession;
            if (!SessionsByID.TryRemove(disconnectedSession.User.ID, out removedSession))
            {
                return;
            }

            SessionsByCallback.TryRemove(disconnectedSession.ID, out removedSession);

            // If this session was the creator, promote someone else
            if (disconnectedSession.IsOwner)
            {
                var firstSession = SessionsByID.Values.FirstOrDefault();
                if (firstSession != null)
                {
                    OwnerSessionID = firstSession.ID;
                }
                else
                {
                    OwnerSessionID = 0;
                }
            }

            // Notify all connected clients
            List<Task> notificationTasks = new List<Task>();

            foreach (var session in SessionsByID.Values)
            {
                if (!session.IsConnected)
                {
                    continue;
                }

                try
                {
                    session.NotifyUserLeftAsync(disconnectedSession.User.ID);
                    //notificationTasks.Add(task);
                }
                catch (Exception ex)
                {

                    Debug.WriteLine("Failed to notify client that a user join: " + ex.Message);
                }
            }            

            //TaskHelper.WaitAllSafely(notificationTasks.ToArray(), TimeSpan.FromSeconds(1));

        }

        public void BroadcastTransmit(VoiceSession broadcaster, byte[] voiceData, bool loopback)
        {
            
            VoiceTransmission transmission = new VoiceTransmission()
                {
                    UserID = broadcaster.User.ID,
                    VoiceData = voiceData
                };

            
            foreach (var session in SessionsByID.Values)
            {
                if (!session.IsConnected)
                {
                    continue;
                }
                
                if (session.ID == broadcaster.ID && !loopback)
                {
                    continue;
                }

                session.Transmit(transmission);
            }
                       
        }

        public void BroadcastEndTransmit(int userID)
        {
            List<Task> notificationTasks = new List<Task>();


            foreach (var session in SessionsByID.Values)
            {
                if (!session.IsConnected)
                {
                    continue;
                }

                try
                {
                    var task = session.EndTransmit(userID);
                    notificationTasks.Add(task);
                }
                catch (Exception ex)
                {

                    Debug.WriteLine("Failed to notify client that a user join: " + ex.Message);
                }
            }

            TaskHelper.WaitAllSafely(notificationTasks.ToArray(), TimeSpan.FromSeconds(1));
        }

        public void BroadcastUserUpdate(VoiceUser voiceUser)
        {
            List<Task> notificationTasks = new List<Task>();

            foreach (var session in SessionsByID.Values)
            {
                if (!session.IsConnected)
                {
                    continue;
                }

                try
                {
                    var task = session.UserUpdate(voiceUser);
                    notificationTasks.Add(task);
                }
                catch (Exception ex)
                {

                    Debug.WriteLine("Failed to notify client that a user join: " + ex.Message);
                }
            }

            TaskHelper.WaitAllSafely(notificationTasks.ToArray(), TimeSpan.FromSeconds(1));
        }

        public void BroadcastChatMessage(ChatMessage chatMessage)
        {
            List<Task> notificationTasks = new List<Task>();

            foreach (var session in SessionsByID.Values)
            {
                if (!session.IsConnected || session.User.ID == chatMessage.AuthorUserID)
                {
                    continue;
                }

                try
                {
                    //var task = session.SendChatMessage(chatMessage);
                    //notificationTasks.Add(task);
                    session.SendChatMessage(chatMessage);
                }
                catch (Exception ex)
                {

                    Debug.WriteLine("Failed to notify client that a user join: " + ex.Message);
                }
            }

            TaskHelper.WaitAllSafely(notificationTasks.ToArray(), TimeSpan.FromSeconds(1));
        }

    }
}